/* 
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.i18n.plugin.parser;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.nuiton.i18n.plugin.AbstractI18nPlugin;
import org.nuiton.i18n.plugin.I18nLogger;
import org.nuiton.util.PluginHelper.SortedProperties;
import org.nuiton.i18n.plugin.parser.event.KeysModifier;
import org.nuiton.util.FileUpdater;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.nuiton.util.SourceEntry;

/**
 * Abstract implementation for parsing goal.
 *
 * @author tony
 */
public abstract class AbstractI18nParser extends AbstractI18nPlugin implements Parser {

    /** @return the outGetter to use for the instance (java.getter,...) */
    protected abstract String getOutGetter();

    /** @return the starting regex expression to catch keys in key modifier */
    protected abstract String getKeyModifierStart();

    /** @return the ending regex expression to catch keys in key modifier */
    protected abstract String getKeyModifierEnd();

    /** @return the default includes to add to directory scanner */
    protected abstract String[] getDefaultIncludes();

    /** @return the default excludes to add to directory scanner */
    protected abstract String[] getDefaultExcludes();

    /** @return the default src directory to use in directory scanner */
    protected abstract File getDefaultBasedir();

    public abstract FileUpdater newFileUpdater(SourceEntry entry);
    /**
     * treate default entry
     *
     * @parameter expression="${i18n.treateDefaultEntry}" default-value="true"
     */
    protected boolean treateDefaultEntry;
    /**
     * Source entries (src+includes+excludes) .
     *
     * @parameter expression="${i18n.entries}"
     */
    protected I18nSourceEntry[] entries;
    /**
     * flag to display touched files while parsing.
     * <p/>
     * Note: the value will be always <code>true</code> if {@link #verbose} is set
     * at <code>true</code>.
     *
     * @parameter expression="${i18n.showTouchedFiles}" default-value="${maven.verbose}"
     * @since 0.9
     */
    protected boolean showTouchedFiles;
    /**
     * flag to save at eachfile treated the getter file
     *
     * @parameter expression="${i18n.safeMode}" default-value="false"
     * @since 0.9
     */
    protected boolean safeMode;
    protected SortedProperties result;
    protected SortedProperties oldParser;
    protected SortedProperties oldLanguage;
    protected int fileTreated = 0;
    protected long t0;
    protected boolean touchFile;
    protected List<File> treadedFiles;

    public boolean isStrictMode() {
        return strictMode;
    }
    
    @Override
    public void init() {
        super.init();
        t0 = System.nanoTime();
        result = new SortedProperties(encoding);
        oldParser = new SortedProperties(encoding);
        oldLanguage = new SortedProperties(encoding);
        out.mkdirs();
        // evenements
        if (keysModifier) {
            addParserEvent(KeysModifier.getInstance(getKeyModifierStart(), getKeyModifierEnd(), encoding));
        }
        treadedFiles = new ArrayList<File>();
        if (!silent && verbose) {
            showTouchedFiles = true;
        }
    }


    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        init();
        if (entries == null || entries.length == 0 && !treateDefaultEntry) {
            // nothing to do
            return;
        }

        if (!silent && safeMode) {
            getLog().info("config - safeMode is on (could be slower).");
        }
        if (!silent && strictMode) {
            getLog().info("config - strictMode is on (all files will be parsed).");
        }

        try {
            // Reprise sur un ancien parsing
            File oldParserFile = getGetterFile(out, getOutGetter(), true);
            File saveFile = getGetterFileBackup(out, getOutGetter());

            oldParser.load(oldParserFile);
            copyFile(oldParserFile, saveFile);

            // Anciennes cles disponnibles
            //fixme : pourquoi on utilise un bundle precis ? le premier ici, je ne comprends pas
            File oldLanguageFile = getI18nFile(src, artifactId, locales[0], true);

            oldLanguage.load(oldLanguageFile);

            // Parsing
            parse();

            // Suppression du fichier sauvegarder
            saveFile.delete();

            int i = treadedFiles.size();
            if (fileTreated == 0) {
                if (!silent) {
                    getLog().info("Nothing to generate - all files are up to date.");
                }
            } else {
                if (!silent) {
                    getLog().info(getVerboseLog().getLogEntry("parsing is done. [treated file(s) : " + i + '/' + fileTreated + "]", fileTreated, 0, t0));
                }
                addGetter();
            }

        } catch (Exception e) {
            getLog().error("Error code parsing ", e);
            throw new MojoFailureException("Error code parsing");
        }

    }

    /**
     * launch the parse on every given entries.
     *
     * @throws IOException if any io pb
     */
    @Override
    public void parse() throws IOException {
        if (treateDefaultEntry) {
            addDefaultEntry();
        }
        long t00 = System.nanoTime();
        for (I18nSourceEntry entry : this.entries) {
            I18nLogger vLog = getVerboseLog();

            vLog.setEntry(entry);

            boolean skip = entry.init(this);

            if (skip) {
                if (!silent && verbose) {
                    getLog().info("skip - " + entry.getSkipMessage());
                }
                continue;
            }

            long t000 = System.nanoTime();
            int nbFiles = entry.getFiles().length;
            if (!silent && verbose) {
                vLog.infoEntry("start", vLog.getLogEntry("[incoming file(s) : " + entry.getFoudFiles() + "]", 0, 0, 0));
            }

            // launch parser for found files
            parseEntry(entry);

            if (!silent && verbose) {
                // log skipped files
                for (String skipFile : entry.getSkipFiles()) {
                    vLog.setFile(new File(entry.getBasedir(), skipFile));
                    vLog.infoFile("skip", null);
                }
            }
            fileTreated += nbFiles;
            if (!silent && verbose) {
                vLog.infoEntry("end", vLog.getLogEntry("[treated file(s)  : " + nbFiles + "]", nbFiles, t000, t00));
            }
            t00 = System.nanoTime();
        }
    }

    /**
     * Add the default entry to entries given in configuration.
     * <p/>
     * This is a convinient method to simplify the configuration of the plugin.
     */
    protected void addDefaultEntry() {
//        List<MySourceEntry> list;
        if (verbose) {
            getLog().info("add default entry");
        }
        boolean hasEntries = entries != null && entries.length > 0;
        I18nSourceEntry[] tmp = new I18nSourceEntry[hasEntries ? entries.length + 1 : 1];
        if (hasEntries) {
            System.arraycopy(entries, 0, tmp, 0, entries.length);
        }
        tmp[tmp.length - 1] = new I18nSourceEntry();
        entries = tmp;
    }

    /**
     * launch parsing on a given entry.
     *
     * @param entry currentEntry to treate
     * @throws IOException if any io pb.
     */
    protected final void parseEntry(SourceEntry entry) throws IOException {
        long t00 = System.nanoTime();
        String[] files = entry.getFiles();
        int beforeEntryResultSize = result.size();
        for (int i = 0, max = files.length; i < max; i++) {
            String file1 = files[i];
            long t000 = System.nanoTime();
            String fileName = entry.getBasedir().getAbsolutePath() + File.separator + file1;
            File file = new File(fileName);
            for (ParserEvent event : events) {
                event.eventChangeFile(file);
            }
            I18nLogger vLog = getVerboseLog();
            vLog.setFile(file);

            touchFile = false;
            int size = result.size();
            if (!silent && verbose) {
                vLog.infoFile("parse", null);
            }
            parseFile(file);

            //TC-20090214 pour des questions de performance, on ne sauvegarde pas
            // a chaque traitement de fichier, les clefs mais une fois pour chaque
            // source entry
            // Detection de nouvelles cles, sauvegarde du fichier pour pouvoir le restaurer en cas de plantage
            if (safeMode) {
                if (size != result.size()) {
                    saveGetterFile();
                }
            }
            if (touchFile) {
                if (showTouchedFiles) {
                    vLog.infoFile("touch", null);
                }
                treadedFiles.add(file);
                if (getLog().isDebugEnabled()) {
                    vLog.debug(vLog.getLogEntry(fileName, i, t000, t00));
                }
            }
            for (ParserEvent event : events) {
                event.eventNextFile(file);
            }
        }

        if (!safeMode && beforeEntryResultSize < result.size()) {
            // Detection de nouvelles cles, sauvegarde du fichier
            saveGetterFile();
        }
    }

    /**
     * Save the result in the getter file.
     *
     * @throws IOException if any io pb
     */
    protected void saveGetterFile() throws IOException {
        File getterFile = getGetterFile(out, getOutGetter(), false);
        result.store(getterFile);
    }

}
