/*
 * #%L
 * Maven helper plugin
 * 
 * $Id: AbstractMojoTest.java 852 2012-07-15 18:24:59Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-helper-plugin/tags/maven-helper-plugin-1.6/src/test/java/org/nuiton/plugin/AbstractMojoTest.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.plugin;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.maven.plugin.Mojo;
import org.apache.maven.project.DefaultProjectBuilderConfiguration;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuilderConfiguration;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.runner.Description;

import java.beans.Introspector;
import java.io.File;
import java.io.IOException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Base test class for a mojo.
 * <p/>
 * <b>Note:</b> replace the previous class {@code org.nuiton.util.BasePluginTestCase}.
 * <p/>
 * Inside each test method, we can use the following objects :
 * <p/>
 * <ul> <li>{@link #getTestDir()} : location of mojo resources (where the pom
 * file for example)</li> <li>{@link #getPomFile()} : location of the pom
 * file</li> <li>{@link #getMojo()} : the instanciated and ready to execute
 * mojo</li> </ul>
 * <p/>
 * To change the behaviour of initialization of mojo, you can override the
 * following methods : <ul> <li>{@link #getBasedir()} </li> <li>{@link
 * #getTestBasedir()} </li> <li>{@link #getTestDir(String, String)}</li>
 * <li>{@link #getPomFile(File, String, String)}</li> <li>{@link
 * #createMojo(File, String)}</li> <li>{@link #setUpMojo(Plugin, File)}</li>
 * </ul>
 *
 * @param <P> type of goal to test
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.3
 */
public abstract class AbstractMojoTest<P extends Plugin> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractMojoTest.class);

    /** the basedir of the project */
    protected static File basedir;

    /** the basedir of all tests (by convention {@code getBasedir()/target/test-classes}). */
    protected static File testBasedir;

    /**
     * Your test rule which offers methodName, testDir, pomFile and mojo inside
     * test methods.
     */
    @Rule
    public MojoTestRule name = new MojoTestRule();

    public File getBasedir() {
        if (basedir == null) {
            basedir = TestHelper.getBasedir();
            if (log.isDebugEnabled()) {
                log.debug("basedir = " + basedir);
            }
        }
        return basedir;
    }

    public File getTestBasedir() {
        if (testBasedir == null) {
            testBasedir = TestHelper.getFile(getBasedir(),
                                             "target",
                                             "test-classes"
            );
            if (log.isDebugEnabled()) {
                log.debug("testBasedir = " + testBasedir);
            }
        }
        return testBasedir;
    }

    public String getRelativePathFromBasedir(File f) {
        return TestHelper.getRelativePath(getBasedir(), f);
    }

    public String getRelativePathFromTestBasedir(File f) {
        return TestHelper.getRelativePath(getTestBasedir(), f);
    }

    /**
     * Obtain the name of the goal according to the methodName.
     * <p/>
     * By convention, we should consider that a test class use always the same
     * goal's name.
     *
     * @param methodName the name of the next test to execute.
     * @return the name of the goal to test for the given method test name.
     */
    protected abstract String getGoalName(String methodName);

    /**
     * Obtain the location of the directory where to find resources for the next
     * test.
     * <p/>
     * By convention, will be the package named by the test class name from the
     * {@link #getTestBasedir()}.
     *
     * @param methodName the method of the next test to execute
     * @param goalName   the common goal name to use
     * @return the directory where to find resources for the test
     */
    protected File getTestDir(String methodName, String goalName) {

        //TC-20091101 use a decipatilize simple name to avoid conflict of
        // package with existing class name.
        String rep = getClass().getPackage().getName() + "." +
                     Introspector.decapitalize(getClass().getSimpleName());
        String[] paths = rep.split("\\.");
        File testDir = TestHelper.getFile(getTestBasedir(), paths);
        if (isVerbose()) {
            log.info("test dir = " + getRelativePathFromBasedir(testDir));
        } else if (log.isDebugEnabled()) {
            log.debug("test dir = " + getRelativePathFromBasedir(testDir));
        }

        return testDir;
    }

    /**
     * Obtain the location of the pom file to use for next mojo test.
     * <p/>
     * By default, the pom file is the file with name {@code methodName+".xml"}
     * in the {@code testDir}.
     *
     * @param testDir    the location of resources for the next test (is the
     *                   result of the method {@link #getTestDir(String,
     *                   String)}.
     * @param methodName the name of the next test
     * @param goalName   the name of the common goal
     * @return the location of the pom file for the next mojo test.
     */
    protected File getPomFile(File testDir, String methodName, String goalName) {
        File pom = new File(testDir, methodName + ".xml");

        if (isVerbose()) {
            log.info("pom file = " + getRelativePathFromBasedir(pom));
        } else if (log.isDebugEnabled()) {
            log.debug("pom file = " + getRelativePathFromBasedir(pom));
        }

        return pom;
    }

    /**
     * Create the mojo base on the given {@code pomFile} for the given {@code
     * goalName}.
     *
     * @param pomFile  the location of the pom file
     * @param goalName the name of the goal to lookup
     * @return the instanciated mojo
     * @throws Exception if any problem while creating the mojo
     */
    @SuppressWarnings("unchecked")
    protected P createMojo(File pomFile, String goalName) throws Exception {
        Mojo lookupMojo = TestHelper.lookupMojo(goalName, pomFile);
        P result = (P) lookupMojo;
        return result;
    }

    /**
     * Initialize the given mojo.
     *
     * @param mojo    the instanciate mojo
     * @param pomFile the pom file used to instanciate the mojo
     * @throws Exception if any pb
     */
    protected void setUpMojo(P mojo, File pomFile) throws Exception {

        MavenProject project = mojo.getProject();
        if (project == null) {

            log.debug("init maven project");
            MavenProjectBuilder projectBuilder = (MavenProjectBuilder)
                    TestHelper.getDelegateMojoTest().getContainer().lookup(
                            MavenProjectBuilder.ROLE
                    );
            ProjectBuilderConfiguration projectBuilderConfiguration =
                    new DefaultProjectBuilderConfiguration();
            project = projectBuilder.build(pomFile, projectBuilderConfiguration);

//            project = new MavenProject();

            mojo.setProject(project);
        }

        mojo.getProject().setFile(pomFile);
    }

    protected P getMojo() {
        return name.getMojo();
    }

    protected String getMethodName() {
        return name.getMethodName();
    }

    protected File getPomFile() {
        return name.getPomFile();
    }

    protected File getTestDir() {
        return name.getTestDir();
    }

    protected boolean isVerbose() {
        return TestHelper.isVerbose();
    }

    protected void clearMojo(P mojo) {

        // by default do nothing
    }

    /**
     * Checks on the given {@code file} that :
     * <ul>
     * <li>file exists</li>
     * <li>the given {@code pattern} exists {@code required = true}
     * (or not {@code required = false}) in the content of the file.</li>
     * </ul>
     *
     * @param file     the file to test
     * @param pattern  the pattern to search
     * @param encoding encoding of the file to read
     * @param required flag to says if pattern should (or not) be found in
     *                 file's content
     * @throws IOException if could not read file
     */
    public void checkExistsPattern(File file,
                                   String pattern,
                                   String encoding,
                                   boolean required) throws IOException {

        // checks file exists
        assertTrue("File '" + file + "' does not exist, but should...",
                   file.exists()
        );

        //obtain file content as a string
        String content = PluginHelper.readAsString(file, encoding);

        checkPattern(file, content, pattern, required);
    }

    public void checkPattern(File file,
                             String content,
                             String pattern,
                             boolean required) throws IOException {

        String errorMessage = required ? "could not find the pattern : " :
                              "should not have found pattern :";

        // checks pattern found (or not) in file's content
        assertEquals(errorMessage + pattern + " in '" + file + "'",
                     required,
                     content.contains(pattern)
        );
    }

    /**
     * To offer inside each test method (annotated by a {@link Test}) the
     * following properties :
     * <p/>
     * <ul> <li>{@link #testDir} : location where to find resources for the
     * test</li> <li>{@link #pomFile} : location of the pom file to use to build
     * the mojo</li> <li>{@link #mojo} : the instanciated and initialized
     * mojo</li> </ul>
     */
    public class MojoTestRule extends TestName {

        /** location of the pom to use */
        private File pomFile;

        /** the mojo to used in the test method (based on the {@link #pomFile}). */
        private P mojo;

        /** the directory where the resources of the test are */
        private File testDir;

        public MojoTestRule() {
            if (log.isDebugEnabled()) {
                log.debug("NEW MojoTest instance for " + getTestClass());
            }
        }

        @Override
        public void starting(Description method) {
            super.starting(method);

            if (isVerbose()) {
                log.info("==============================================================================================");
            }
            String methodName = name.getMethodName();

            log.info("NEW Mojo test starting : " + getTestClass().getName() +
                     "#" + methodName);

            String goalName = getGoalName(methodName);

            testDir = getTest().getTestDir(methodName, goalName);

            pomFile = getTest().getPomFile(testDir, methodName, goalName);

            try {
                Assert.assertTrue("could not find pom " +
                                  pomFile.getAbsoluteFile(), pomFile.exists());

                mojo = createMojo(pomFile, goalName);

                getTest().setUpMojo(mojo, pomFile);

            } catch (Exception ex) {
                throw new IllegalStateException(
                        "could not init test " + getClass() + " - " +
                        methodName + " for reason " + ex.getMessage(), ex);
            }
        }

        @Override
        public void finished(Description method) {
            super.finished(method);
            if (mojo != null) {
                clearMojo(mojo);
            }
            pomFile = null;
            mojo = null;
            testDir = null;
        }

        public P getMojo() {
            return mojo;
        }

        public File getPomFile() {
            return pomFile;
        }

        public File getTestDir() {
            return testDir;
        }

        public AbstractMojoTest<P> getTest() {
            return AbstractMojoTest.this;
        }

        public Class<?> getTestClass() {
            return getTest().getClass();
        }
    }
}
