/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.vault;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.keycloak.vault.AbstractVaultProviderFactory;
import org.keycloak.vault.FilesPlainTextVaultProvider;
import org.keycloak.vault.Scenario;
import org.keycloak.vault.SecretContains;
import org.keycloak.vault.VaultKeyResolver;
import org.keycloak.vault.VaultRawSecret;

public class PlainTextVaultProviderTest {
    private static final Logger logger = Logger.getLogger("org.keycloak.vault");
    private BlockingQueue<String> logMessages;
    private final ByteArrayOutputStream errContent = new ByteArrayOutputStream();
    private final PrintStream originalErr = System.err;
    private Handler logHandler;

    @Before
    public void setUp() {
        this.logMessages = new LinkedBlockingQueue<String>();
        logger.setLevel(Level.WARNING);
        this.logHandler = new Handler(){

            @Override
            public void publish(LogRecord record) {
                PlainTextVaultProviderTest.this.logMessages.add(record.getMessage());
            }

            @Override
            public void flush() {
            }

            @Override
            public void close() throws SecurityException {
            }
        };
        logger.addHandler(this.logHandler);
        System.setErr(new PrintStream(this.errContent));
    }

    @After
    public void tearDown() {
        logger.removeHandler(this.logHandler);
        System.setErr(this.originalErr);
    }

    @Test
    public void shouldObtainSecret() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_UNDERSCORE_KEY.getVaultKeyResolver()));
        VaultRawSecret secret1 = provider.obtainSecret("key1");
        Assert.assertNotNull((Object)secret1);
        Assert.assertNotNull(secret1.get().get());
        MatcherAssert.assertThat((Object)secret1, SecretContains.secretContains("secret1"));
    }

    @Test
    public void shouldReplaceUnderscoreWithTwoUnderscores() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test_realm", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_UNDERSCORE_KEY.getVaultKeyResolver()));
        VaultRawSecret secret1 = provider.obtainSecret("underscore_key1");
        Assert.assertNotNull((Object)secret1);
        Assert.assertNotNull(secret1.get().get());
        MatcherAssert.assertThat((Object)secret1, SecretContains.secretContains("underscore_secret1"));
    }

    @Test
    public void shouldReturnEmptyOptionalOnMissingSecret() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_UNDERSCORE_KEY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("non-existing-key");
        Assert.assertNotNull((Object)secret);
        Assert.assertFalse((boolean)secret.get().isPresent());
    }

    @Test
    public void shouldOperateOnNonExistingVaultDirectory() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.NON_EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_UNDERSCORE_KEY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("non-existing-key");
        Assert.assertNotNull((Object)secret);
        Assert.assertFalse((boolean)secret.get().isPresent());
    }

    @Test
    public void shouldOperateOnRealmDirectory() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_FILESEPARATOR_KEY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("key2");
        Assert.assertNotNull((Object)secret);
        Assert.assertNotNull(secret.get().get());
        MatcherAssert.assertThat((Object)secret, SecretContains.secretContains("secret2"));
    }

    @Test
    public void shouldObtainSecretWithMultipleResolvers() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_UNDERSCORE_KEY.getVaultKeyResolver(), AbstractVaultProviderFactory.AvailableResolvers.REALM_FILESEPARATOR_KEY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("key2");
        Assert.assertNotNull((Object)secret);
        Assert.assertNotNull(secret.get().get());
        MatcherAssert.assertThat((Object)secret, SecretContains.secretContains("secret2"));
    }

    @Test
    public void shouldReflectChangesInASecretFile() throws Exception {
        Path temporarySecretFile = Files.createTempFile("vault", null, new FileAttribute[0]);
        Path vaultDirectory = temporarySecretFile.getParent();
        String secretName = temporarySecretFile.getFileName().toString();
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(vaultDirectory, "ignored", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver()));
        String secret1AsString = null;
        String secret2AsString = null;
        Files.write(temporarySecretFile, "secret1".getBytes(), new OpenOption[0]);
        try (VaultRawSecret secret1 = provider.obtainSecret(secretName);){
            secret1AsString = StandardCharsets.UTF_8.decode((ByteBuffer)secret1.get().get()).toString();
        }
        Files.write(temporarySecretFile, "secret2".getBytes(), new OpenOption[0]);
        try (VaultRawSecret secret2 = provider.obtainSecret(secretName);){
            secret2AsString = StandardCharsets.UTF_8.decode((ByteBuffer)secret2.get().get()).toString();
        }
        Assert.assertEquals((Object)"secret1", (Object)secret1AsString);
        Assert.assertEquals((Object)"secret2", (Object)secret2AsString);
    }

    @Test
    public void shouldNotOverrideFileWhenDestroyingASecret() throws Exception {
        Path temporarySecretFile = Files.createTempFile("vault", null, new FileAttribute[0]);
        Path vaultDirectory = temporarySecretFile.getParent();
        String secretName = temporarySecretFile.getFileName().toString();
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(vaultDirectory, "ignored", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver()));
        Files.write(temporarySecretFile, "secret".getBytes(), new OpenOption[0]);
        VaultRawSecret secretAfterFirstRead = provider.obtainSecret(secretName);
        MatcherAssert.assertThat((Object)secretAfterFirstRead, SecretContains.secretContains("secret"));
        secretAfterFirstRead.close();
        VaultRawSecret secretAfterSecondRead = provider.obtainSecret(secretName);
        MatcherAssert.assertThat((Object)secretAfterFirstRead, (Matcher)CoreMatchers.not(SecretContains.secretContains("secret")));
        MatcherAssert.assertThat((Object)secretAfterSecondRead, SecretContains.secretContains("secret"));
    }

    @Test
    public void shouldPreventPathFileSeparatorInVaultSecretId() {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_FILESEPARATOR_KEY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret(".../key1");
        Assert.assertNotNull((Object)secret);
        Assert.assertFalse((boolean)secret.get().isPresent());
        Assert.assertTrue((boolean)this.logMessages.stream().anyMatch(msg -> msg.contains("Key .../key1 contains invalid file separator character")));
    }

    @Test
    public void shouldNotValidateWithInvalidPath() {
        Path vaultPath = Paths.get("/vault", new String[0]);
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(vaultPath, "test_realm", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.REALM_FILESEPARATOR_KEY.getVaultKeyResolver()));
        VaultKeyResolver resolver = AbstractVaultProviderFactory.AvailableResolvers.REALM_FILESEPARATOR_KEY.getVaultKeyResolver();
        String key = "key1";
        String resolvedKey = "../key1";
        boolean isValid = provider.validate(resolver, key, resolvedKey);
        Assert.assertFalse((boolean)isValid);
    }

    @Test
    public void shouldValidateWithDifferentResolver() {
        Path vaultPath = Paths.get("/vault", new String[0]);
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(vaultPath, "test_realm", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver()));
        VaultKeyResolver resolver = AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver();
        String key = "key1";
        String resolvedKey = "key1";
        boolean isValid = provider.validate(resolver, key, resolvedKey);
        Assert.assertTrue((boolean)isValid);
    }

    @Test
    public void shouldSearchForEscapedKeyOnlySecret() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("keyonly_escaped");
        Assert.assertNotNull((Object)secret);
        Assert.assertNotNull(secret.get().get());
        MatcherAssert.assertThat((Object)secret, SecretContains.secretContains("expected_secret_value"));
    }

    @Test
    public void shouldSearchForKeyOnlyLegacy() throws Exception {
        FilesPlainTextVaultProvider provider = new FilesPlainTextVaultProvider(Scenario.EXISTING.getPath(), "test", Arrays.asList(AbstractVaultProviderFactory.AvailableResolvers.KEY_ONLY.getVaultKeyResolver()));
        VaultRawSecret secret = provider.obtainSecret("keyonly_legacy");
        Assert.assertNotNull((Object)secret);
        Assert.assertFalse((boolean)secret.get().isPresent());
        Assert.assertTrue((boolean)this.logMessages.stream().anyMatch(msg -> msg.contains("Secret was found using legacy key 'keyonly_legacy'. Please rename the key to 'keyonly__legacy' and repeat the action.")));
    }
}

