/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMember;
import javassist.CtMethod;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.AccessFlag;
import javassist.bytecode.Bytecode;
import javassist.bytecode.ClassFile;
import javassist.bytecode.CodeAttribute;
import javassist.bytecode.ConstPool;
import javassist.bytecode.Descriptor;
import javassist.bytecode.DuplicateMemberException;
import javassist.bytecode.ExceptionsAttribute;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.Opcode;
import javassist.bytecode.StackMapTable;
import javassist.util.proxy.FactoryHelper;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistField;
import org.jboss.reflect.plugins.javassist.JavassistMethod;
import org.jboss.reflect.plugins.javassist.JavassistUtil;
import org.jboss.util.Strings;
import org.jboss.util.UnreachableStatementException;

/**
 * <p>Abstract base class to create implementations of the {@link JavassistMethod}, {@link JavassistConstructor} 
 * and {@link JavassistField} interfaces.</p>
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class JavassistMemberFactory
{
   protected static final String OBJECT_NAME = Object.class.getName();

   protected static final String ILLEGAL_ARGUMENT_EXCEPTION_NAME = IllegalArgumentException.class.getName();

   private static final String SHORT_NAME = Short.class.getName();

   private static final String LONG_NAME = Long.class.getName();

   private static final String INTEGER_NAME = Integer.class.getName();

   private static final String FLOAT_NAME = Float.class.getName();

   private static final String DOUBLE_NAME = Double.class.getName();

   private static final String CHARACTER_NAME = Character.class.getName();

   protected static final String BYTE_NAME = Byte.class.getName();

   protected static final String BOOLEAN_NAME = Boolean.class.getName();

   /** Descriptor for new IllegalArgumentException(String) */
   protected static final String ILLEGAL_ARGUMENT_EXCEPTION_CONSTRUCTOR_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class<?>[] {String.class}, Void.TYPE);

   /** Descriptor for StringBuilder.toString() */
   protected static final String STRINGBUILDER_TOSTRING_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class<?>[0], String.class);

   /** Descriptor for StringBuilder.append(Object) */
   protected static final String STRINGBUILDER_APPEND_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class<?>[] {Object.class}, StringBuilder.class);

   /** Descriptor for new StringBuilder(String) */
   protected static final String STRINGBUILDER_CONSTRUCTOR_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[]{String.class}, Void.TYPE);

   /** Descriptor for Boolean.booleanValue() */
   private final static String BOOLEAN_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Boolean.TYPE);
   
   /** Descriptor for Byte.byteValue() */
   private final static String BYTE_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Byte.TYPE); 
   
   /** Descriptor for Character.charValue() */
   private final static String CHAR_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Character.TYPE);
   
   /** Descriptor for Double.doubleValue() */   
   private final static String DOUBLE_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Double.TYPE);
   
   /** Descriptor for Float.floatValue() */   
   private final static String FLOAT_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Float.TYPE);
   
   /** Descriptor for Integer.intValue() */   
   private final static String INTEGER_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Integer.TYPE); 

   /** Descriptor for Long.longValue() */   
   private final static String LONG_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Long.TYPE);
   
   /** Descriptor for Short.shortValue() */   
   private final static String SHORT_VALUE_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[0], Short.TYPE);
   
   /** Descriptor for Boolean.valueOf */
   private final static String BOOLEAN_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Boolean.TYPE}, Boolean.class);

   /** Descriptor for Byte.valueOf */
   private final static String BYTE_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Byte.TYPE}, Byte.class);

   /** Descriptor for Character.valueOf */
   private final static String CHARACTER_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Character.TYPE}, Character.class);

   /** Descriptor for Double.valueOf */
   private final static String DOUBLE_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Double.TYPE}, Double.class);

   /** Descriptor for Float.valueOf */
   private final static String FLOAT_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Float.TYPE}, Float.class);

   /** Descriptor for Integer.valueOf */
   private final static String INTEGER_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Integer.TYPE}, Integer.class);

   /** Descriptor for Long.valueOf */
   private final static String LONG_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Long.TYPE}, Long.class);

   /** Descriptor for Short.valueOf */
   private final static String SHORT_VALUE_OF_DESCRIPTOR = RuntimeSupport.makeDescriptor(new Class[] {Short.TYPE}, Short.class);
   
   /** The method class counter */
   protected static final AtomicInteger counter = new AtomicInteger(0);

   /** The super class to use for the implementation */
   private final Class<?> superClass;
   
   /** If true the class bytes are output to the file system so they can be inspected with a decompiler/javap */
   private final boolean debug;
   
   /** Cached string representation of the signature of the accessed member */
   private String accessedMember;
   
   /**
    * In AS the JavassistMethod, -Constructor and -Field classloaders are not deployed in 
    * the system classpath. When generating an accessor for something from the system classpath
    * we need to make sure it happens at the level that can see the implemented interfaces
    * 
    */
   private static final ParentLoaderHandler PARENT_LOADER_HANDLER;
   
   static
   {
      PARENT_LOADER_HANDLER = AccessController.doPrivileged(new PrivilegedAction<ParentLoaderHandler>()
      {
         public ParentLoaderHandler run()
         {
            ClassLoader reflectLoader = JavassistMethod.class.getClassLoader();
            if (reflectLoader == null)
               reflectLoader = ClassLoader.getSystemClassLoader();
            ClassLoader loader = reflectLoader.getParent();
            Set<ClassLoader> parents = loader == null ? null : new HashSet<ClassLoader>(); 
            while (loader != null)
            {
               parents.add(loader);
               loader = loader.getParent();
            }
            return new ParentLoaderHandler(reflectLoader, parents);
         }
      });
   }

   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param check whether the methods implemented from the interfaces should check the parameters
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistMemberFactory(Class<?> superClass, boolean debug)
   {
      this.superClass = superClass;
      this.debug = debug;
   }
   
   /**
    * Creates a new {@link JavassistMethod} implementation for a given method
    * 
    * @param superClass the super class of the JavassistMethod implementation
    * @param ctMethod the CtMethod for which we want to create a JavassistMethod implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistMethod createJavassistMethod(Class<?> superClass, CtMethod ctMethod, boolean debug)
   {
      JavassistMemberFactory factory = new JavassistMethodFactory(superClass, ctMethod, debug);
      Class<JavassistMethod> member = factory.makeClass(JavassistMethod.class, ctMethod.getDeclaringClass());
      return wrapInErrorChecker(factory.instantiate(member), ctMethod);
   }

   /**
    * Creates a new {@link JavassistConstructor} implementation for a given constructor
    * 
    * @param superClass the super class of the JavassistConstructor implementation
    * @param ctConstructor the CtConstructor for which we want to create a JavassistConstructor implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistConstructor createJavassistConstructor(Class<?> superClass, CtConstructor ctConstructor, boolean debug)
   {
      JavassistMemberFactory factory = new JavassistConstructorFactory(superClass, ctConstructor, debug);
      Class<JavassistConstructor> member = factory.makeClass(JavassistConstructor.class, ctConstructor.getDeclaringClass());
      return wrapInErrorChecker(factory.instantiate(member), ctConstructor);
   }

   /**
    * Creates a new {@link JavassistField} implementation for a given field
    * 
    * @param superClass the super class of the JavassistField implementation
    * @param ctField the CtField for which we want to create a JavassistField implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static JavassistField createJavassistField(Class<?> superClass, CtField ctField, boolean debug)
   {
      JavassistMemberFactory factory = new JavassistFieldFactory(superClass, ctField, debug);
      Class<JavassistField> member = factory.makeClass(JavassistField.class, ctField.getDeclaringClass());
      return wrapInErrorChecker(factory.instantiate(member), ctField);
   }

   /**
    * Instantiates the class
    * 
    * @param clazz the class
    * @return the instance
    * @throws RuntimeException if an error ocurred
    */
   protected <T> T instantiate(Class<T> clazz)
   {
      try
      {
         return clazz.newInstance();
      }
      catch (Exception e)
      {
         // AutoGenerated
         throw new RuntimeException(e);
      }
   }
   
   
   /**
    * Creates a JavassistMethod/-Field/-Constructor implementation
    * 
    * @return the generated class
    */
   protected <T> Class<T> makeClass(Class<T> expected, CtClass target)
   {
      //Create a new public class
      final ClassFile cf = new ClassFile(false, getGeneratedClassName(), superClass.getName());
      cf.setAccessFlags(AccessFlag.PUBLIC);
      cf.setInterfaces(getInterfaceNames());

      try
      {
         //Add the constructors from the super class 
         makeConstructors(cf);
   
         //implement the methods from the interfaces
         implementMethods(cf);
   
         if (debug)
            debugWriteFile(cf);
      }
      catch(Exception e)
      {
         throw new RuntimeException("Error creating " + expected.getSimpleName() + " for " + target.getName(), e);
      }


      ClassLoader cl = target.getClassPool().getClassLoader();
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();
      
      return toClass(expected, target, cf, cl);
   }
   
   private <T> Class<T> toClass(final Class<T> expected, final CtClass target, final ClassFile cf, final ClassLoader cl) 
   {
      final ClassLoader actualLoader = PARENT_LOADER_HANDLER.getActualLoader(cl);
      Throwable t = null;
      try
      {
         if (System.getSecurityManager() == null)
            return FactoryHelper.toClass(cf, actualLoader);
         else
            return AccessController.doPrivileged(new PrivilegedExceptionAction<Class<T>>()
                  {
                  public Class<T> run() throws Exception
                  {
                     return FactoryHelper.toClass(cf, actualLoader);
                  }
               });
      }
      catch(CannotCompileException e)
      {
         t = e;
      }
      catch (PrivilegedActionException e)
      {
         t = e.getCause();
      }
      throw new RuntimeException("Error creating " + expected.getSimpleName() + " for " + target.getName() + " with classloader " + actualLoader + "(" + cl + ")", t);
   }
   
   /**
    * Implements the methods of the interface
    * 
    * @param cf the class file of the class we are creating
    */
   void implementMethods(ClassFile cf) throws DuplicateMemberException
   {
      int i = 0;
      while (true)
      {
         MethodInfo minfo = implementMethod(i++, cf.getConstPool());
         if (minfo == null)
            break;
         cf.addMethod(minfo);
      }
   }
   
   /**
    * Returns the signature string of the accessed member
    * 
    * @return the signature
    */
   String getAccessedMember()
   {
      if (accessedMember == null)
      {
         accessedMember = initAccessedMember();
      }
      return accessedMember;
   }
   
   /**
    * Create the method 
    * 
    * @param index the current index (starting at 0)
    * @param cp the constant pool
    * @return the created method, or null if there was no method with that index
    */
   abstract MethodInfo implementMethod(int index, ConstPool cp);
   
   /**
    * Get the names of the interfaces to implement
    */
   abstract String[] getInterfaceNames();
   
   /**
    * Get the name of the class we are creating
    * 
    * @return the class name
    */
   abstract String getGeneratedClassName();
   
   /**
    * Initialize the signature string of the accessed member
    * 
    *  @return the signature
    */
   abstract String initAccessedMember();
   
   /**
    * Counts the maximum stack size of the methods being implemented
    * 
    * @param offset the offset
    * @param params an array containing the parameters being implemented
    * @return the max stack size
    */
   int countParameterStackSize(int offset, CtClass...params)
   {
      int stacksize = offset;
      int n = params.length;
      for (int i = 0; i < n; ++i)
      {
         stacksize++;
         if (params[i].equals(CtClass.longType) || params[i].equals(CtClass.doubleType))
            stacksize++;
      }

      return stacksize;
   }
   
   /**
    * Adds the constructors from the super class to the and delegates to them
    * via super calls
    * 
    * @param cf the class file of the class we are creating
    * @throws CannotCompileException if any of the constructors could not be added
    */
   private void makeConstructors(ClassFile cf) throws CannotCompileException
   {
      Constructor<?>[] cons = SecurityActions.getDeclaredConstructors(superClass);
      for (int i = 0; i < cons.length; i++)
      {
         Constructor<?> c = cons[i];
         int mod = c.getModifiers();
         if (!Modifier.isFinal(mod) && !Modifier.isPrivate(mod))
         {
            MethodInfo m = makeConstructor(c, cf.getConstPool());
            cf.addMethod(m);
         }
      }
   }

   /**
    * Add a particular constructor from the super class and delegate to it via a 
    * super call
    *
    * @param cons the constructor we are implementing
    * @param cp the constant pool of the class we are generating
    * @return the method info for the created constructor 
    */
   private MethodInfo makeConstructor(Constructor<?> cons, ConstPool cp)
   {
      String desc = RuntimeSupport.makeDescriptor(cons.getParameterTypes(), Void.TYPE);
      MethodInfo minfo = new MethodInfo(cp, "<init>", desc);
      minfo.setAccessFlags(Modifier.PUBLIC);
      setThrows(minfo, cp, cons.getExceptionTypes());
      Bytecode code = new Bytecode(cp, 0, 0);

      int pc = code.currentPc();

      code.addAload(0);
      int s = addLoadParameters(code, cons.getParameterTypes(), 1);
      code.addInvokespecial(superClass.getName(), "<init>", desc);
      code.addOpcode(Opcode.RETURN);
      code.setMaxLocals(s + 1);
      CodeAttribute ca = code.toCodeAttribute();
      minfo.setCodeAttribute(ca);

      StackMapTable.Writer writer = new StackMapTable.Writer(32);
      writer.sameFrame(pc);
      ca.setAttribute(writer.toStackMapTable(cp));
      return minfo;
   }

   /**
    * Adds a throws clause to a method info
    * 
    * @param minfo the method info being created
    * @param cp the constant pool of the class file the minfo will be added to
    * @param exceptions the exceptions to be added to the throws clause
    */
   void setThrows(MethodInfo minfo, ConstPool cp, Class<?>[] exceptions)
   {
      if (exceptions.length == 0)
         return;

      String[] list = new String[exceptions.length];
      for (int i = 0; i < exceptions.length; i++)
         list[i] = exceptions[i].getName();

      ExceptionsAttribute ea = new ExceptionsAttribute(cp);
      ea.setExceptions(list);
      minfo.setExceptionsAttribute(ea);
   }

   /**
    * Writes the class file bytes to the local file system so they can be inspected
    * with javap or another decompiler.
    * 
    * @param cf the class file
    * @throws IOException if an error occurred writing the file
    */
   void debugWriteFile(ClassFile cf) throws IOException
   {
      FileOutputStream fout = new FileOutputStream(cf.getName() + ".class");
      DataOutputStream out = new DataOutputStream(fout);
      try 
      {
         cf.write(out);
      }
      finally {
         out.close();
      }
   }

   /**
    * Adds instructions to load all the parameters for a method/constructor
    * 
    * @param code the byte code of the method/constructor being constructed
    * @param params the types of the parameters of the method/constructor
    * @param offset the offset
    * @return the stack size
    */
   int addLoadParameters(Bytecode code, Class<?>[] params, int offset)
   {
      int stacksize = 0;
      int n = params.length;
      for (int i = 0; i < n; ++i)
         stacksize += addLoad(code, stacksize + offset, params[i]);

      return stacksize;
   }

   /**
    * Adds the instruction to load a particular parameter for a method constructor
    * 
    * @param code the byte code of the method/constructor being constructed
    * @param index the index of the parameter
    * @param type the type of the parameter
    * @return the stack size required for loading the parameter
    */
   int addLoad(Bytecode code, int index, Class<?> type)
   {
      if (type.isPrimitive())
      {
         if (type == Long.TYPE)
         {
            code.addLload(index);
            return 2;
         }
         else if (type == Float.TYPE)
            code.addFload(index);
         else if (type == Double.TYPE)
         {
            code.addDload(index);
            return 2;
         }
         else
            code.addIload(index);
      }
      else
         code.addAload(index);

      return 1;
   }

   /**
    * Get the boxed type
    * 
    * TODO JBMICROCONT-119 integer progression?
    * @param type the type to box
    * @return the boxed type name
    */
   String getBoxedType(CtClass type)
   {
      if (type.isArray())
      {
         return Descriptor.of(type);
      }
      if (type.isPrimitive())
      {
         if (CtClass.booleanType.equals(type))
            return BOOLEAN_NAME;
         else if (CtClass.byteType.equals(type))
            return BYTE_NAME;
         else if (CtClass.charType.equals(type))
            return CHARACTER_NAME;
         else if (CtClass.doubleType.equals(type))
            return DOUBLE_NAME;
         else if (CtClass.floatType.equals(type))
            return FLOAT_NAME;
         else if (CtClass.intType.equals(type))
            return INTEGER_NAME;
         else if (CtClass.longType.equals(type))
            return LONG_NAME;
         else if (CtClass.shortType.equals(type))
            return SHORT_NAME;
         throw new UnreachableStatementException();
      }
      return type.getName();
   }

   String getArrayType(CtClass type)
   {
      StringBuilder buf = new StringBuilder();

      //int dims = 0;
      while (type.isArray())
      {
         buf.append("L");
         try
         {
            type = type.getComponentType();
         }
         catch(NotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      buf.append(type.getName());
      buf.append(";");

      return buf.toString();
   }
   
   /**
    * Casts the value currently on the stack to the target type, and if a primitive
    * unboxes it. The byte code instructions to do this are added to the 
    * <code>code</code> parameter.
    * 
    * @param code the byte code of the method/constructor we are creating
    * @param type the target type we want to cast to
    */
   void castAndUnbox(Bytecode code, CtClass type)
   {
      if (type.getName().equals(OBJECT_NAME))
         return;

      code.addCheckcast(getBoxedType(type));
      if (type.isPrimitive())
      {
         if (CtClass.booleanType.equals(type))
         {
            code.addInvokevirtual(BOOLEAN_NAME, "booleanValue", BOOLEAN_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.byteType.equals(type))
         {
            code.addInvokevirtual(BYTE_NAME, "byteValue", BYTE_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.charType.equals(type))
         {
            code.addInvokevirtual(CHARACTER_NAME, "charValue", CHAR_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.doubleType.equals(type))
         {
            code.addInvokevirtual(DOUBLE_NAME, "doubleValue", DOUBLE_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.floatType.equals(type))
         {
            code.addInvokevirtual(FLOAT_NAME, "floatValue", FLOAT_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.intType.equals(type))
         {
            code.addInvokevirtual(INTEGER_NAME, "intValue", INTEGER_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.longType.equals(type))
         {
            code.addInvokevirtual(LONG_NAME, "longValue", LONG_VALUE_DESCRIPTOR);
            return;
         }
         else if (CtClass.shortType.equals(type))
         {
            code.addInvokevirtual(SHORT_NAME, "shortValue", SHORT_VALUE_DESCRIPTOR);
            return;
         }
         throw new UnreachableStatementException();
      }
      
   }
   
   /**
    * Adds the byte code instructions to the <code>code</code> paramter to box the value currently on the stack to the target type.
    * 
    * @param code the byte code of the method/constructor we are currently creating
    * @param type the type we want to cast to
    */
   void boxReturnValue(Bytecode code, CtClass type)
   {
      if (type.isPrimitive())
      {
         if (CtClass.booleanType.equals(type))
         {
            code.addInvokestatic(BOOLEAN_NAME, "valueOf", BOOLEAN_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.byteType.equals(type))
         {
            code.addInvokestatic(BYTE_NAME, "valueOf", BYTE_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.charType.equals(type))
         {
            code.addInvokestatic(CHARACTER_NAME, "valueOf", CHARACTER_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.doubleType.equals(type))
         {
            code.addInvokestatic(DOUBLE_NAME, "valueOf", DOUBLE_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.floatType.equals(type))
         {
            code.addInvokestatic(FLOAT_NAME, "valueOf", FLOAT_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.intType.equals(type))
         {
            code.addInvokestatic(INTEGER_NAME, "valueOf", INTEGER_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.longType.equals(type))
         {
            code.addInvokestatic(LONG_NAME, "valueOf", LONG_VALUE_OF_DESCRIPTOR);
            return;
         }
         else if (CtClass.shortType.equals(type))
         {
            code.addInvokestatic(SHORT_NAME, "valueOf", SHORT_VALUE_OF_DESCRIPTOR);
            return;
         }
         throw new UnreachableStatementException();
      }
   }

   /**
    * Wraps the generated JavassistMethod in an implementation that checks the 
    * parameters
    * 
    * @param m The wrapped method
    * @param method the target method for information about the parameters
    * @return the error checking wrapper 
    */
   private static JavassistMethod wrapInErrorChecker(JavassistMethod m, CtMethod method)
   {
      if (m == null || method == null)
         throw new IllegalArgumentException("Null method");
      
      int numParameters = 0;
      try
      {
         numParameters = method.getParameterTypes().length;
      }
      catch(NotFoundException e)
      {
         throw new IllegalArgumentException("Could not load the parameters for " + method);
      }
      
      return new ErrorCheckingJavassistMethod(m, method, numParameters);
   }
   
   /**
    * Wraps the generated JavassistConstructor in an implementation that checks the 
    * parameters
    * 
    * @param c The wrapped constructor
    * @param constructor the target constructor for information about the parameters
    * @return the error checking wrapper 
    */
   private static JavassistConstructor wrapInErrorChecker(JavassistConstructor c, CtConstructor constructor)
   {
      if (c == null || constructor == null)
         throw new IllegalArgumentException("Null constructor");
      
      int numParameters = 0;
      try
      {
         numParameters = constructor.getParameterTypes().length;
      }
      catch(NotFoundException e)
      {
         throw new IllegalArgumentException("Could not load the parameters for " + constructor);
      }
      
      return new ErrorCheckingJavassistConstructor(c, constructor, numParameters);
   }
   
   /**
    * Wraps the generated JavassistField in an implementation that checks the 
    * parameters
    * 
    * @param f The wrapped field
    * @param field the target field for information about the parameters
    * @return the error checking wrapper 
    */
   private static JavassistField wrapInErrorChecker(JavassistField f, CtField field)
   {
      if (f == null)
         throw new IllegalArgumentException("Null field");
      
      return new ErrorCheckingJavassistField(f, field);
   }
   
   private static class ErrorChecker
   {
      protected boolean checkNumberOfParameters(Object[] args, int numParameters)
      {
         if (args == null && numParameters > 0)
            return false;
         if (args != null && args.length != numParameters)
            return false;
         return true;
      }
      
      protected boolean isStatic(CtMember member)
      {
         return Modifier.isStatic(member.getModifiers());
      }
      
      protected void handleWrongParameters(String context, String target, Class<?>[] expected, Object[] args)
      {
         List<String> actual = new ArrayList<String>();
         if (args != null)
         {
            for (Object argument : args)
            {
               if (argument == null)
                  actual.add(null);
               else
                  actual.add(argument.getClass().getName());
            }
         }
         throw new IllegalArgumentException("Wrong arguments. " + context + " for target " + target + " expected=" + expected + " actual=" + actual);
         
      }

      protected void handleWrongTarget(Object target, Class<?> expected, String name)
      {
         throw new IllegalArgumentException("Wrong target for " + name + " " + target.getClass().getName() + " is not a " + expected.getName());
      }
      
      protected void handleNullTarget(AccessibleObject ao)
      {
         throw new IllegalArgumentException("Null target calling non-static " + ao);      
      }
      
   }
   
   private static class ErrorCheckingJavassistMethod extends ErrorChecker implements JavassistMethod
   {
      private final JavassistMethod delegate;
      private final CtMethod ctMethod;
      private final int numParameters;
      
      private ErrorCheckingJavassistMethod(JavassistMethod delegate, CtMethod ctMethod, int numParameters)
      {
         this.delegate = delegate;
         this.ctMethod = ctMethod;
         this.numParameters = numParameters;
      }

      public Object invoke(Object target, Object[] args) throws Throwable
      {
         if (!checkNumberOfParameters(args, numParameters))
            throw new IllegalArgumentException("Wrong number of parameters for " + ctMethod.getDeclaringClass() + "." + ctMethod.getName() + ctMethod.getSignature());
         
         try
         {
            return delegate.invoke(target, args);
         }
         catch(ClassCastException e)
         {
            Method method = JavassistUtil.ctMethodToMethod(ctMethod); 
            if (!isStatic(ctMethod))
            {
               if (!method.getDeclaringClass().isAssignableFrom(target.getClass()))
                  handleWrongTarget(target, method.getDeclaringClass(), method.getName());
            }
            
            Class<?>[] params = method.getParameterTypes();
            for (int i = 0 ; i < args.length ; i++)
            {
               if (!params[i].isAssignableFrom(args[i].getClass()))
                  handleWrongParameters(method.getName(), Strings.defaultToString(target), method.getParameterTypes(), args);
            }
            
            throw e;
         }
         catch(NullPointerException e)
         {
            Method method = JavassistUtil.ctMethodToMethod(ctMethod); 
            if (!isStatic(ctMethod) && target == null)
               handleNullTarget(method);
            
            CtClass[] parameters = ctMethod.getParameterTypes();
            for (int i = 0 ; i < parameters.length ; i++)
            {
               if (parameters[i].isPrimitive() && args[i] == null)
                  handleWrongParameters(method.getName(), Strings.defaultToString(target), method.getParameterTypes(), args);
            }
            
            throw e;
         }
      }
      
   }

   private static class ErrorCheckingJavassistConstructor extends ErrorChecker implements JavassistConstructor
   {
      private final JavassistConstructor delegate;
      private final CtConstructor ctConstructor;
      private final int numParameters;
      
      private ErrorCheckingJavassistConstructor(JavassistConstructor delegate, CtConstructor ctConstructor, int numParameters)
      {
         this.delegate = delegate;
         this.ctConstructor = ctConstructor;
         this.numParameters = numParameters;
      }

      public Object newInstance(Object[] args) throws Throwable
      {
         if (!checkNumberOfParameters(args, numParameters))
            throw new IllegalArgumentException("Wrong number of parameters for " + ctConstructor.getDeclaringClass() + "." + ctConstructor.getName() + ctConstructor.getSignature());
         
         try
         {
            return delegate.newInstance(args);
         }
         catch(ClassCastException e)
         {
            Constructor<?> constructor = JavassistUtil.ctConstructorToConstructor(ctConstructor);
            Class<?>[] params = constructor.getParameterTypes();
            for (int i = 0 ; i < args.length ; i++)
            {
               if (!params[i].isAssignableFrom(args[i].getClass()))
                  handleWrongParameters("new", Strings.defaultToString(ctConstructor.getDeclaringClass().getName()), constructor.getParameterTypes(), args);
            }
               
            throw e;
         }
         catch(NullPointerException e)
         {
            CtClass[] parameters = ctConstructor.getParameterTypes();
            for (int i = 0 ; i < parameters.length ; i++)
            {
               if (parameters[i].isPrimitive() && args[i] == null)
               {
                  Constructor<?> constructor = JavassistUtil.ctConstructorToConstructor(ctConstructor);
                  handleWrongParameters("new", Strings.defaultToString(ctConstructor.getDeclaringClass().getName()), constructor.getParameterTypes(), args);
               }
            }
            
            throw e;
         }
      }
   }

   private static class ErrorCheckingJavassistField extends ErrorChecker implements JavassistField
   {
      private final JavassistField delegate;
      private final CtField ctField;
      
      private ErrorCheckingJavassistField(JavassistField delegate, CtField ctField)
      {
         this.delegate = delegate;
         this.ctField = ctField;
      }

      public Object get(Object target) throws Throwable
      {
         try
         {
            return delegate.get(target);
         }
         catch (ClassCastException e)
         {
            Field field = JavassistUtil.ctFieldToField(ctField);
            
            if (!isStatic(ctField) && !field.getDeclaringClass().isAssignableFrom(target.getClass()))
               handleWrongTarget(target, field.getDeclaringClass(), field.getName());
            
            throw e;
         }
         catch(NullPointerException e)
         {
            Field field = JavassistUtil.ctFieldToField(ctField);
            if (!isStatic(ctField) && target == null)
               handleNullTarget(field);
            throw e;
         }
      }

      public void set(Object target, Object value) throws Throwable
      {
         try
         {
            delegate.set(target, value);
         }
         catch (ClassCastException e)
         {
            Field field = JavassistUtil.ctFieldToField(ctField);
            Class<?> type = field.getType();

            if (!isStatic(ctField) && !field.getDeclaringClass().isAssignableFrom(target.getClass()))
               handleWrongTarget(target, field.getDeclaringClass(), field.getName());

            if (!type.isAssignableFrom(value.getClass()))
               throw new IllegalArgumentException("Wrong arguments. Setting " + field.getName() + " for target " + target + " expected=" + field.getType() + " actual=" + value.getClass());
         }
         catch(NullPointerException e)
         {
            Field field = JavassistUtil.ctFieldToField(ctField);
            if (!isStatic(ctField) && target == null)
               handleNullTarget(field);
            if (ctField.getType().isPrimitive() && value == null)
               throw new IllegalArgumentException("Null value setting non-static field. " + field);
            throw e;
         }
      }
   }
   
   
   private static class ParentLoaderHandler
   {
      final ClassLoader reflectLoader;
      final Set<ClassLoader> parentLoaders;
      
      public ParentLoaderHandler(ClassLoader reflectLoader, Set<ClassLoader> parentLoaders)
      {
         if (reflectLoader == null)
            throw new IllegalArgumentException("Null reflect loader");
         this.reflectLoader = reflectLoader;
         this.parentLoaders = parentLoaders;
      }
      
      private ClassLoader getActualLoader(ClassLoader loader)
      {
         if (parentLoaders != null)
            if (parentLoaders.contains(loader))
               return reflectLoader;
         return loader;
      }
   }
}