/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;

import javassist.CtBehavior;
import javassist.CtClass;
import javassist.CtField;
import javassist.NotFoundException;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ArrayType;
import javassist.bytecode.SignatureAttribute.BaseType;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.ClassType;
import javassist.bytecode.SignatureAttribute.MethodSignature;
import javassist.bytecode.SignatureAttribute.ObjectType;
import javassist.bytecode.SignatureAttribute.Type;
import javassist.bytecode.SignatureAttribute.TypeArgument;
import javassist.bytecode.SignatureAttribute.TypeParameter;
import javassist.bytecode.SignatureAttribute.TypeVariable;

import org.jboss.reflect.spi.TypeInfo;

/**
 * Utilities to figure out generic information 
 *  
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JavassistHelper
{
   /**
    * Looking at the classes between clazz and search, determine the type of the type parameter with the passed in index
    * 
    * @param clazz the sub class
    * @param search the parent class or interface we are searching for
    * @param parameter the index of the type parameter we are looking for
    * @return the generic type
    */
   static ClassType determineType(CtClass clazz, CtClass search, int parameter)
   {
      Stack<CtClass> hierarchy = new Stack<CtClass>();      
      try
      {
         determineHierarchy(hierarchy, clazz, search);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      return determineType(hierarchy, parameter);
   }

   /**
    * Looking at the classes between clazz and search, determine the type of the type parameter with the passed in index
    *
    * @param actualTypeArguments the actual type arguments
    * @param clazz the sub class
    * @param search the parent class or interface we are searching for
    * @param parameter the index of the type parameter we are looking for
    * @return the type
    */
   static TypeInfo determineInfoIndex(TypeInfo[] actualTypeArguments, CtClass clazz, CtClass search, int parameter)
   {
      Stack<CtClass> hierarchy = new Stack<CtClass>();      
      try
      {
         determineHierarchy(hierarchy, clazz, search);
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      return actualTypeArguments[determineInfoIndex(actualTypeArguments, hierarchy, parameter)];
   }

   /**
    * Get the full name of the generic javassist object type
    *
    * @param type the type
    * @param spy used to determine actual types of type variables
    * @return the generic type name
    */
   static String getGenericName(ObjectType type, JavassistTypeVariableSpy spy)
   {
      StringBuilder sb = new StringBuilder();
      appendTypeGenericInfo(type, spy, sb, null);
      return sb.toString();
   }
   
   /**
    * Get the fully qualified name for a generic type
    * 
    * @param type the type
    * @return the name
    */
   static String getClassNameForGenericType(ClassType type)
   {
      if (type.getDeclaringClass() == null)
         return type.getName();

      StringBuilder sb = new StringBuilder(type.getName());
      while(type.getDeclaringClass() != null)
      {
         sb.insert(0, type.getDeclaringClass().getName() + "$");
         type = type.getDeclaringClass();
      }
      
      return sb.toString();
   }
   
   /**
    * Append the information of the type to the name builder
    *
    * @param type the type
    * @param spy used to determine actual types of type variables
    * @param sb the string builder receiving the name
    * @param doneVars the names of the done type variables
    * @return the generic type name
    */
   private static void appendTypeGenericInfo(ObjectType type, JavassistTypeVariableSpy spy, StringBuilder sb, Set<String> doneVars)
   {
      
      if (type instanceof ClassType)
      {
         ClassType ctype = (ClassType)type;
         sb.append(getClassNameForGenericType(ctype));

         TypeArgument[] arguments = ctype.getTypeArguments();
         
         if (arguments != null && arguments.length > 0)
         {
            StringBuilder params = new StringBuilder();
            params.append("<");
            
            for (int i = 0 ; i < arguments.length ; i++)
            {
               if (i > 0)
                  params.append(", ");
               
               if (arguments[i].getType() == null)
                  params.append(Object.class.getName());
               else
                  appendTypeGenericInfo(arguments[i].getType(), spy, params, doneVars);
            }
            if (params.length() > 1)
            {
               params.append(">");
               sb.append(params.toString());
            }
         }
      }
      else if (type instanceof SignatureAttribute.TypeVariable)
      {
         TypeVariable tv = (TypeVariable)type; 
         Type real = spy.getTypeBound(tv);
         if (real instanceof ObjectType == false)
            throw new IllegalStateException("Type is not an instance of ObjectType " + real);
         
         if (doneVars == null)
            doneVars = new HashSet<String>();
         tv.getName();
         if (!doneVars.contains(tv.getName()))
         {
            doneVars.add(tv.getName());
            appendTypeGenericInfo((ObjectType)real, spy, sb, doneVars);
         }
      }
      else if (type instanceof ArrayType)
      {
         ArrayType array = (ArrayType)type;
         
         if (array.getComponentType() instanceof BaseType)
            sb.append(array.getComponentType());
         else
            appendTypeGenericInfo((ObjectType)array.getComponentType(), spy, sb, doneVars);
         
         for (int i = 0 ; i < array.getDimension() ; i++)
            sb.append("[]");
      }
      else
      {
         //TODO This might need implementing once we test wildcards
         throw new IllegalArgumentException("Unhandled type " + type);
      }
   }

   /**
    * Determine the type of the parameter in the top-level class. 
    * 
    * @param hierarchy the hierarchy of classes as determined by {@link JavassistHelper#determineHierarchy(Stack, CtClass, CtClass)}
    * @param parameter the index of the parameter
    * @return the type
    */
   private static ClassType determineType(Stack<CtClass> hierarchy, int parameter)
   {
      TypeDecider decider = new TypeDecider();
      decider.determineType(hierarchy, parameter);
      return decider.classType;
   }
   
   private static int determineInfoIndex(TypeInfo[] actualTypeArguments, Stack<CtClass> hierarchy, int parameter)
   {
      TypeDecider decider = new TypeDecider();
      decider.determineType(hierarchy, parameter);
      return decider.lastIndex;
   }

   /**
    * Gets the ClassSignature for a class
    * 
    * @param clazz the CtClass
    * @return the ClassSignature
    * @throws IllegalArgumentException if clazz is null 
    */
   public static ClassSignature getClassSignature(CtClass clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null clazz");

      if (clazz.isArray() || clazz.isPrimitive())
         return null;
      //Use getClassFile2 since in some cases the class has been frozen
      SignatureAttribute signature = (SignatureAttribute)clazz.getClassFile2().getAttribute(SignatureAttribute.tag);
      if (signature == null)
         return null;
      String sig = signature.getSignature();
   
      try
      {
         return SignatureAttribute.toClassSignature(sig);
      }
      catch (BadBytecode e)
      {
         throw new IllegalStateException(e);
      }
   }
   
   /**
    * Creates the generic parameter types for a constructor or method
    * 
    * @param behavior the method or constructor
    * @param sig the method/constructor signature
    * @param typeInfo the javassist type info
    * @return the parameter types 
    */
   public static TypeInfo[] createParameterTypes(final CtBehavior behavior, final MethodSignature sig, final JavassistTypeInfo typeInfo)
   {
      SignatureAttribute.Type[] types = sig.getParameterTypes();
      TypeInfo[] parameterTypes = new TypeInfo[types.length];
      for (int i = 0 ; i < types.length ; i++)
         parameterTypes[i] = typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), types[i], JavassistTypeVariableSpy.createForBehavior(behavior, sig));
      return parameterTypes;
   }
   
   /**
    * Gets the ClassSignature for a method/constructor
    * 
    * @param behaviour the method/constructor
    * @return the ClassSignature
    * @throws IllegalArgumentException if clazz is null 
    */
   public static MethodSignature getMethodSignature(CtBehavior behaviour)
   {
      if (behaviour == null)
         throw new IllegalArgumentException("Null method/constructor");
      
      //Use getMethodInfo2 since in some cases the class has been frozen
      SignatureAttribute signature = (SignatureAttribute)behaviour.getMethodInfo2().getAttribute(SignatureAttribute.tag);
      if (signature == null)
         return null;
      String sig = signature.getSignature();
      try
      {
         return SignatureAttribute.toMethodSignature(sig);
      }
      catch (BadBytecode e)
      {
         throw new IllegalStateException(e);
      }
   }

   /**
    * Gets the FieldSignature for a field
    * 
    * @param field the field
    * @return the FieldSignature
    * @throws IllegalArgumentException if clazz is null 
    */
   public static ObjectType getFieldSignature(CtField field)
   {
      if (field == null)
         throw new IllegalArgumentException("Null method/constructor");
      
      //Use getMethodInfo2 since in some cases the class has been frozen
      SignatureAttribute signature = (SignatureAttribute)field.getFieldInfo2().getAttribute(SignatureAttribute.tag);
      if (signature == null)
         return null;
      String sig = signature.getSignature();
      try
      {
         return SignatureAttribute.toFieldSignature(sig);
      }
      catch (BadBytecode e)
      {
         throw new IllegalStateException(e);
      }
   }

   /**
    * Figures out the path between the passed in classes
    * 
    * @param hierarchy receives the CtClasses that make up the hierarchy. This parameter may be null, in which 
    * case is does not receive the classes 
    * @param current the sub class
    * @param search the parent class or interface we are searching for
    * @return true if the current inherits from search
    * @throws IllegalArgumentException if current or search is null
    */
   public static boolean determineHierarchy(Stack<CtClass> hierarchy, CtClass current, CtClass search) throws NotFoundException
   {
      if (current == null)
         throw new IllegalArgumentException("Null current");
      if (search == null)
         throw new IllegalArgumentException("Null search");
   
      if (hierarchy != null)
         hierarchy.push(current);
      
      if (current == null)
         return false;
   
      if (current.equals(search))
         return true;
      
      CtClass[] interfaces = current.getInterfaces();
      if (search.isInterface() && interfaces != null)
      {
         for (int i = 0 ; i < interfaces.length ; i++)
         {
            boolean result = determineHierarchy(hierarchy, interfaces[i], search);
            if (result)
               return true;
            if (hierarchy != null)
               hierarchy.pop();
         }
      }
      
      CtClass superClass = current.getSuperclass();
      if (superClass == null)
         return false;
      
      boolean result = determineHierarchy(hierarchy, superClass, search);
      if (result)
         return true;
      
      if (hierarchy != null)
         hierarchy.pop();
      return false;
   }
   
   /**
    * Finds the TypeArgument used for the parent class/interface
    * 
    * @param parent the parent class or interface
    * @param classSig the signature of the "current" class, i.e. the child of <code>parent</code>/
    * @param index the index of the type parameter in the parent class
    * @return the found TypeArgument
    * @throws IllegalArgumentException if the index is greater than the length of TypeArguments found in the classSig, or if any of the parameters are null  
    */
   private static TypeArgument findSuperClassOrInterfaceArguments(CtClass parent, ClassSignature classSig, int index)
   {
      if (parent == null)
         throw new IllegalArgumentException();
      TypeArgument[] arguments = null; 
      if (parent.isInterface())
      {
         ClassType[] types = classSig.getInterfaces();
         for (int i = 0 ; i < types.length ; i++)
         {
            if (types[i].getName().equals(parent.getName()))
            {
               arguments = types[i].getTypeArguments();
               break;
            }
         }
         if (arguments == null)
            throw new IllegalStateException("Could not find " + parent.getName() + " in " + Arrays.toString(types));
      }
      else
      {
         arguments =  classSig.getSuperClass().getTypeArguments();
      }
      
      if (arguments.length <= index)
         throw new IllegalArgumentException("Argument " + index + " requested, but only " + arguments.length + " exist");
      return arguments[index];
   }


   private static class TypeDecider
   {
      int lastIndex;
      CtClass last = null;
      ClassType classType;
      
      private void determineType(Stack<CtClass> hierarchy, int parameter)
      {
         CtClass clazz = null;
         
         TypeParameter targetType = null;
         lastIndex = parameter;
         while (true)
         {
            if (hierarchy.empty())
               break;
            last = clazz;
            clazz = hierarchy.pop();
            
            ClassSignature classSig = getClassSignature(clazz);
            if (classSig != null)
            {
               TypeParameter[] typeParameters = classSig.getParameters();
      
               if (last == null)
               {
                  if (typeParameters.length <= parameter)
                     throw new IllegalArgumentException("Parameter " + parameter + " requested, but only " + typeParameters.length + " exist.");
                  targetType = typeParameters[parameter];
               }
               else
               {
                  TypeArgument argument = findSuperClassOrInterfaceArguments(last, classSig, lastIndex);
                  ObjectType type = argument.getType();
                  if (type == null)
                     continue;
                  if (type instanceof ClassType)
                  {
                     //name = ((ClassType) type).getName();
                     classType = (ClassType)type;
                     return;
                  }
                     
                  String name = null; 
                  if (type instanceof TypeVariable)
                     name= ((TypeVariable)type).getName();

                  for (int i = 0 ; i < typeParameters.length ; i++)
                  {
                     if (typeParameters[i].getName().equals(name))
                     {
                        lastIndex = i;
                        targetType = typeParameters[i];
                        break;
                     }
                  }
               }
            }
            else
            {
               break;
            }
         }
         if (targetType != null)
         {
            //TODO also check interfaces
             classType = (ClassType)targetType.getClassBound();
             return;
         }
      }
   }   
}
