/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.utils;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.ui.components.ImageContextLink;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;

/**
 * Classe permettant de gérer l'enregistrement d'images et la création de
 * miniatures.
 *
 * @author rannou
 * @version $Id: ImageUtil.java 2789 2009-11-04 15:26:08Z fdesbois $
 */
public class ImageUtil {

    /** log. */
    private static final Log log = LogFactory.getLog(ImageUtil.class);

    /**
     * Sauvegarde d'une image et de sa miniature.
     *
     * @param dir Le répertoire de destination.
     * @param imgChoice L'UIO de l'image à enregistrer.
     */
    public static void saveImage(ImageChoiceUIO imgChoice, File dir) {
        List<ImageChoiceUIO> imgChoices = new ArrayList<ImageChoiceUIO>();
        imgChoices.add(imgChoice);
        saveImages(imgChoices, dir);
    }

    /**
     * Sauvegarde des images et de leurs miniatures.
     *
     * @param imgChoices Les UIO des images à enregistrer.
     * @param dir Le répertoire de destination.
     */
    public static void saveImages(List<ImageChoiceUIO> imgChoices, File dir) {
        if (imgChoices != null && imgChoices.size() > 0) {
            if (!dir.exists()) {
                dir.mkdirs();
            }
            if (log.isDebugEnabled()) {
                log.debug("Destination folder: " + dir.getAbsolutePath());
            }
            for (ImageChoiceUIO imgChoice : imgChoices) {
                if (imgChoice.getImg() != null
                        && (imgChoice.getImg().getContentType().contains(
                                "image") || imgChoice.getImg().getContentType()
                                .contains("IMAGE"))) {

                    // Enregistrement de l'image
                    File img = new File(dir, imgChoice.getName());
                    imgChoice.getImg().write(img);
                    if (log.isDebugEnabled()) {
                        log.debug("Image saved: "
                                + imgChoice.getImg().getFileName() + " (size="
                                + imgChoice.getImg().getSize() + ")");
                    }

                    // Création de la miniature
                    createThumbnail(dir, img, 100);
                }
            }
        }
    }

    /**
     * Création d'une miniature.
     *
     * @param dir Le répertoire de destination.
     * @param img L'image à miniaturiser.
     * @param width La largeur de la miniature.
     */
    private static void createThumbnail(File dir, File img, int width) {
        ImageIcon ii = new ImageIcon(img.getAbsolutePath());
        Image image = ii.getImage();
        double imageRatio = (double) image.getHeight(null)
                / (double) image.getWidth(null);
        int height = (int) (width * imageRatio);

        BufferedImage thumbImage = new BufferedImage(width, height,
                BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics2D = thumbImage.createGraphics();
        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        graphics2D.drawImage(image, 0, 0, width, height, null);
        File thumbCopied = new File(dir, ImageContextLink.THUMB_PREFIX + img.getName());
        try {
            ImageIO.write(thumbImage, "jpg", thumbCopied);
        } catch (IOException e) {
            log.error("Erreur à l'enregistrement de la miniature", e);
        }
        if (log.isDebugEnabled()) {
            log.debug("Thumbnail created: " + thumbCopied.getName() + " (size="
                    + thumbCopied.length() + ")");
        }
    }
}
