/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: ResultForPoll.java 3333 2012-04-30 15:52:41Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/ResultForPoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.views.util.UrlHelper;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.services.impl.PollResultsService;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.ui.actions.DateConverter;
import org.chorem.pollen.ui.actions.PageSkin;
import org.nuiton.util.StringUtil;

import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Display results of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class ResultForPoll extends AbstractPollUriIdAction {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ResultForPoll.class);

    private Boolean byGroup;

    protected boolean userAllowed;

    /**
     * Résultats du sondage.
     * <p/>
     * La map est indexée par les noms de choix.
     */
    private transient Multimap<String, String> choicesResults;

    /** Résultats du sondage. */
    private transient Multimap<String, String> subtitles;

    private Poll poll;

    private List<PollResult> results;

//    private List<PollResult> ranking;

    private List<PollResult> topRanking;

    @Override
    public PageSkin getSkin() {
        return PageSkin.RESULT;
    }

    public Poll getPoll() {
        return poll;
    }

    public boolean isUserAllowed() {
        return userAllowed;
    }

    public List<PollResult> getResults() {
        return results;
    }

    public Date getResultAsDate(PollResult result) {
        Date date = new Date(Long.valueOf(result.getName()));
        return date;
    }

    public Multimap<String, String> getChoicesResults() {
        return choicesResults;
    }

    public Multimap<String, String> getSubtitles() {
        return subtitles;
    }

    public String getVictoryMessage() {

        String victoryMessage;

        if (poll.getVoteCountingType() == VoteCountingType.NUMBER) {
            victoryMessage = null;
        } else if (CollectionUtils.isEmpty(topRanking)) {
            victoryMessage = "";
        } else if (topRanking.size() == 1) {
            victoryMessage = _("pollen.common.victory");
        } else {
            victoryMessage = _("pollen.common.victories");
        }
        return victoryMessage;
    }

    /**
     * @param result la résultat à utiliser.
     * @return la valeur du résultat courant sans le .0 final
     */
    public String getTrimValue(PollResult result) {

        String value = result.getValue();

        // le résultat peut-être un double : 1,0 -> 1 et 1,2 -> 1,2
        if (value.endsWith(".0")) {
            value = value.substring(0, value.indexOf('.'));
        }
        return value;
    }

    @Override
    public String execute() throws Exception {

        String pollId = getPollId();
        Preconditions.checkNotNull(pollId);

        PollService service = newService(PollService.class);

        poll = service.getPollByPollId(pollId);

        if (poll == null) {
            addFlashError(_("pollen.error.pollNotFound"));
        } else {

            if (byGroup == null) {

                // default byGroup value
                byGroup = isGroupPoll();
            }

            String accountId = getAccountId();
            if (poll.isPublicResults()) {
                userAllowed = true;
            } else if (StringUtils.isNotEmpty(accountId)) {

                PollAccount creator = poll.getCreator();
                if (accountId.equals(creator.getAccountId())) {
                    userAllowed = true;
                }
            }

            if (!userAllowed) {
                addFlashError(_("pollen.error.userNotAllowed"));

            } else if (poll.isRunning(serviceContext.getCurrentTime())) {
                addFlashMessage(_("pollen.information.pollRunning"));
                loadResults();

            } else {
                loadResults();
            }
        }

        return SUCCESS;
    }

    public List<PollResult> getTopRanking() {
        return topRanking;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.CONDORCET;
    }

    public boolean isNumberVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NUMBER;
    }

    public boolean isByGroup() {
        return byGroup;
    }

    public void setByGroup(boolean byGroup) {
        this.byGroup = byGroup;
    }

    protected void loadResults() throws PollNotFoundException {

        PollResultsService service = newService(PollResultsService.class);

        PollResultList resultListDTO =
                service.getResults(poll.getPollId());

        results = resultListDTO.getPollResults();

        if (log.isDebugEnabled()) {
            for (PollResult res : results) {
                log.debug(res
                          + ", (voteCounting=" + res.getVoteCountingType()
                          + ", byGroup=" + res.isByGroup() + ")");
            }
        }

        if (isNumberVoteCounting()) {

            // resultats des choix cachés
            choicesResults = LinkedHashMultimap.create();

            // sous-titres des vrais choix (les choix pas cachés)
            subtitles = ArrayListMultimap.create();

            results = service.createNumberVoteCountingResult(
                    resultListDTO,
                    choicesResults,
                    subtitles);
        }

        topRanking = service.getTopRanking(results);

        if (log.isInfoEnabled()) {
            log.info("topRanking = " + topRanking);
        }
    }

    /** @return Récupération de la liste des résultats pour le diagramme. */
    public List<String> getChoices() {

        List<String> choiceValues = Lists.newArrayList();

        for (PollResult result : results) {
            String name = result.getName();
            String value = result.getValue();

            if (isDateType()) {
                // mise en forme de la date
                Date date = new Date(Long.parseLong(name));
                name = DateConverter.convertToString(date);
            }

            choiceValues.add(name);
            choiceValues.add(value);
        }
        return choiceValues;
    }

    public String getNormalPollResultChartUrl(int width, int height) {
        URL applicationUrl = getApplicationUrl();
        StringBuilder url = new StringBuilder(applicationUrl.toString());

        url.append("/io/generateChart");

        Map<String, Object> params = Maps.newHashMap();
        params.put("width", width);
        params.put("height", height);
        params.put("title", "");
        params.put("values", getChoices());


        UrlHelper.buildParametersString(params, url, "&");

        String chartUrl = url.toString();
        return chartUrl;
    }

    public Set<String> getChoicesResultNames() {
        return choicesResults.keySet();
    }

    public String getChoicesResultCharUrl(int width,
                                          int height,
                                          String choiceName) {

        URL applicationUrl = getApplicationUrl();
        StringBuilder url = new StringBuilder(applicationUrl.toString());

        url.append("/io/generateChart");

        Map<String, Object> params = Maps.newHashMap();
        params.put("width", width);
        params.put("height", height);
        params.put("title", "");
        params.put("values", choicesResults.get(choiceName));
        params.put("subtitles", subtitles.get(choiceName));

        UrlHelper.buildParametersString(params, url, "&");

        String chartUrl = url.toString();
        return chartUrl;

    }

    public String getChoicesAsString() {
        List<String> choices = getChoices();
        return StringUtil.join(choices, ",", true);
    }
}
