/*
 * #%L
 * Pollen :: Persistence
 * $Id: PollAccountDAOImpl.java 3760 2012-12-01 12:28:56Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.1/pollen-persistence/src/main/java/org/chorem/pollen/business/persistence/PollAccountDAOImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.business.persistence;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.PagerBeanUtil;

import java.util.List;

public class PollAccountDAOImpl<E extends PollAccount> extends PollAccountDAOAbstract<E> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollAccountDAOImpl.class);

    /**
     * Get a restricted poll account for a given poll (by his {@link Poll#getPollId()})
     * using the {@link PollAccount#getAccountId()} of the pollAccount
     * and/or the {@link PollAccount#getEmail()}.
     * <p/>
     * <strong>Note:</strong> {@code accountId} and {@code user} can not
     * be null at the same time.
     *
     * @param pollId    pollId of the poll
     * @param accountId the accountId of the pollAccount to find
     * @param user      the connected user account to test
     * @return the found pollAccount, or {@code null} if not foud
     * @throws TopiaException if any db pb
     */
    public E findRestrictedPollAccount(String pollId,
                                       String accountId,
                                       UserAccount user) throws TopiaException {

        Preconditions.checkArgument(StringUtils.isNotBlank(pollId));
        Preconditions.checkArgument(StringUtils.isNotBlank(accountId) ||
                                    user != null);

        String hql = "SELECT p.pollAccount FROM PersonToListImpl p, PollImpl poll WHERE " +
                     "poll.pollId = :pollId AND " +
                     "p in elements (poll.votingList.pollAccountPersonToList) AND ";


        boolean withAccountId = StringUtils.isNotBlank(accountId);
        boolean withEmail = user != null;

        List<String> params = Lists.newArrayList("pollId", pollId);

        if (withAccountId && withEmail) {

            // try with both email or accountId
            params.add("accountId");
            params.add(accountId);
            params.add("email");
            params.add(user.getEmail());
            hql += "(p.pollAccount.accountId = :accountId OR p.pollAccount.email = :email)";
        } else if (withAccountId) {

            // try only with accountId
            params.add("accountId");
            params.add(accountId);
            hql += "p.pollAccount.accountId = :accountId";
        } else {

            // try only with email
            params.add("email");
            params.add(user.getEmail());
            hql += "p.pollAccount.email = :email";
        }

        E result = null;
        List<E> results = findAllByQuery(hql, params.toArray());
        if (CollectionUtils.isEmpty(results)) {
            result = null;
        } else if (results.size() == 1) {
            result = results.get(0);
        } else {

            // limit case when there is two different poll account
            // (one for connected user and one for incoming accountId)
            // In this case, we must use the accountId one
            for (E e : results) {
                String accountId1 = e.getAccountId();
                if (accountId1.equals(accountId)) {
                    result = e;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Get the pollAccount of a voter for a given {@code pollId} using his
     * {@code accountId} or {@code user}.
     * <p/>
     * <strong>Note:</strong> {@code accountId} and {@code user} can not
     * be null at the same time.
     *
     * @param pollId    the poll where to seek
     * @param accountId the account id to test
     * @param user      the connected user account to test
     * @return the found pollAccount, or {@code null} if not found
     * @throws TopiaException if any db pb
     */
    public E findVoterPollAccount(String pollId,
                                  String accountId,
                                  UserAccount user) throws TopiaException {

        Preconditions.checkArgument(StringUtils.isNotBlank(pollId));
        Preconditions.checkArgument(StringUtils.isNotBlank(accountId) ||
                                    user != null);

        String hql = "SELECT distinct(e) FROM PollAccountImpl e, PollImpl p, VoteImpl v WHERE " +
                     "p.pollId = :pollId AND " +
                     "v in elements (p.vote) AND " +
                     "e = v.pollAccount AND ";

        boolean withAccountId = StringUtils.isNotBlank(accountId);
        boolean withEmail = user != null;

        List<Object> params = Lists.<Object>newArrayList("pollId", pollId);

        if (withAccountId && withEmail) {

            // try with both email or accountId
            params.add("accountId");
            params.add(accountId);
            params.add("user");
            params.add(user);
            hql += "(e.accountId = :accountId OR e.userAccount = :user)";
        } else if (withAccountId) {

            // try only with accountId
            params.add("accountId");
            params.add(accountId);
            hql += "e.accountId = :accountId";
        } else {

            // try only with email
            params.add("user");
            params.add(user);
            hql += "e.userAccount = :user";
        }

        List<E> allVoter = findAllByQuery(hql, params.toArray());
        E result = null;
        if (CollectionUtils.isNotEmpty(allVoter)) {
            if (allVoter.size() > 1) {
                if (log.isWarnEnabled()) {
                    log.warn("Multiple voter found for poll " + pollId);
                    for (E e : allVoter) {
                        if (log.isWarnEnabled()) {
                            log.warn("accountId: " + e.getAccountId());
                        }
                    }
                }
            }
            result = allVoter.get(0);
        }
        return result;
    }

    public List<E> findFavoriteListUsers(PersonList favoriteList,
                                         TopiaPagerBean pager) throws TopiaException {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(pager);

        int records = favoriteList.sizePollAccount();
        pager.setRecords(records);
        PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

        String hql = createSimpleQuery("e") + " WHERE e.personList = :owner";

        List<E> result = findAllByQueryAndPager(hql, pager, "owner", favoriteList);
        return result;
    }

    public boolean isPollAccountExists(PersonList personListToUpdate,
                                       PollAccount pollAccount) throws TopiaException {

        Preconditions.checkNotNull(personListToUpdate);
        Preconditions.checkNotNull(pollAccount);

        String hql = "SELECT COUNT(*) " + createSimpleQuery("e") + " WHERE " +
                     "e.personList = :person AND " +
                     "e.email = :email";
        String pollAccountId = pollAccount.getTopiaId();

        List<Object> params = Lists.<Object>newArrayList(
                "person", personListToUpdate,
                "email", pollAccount.getEmail());

        if (StringUtils.isNotBlank(pollAccountId)) {

            // check also by account topiaId

            hql += " AND e.topiaId = :topiaId";

            params.add("topiaId");
            params.add(pollAccountId);

        }

        boolean result = existsByQuery(hql, params.toArray());
        return result;
    }

}
