/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.utils;

import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.ChoiceTypeDAO;
import org.chorem.pollen.business.persistence.PollTypeDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.persistence.VoteCountingDAO;
import org.hibernate.exception.SQLGrammarException;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaDAO;

/**
 * Charge et génère le contexte global.
 *
 * @author rannou
 * @version $Id: ContextUtil.java 2731 2009-08-18 12:22:46Z nrannou $
 */
public class ContextUtil {
    /**
     * Instance de la classe (patron singleton)
     */
    private static ContextUtil instance = null;

    /**
     * Contexte global
     */
    private TopiaContext context = null;

    /**
     * Propriétés de la base de données.
     */
    private Properties conf;

    /** log. */
    private static final Log log = LogFactory.getLog(ContextUtil.class);

    /**
     * Retourne le contexte global.
     *
     * @return le contexte global
     */
    public TopiaContext getContext() {
        return context;
    }

    /**
     * Retourne la configuration de la base de données.
     *
     * @return la configuration
     */
    public Properties getConf() {
        return conf;
    }

    /**
     * Construction du contexte global
     */
    public void buildContext() {
        if (context == null || context.isClosed()) {
            try {
                if (log.isInfoEnabled()) {
                    log.info("Construction du contexte global...");
                }
                context = TopiaContextFactory.getContext(conf);
            } catch (TopiaNotFoundException e) {
                if (log.isErrorEnabled()) {
                    log.error("Erreur lors de la construction du contexte", e);
                }
                e.printStackTrace();
            }
        }
    }

    /**
     * Fermeture du contexte global
     */
    public void closeContext() {
        if (context != null && !context.isClosed()) {
            try {
                if (log.isInfoEnabled()) {
                    log.info("Fermeture du contexte global...");
                }
                context.closeContext();
            } catch (TopiaException e) {
                if (log.isErrorEnabled()) {
                    log.error("Erreur lors de la fermeture du contexte", e);
                }
                e.printStackTrace();
            }
        }
    }

    /**
     * Retourne l'instance unique de la classe (patron singleton)
     *
     * @return L'instance de la classe
     */
    public static ContextUtil getInstance() {
        if (instance == null) {
            instance = new ContextUtil();
        }
        return instance;
    }

    /**
     * Constructeur. Construit le contexte global après avoir lu le fichier de
     * configuration de Pollen.
     */
    private ContextUtil() {

        // Chargement du fichier de configuration
        conf = PropertiesLoader.loadPropertiesFile("pollen.properties");
        conf.setProperty("topia.persistence.classes", PollenModelDAOHelper
                .getImplementationClassesAsString());

        // Construction du contexte global
        buildContext();

        // Initialisation de la base de données. Si elle n'existe pas on la crée
        // FIXME Il ne faudrait pas se baser sur l'exception SQLGrammarException pour détecter que la base n'existe pas
        try {
            initDB();
        } catch (TopiaException e) {
            log.error("Base de données inexistante");
            if (e.getCause() instanceof SQLGrammarException) {
                log.info("Création de la base de données...");
                try {
                    context.createSchema();
                    log.info("Base de données créée");
                    initDB();
                } catch (TopiaException ex) {
                    log.error("Échec lors de la création de la base", ex);
                }
            }
        }
    }

    /**
     * Chargement des valeurs initiales dans la base de données.
     *
     * @throws TopiaException
     */
    protected void initDB() throws TopiaException {
        TopiaContext transaction = null;
        transaction = context.beginTransaction();

        // Chargement des types de choix dans la table choiceType
        ChoiceTypeDAO choiceTypeDAO = PollenModelDAOHelper
                .getChoiceTypeDAO(transaction);
        loadDB("choiceType", choiceTypeDAO);
        transaction.commitTransaction();

        // Chargement des types de sondage dans la table pollType
        PollTypeDAO pollTypeDAO = PollenModelDAOHelper
                .getPollTypeDAO(transaction);
        loadDB("pollType", pollTypeDAO);
        transaction.commitTransaction();

        // Chargement des types de dépouillement dans la table voteCounting
        VoteCountingDAO voteCountingDAO = PollenModelDAOHelper
                .getVoteCountingDAO(transaction);
        loadDB("voteCounting", voteCountingDAO);
        transaction.commitTransaction();

        // Chargement de l'utilisateur par défaut
        UserAccountDAO userAccountDAO = PollenModelDAOHelper
                .getUserAccountDAO(transaction);
        loadAdmin(userAccountDAO);
        transaction.commitTransaction();

        transaction.closeContext();
    }

    /**
     * Chargement des types dans la base de données à partir du fichier de
     * configuration.
     *
     * @param element la clé du type dans le fichier de configuration
     * @param dao le DAO correspondant au type
     * @throws TopiaException
     */
    private void loadDB(String element, TopiaDAO dao) throws TopiaException {
        if (dao.findAll().size() == 0) {
            String strType = conf.getProperty(element);
            String[] tabTypes = strType.split(",");
            for (int i = 0; i < tabTypes.length; i++) {
                dao.create("name", tabTypes[i]);
            }

            if (log.isInfoEnabled()) {
                log.info("Types " + element + " créés.");
            }
        }
    }

    /**
     * Chargement de l'utilisateur par défaut dans la base de données à partir
     * du fichier de configuration.
     *
     * @param dao le DAO pour les utilisateurs
     * @throws TopiaException
     */
    private void loadAdmin(UserAccountDAO dao) throws TopiaException {
        if (dao.findAllByAdministrator(true).size() == 0) {
            UserAccount userEntity = dao.create();
            userEntity.setLogin(conf.getProperty("adminLogin"));
            userEntity.setPassword(MD5
                    .encode(conf.getProperty("adminPassword")));
            userEntity.setEmail(conf.getProperty("adminEmail"));
            userEntity.setAdministrator(true);

            if (log.isInfoEnabled()) {
                log.info("Utilisateur " + userEntity.getLogin() + " créé.");
            }
        }
    }

    /**
     * Méthode exécutée lorsqu'une exception est détectée.
     *
     * @param e l'exception
     * @param transaction la transaction courante
     */
    public static void doCatch(TopiaException e, TopiaContext transaction) {

        // rollback de la transaction courante
        if (transaction != null) {
            try {
                transaction.rollbackTransaction();
            } catch (TopiaException ex) {
                if (log.isErrorEnabled()) {
                    log.error("Échec lors du rollback de la transaction", ex);
                }
            } finally {
                try {
                    transaction.closeContext();
                } catch (TopiaException ex) {
                    if (log.isErrorEnabled()) {
                        log.error(
                                "Échec lors de la fermeture de la transaction",
                                ex);
                    }
                }
            }
        }

        if (log.isErrorEnabled()) {
            log.error("Échec lors du déroulement de la transaction", e);
        }
    }
}