/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.utils;

import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.ChoiceTypeDAO;
import org.chorem.pollen.business.persistence.PollTypeDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.VoteCountingDAO;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;

/**
 * Charge et génère le contexte global.
 *
 * @author rannou
 * @version $Id: ContextUtil.java 2615 2009-07-01 13:47:45Z nrannou $
 */
public class ContextUtil {
    /**
     * Instance de la classe (patron singleton)
     */
    private static ContextUtil instance = null;

    /**
     * Contexte global
     */
    private TopiaContext context = null;

    /**
     * Transaction courante
     */
    private TopiaContext transaction = null;

    /**
     * Propriétés de la base de données.
     */
    private Properties conf;

    /** log. */
    private static final Log log = LogFactory.getLog(ContextUtil.class);

    /**
     * Retourne le contexte global.
     *
     * @return le contexte global
     */
    public TopiaContext getContext() {
        return context;
    }

    /**
     * Construction du contexte global
     */
    public void buildContext() {
        if (context == null || context.isClosed()) {
            try {
                if (log.isInfoEnabled()) {
                    log.info("Construction du contexte global...");
                }
                context = TopiaContextFactory.getContext(conf);
            } catch (TopiaNotFoundException e) {
                if (log.isErrorEnabled()) {
                    log.error("Erreur lors de la construction du contexte");
                }
                e.printStackTrace();
            }
        }
    }

    /**
     * Retourne l'instance unique de la classe (patron singleton)
     *
     * @return L'instance de la classe
     */
    public static ContextUtil getInstance() {
        if (instance == null) {
            instance = new ContextUtil();
        }
        return instance;
    }

    /**
     * Constructeur. Construit le contexte global après avoir lu le fichier de
     * configuration de Pollen.
     */
    private ContextUtil() {

        // Chargement du fichier de configuration
        conf = PropertiesLoader.loadPropertiesFile("pollen.properties");
        conf.setProperty("topia.persistence.classes", PollenModelDAOHelper
                .getImplementationClassesAsString());

        // Construction du contexte global
        buildContext();

        // Initialisation de la base de données
        initDB();
    }

    /**
     * Chargement des types dans la base de données à partir du fichier de
     * configuration.
     *
     * @throws TopiaException
     */
    protected void initDB() {
        try {
            transaction = context.beginTransaction();

            // Chargement des types de choix dans la table choiceType
            ChoiceTypeDAO choiceTypeDAO = PollenModelDAOHelper
                    .getChoiceTypeDAO(transaction);
            loadDB("choiceType", choiceTypeDAO);

            // Chargement des types de sondage dans la table pollType
            PollTypeDAO pollTypeDAO = PollenModelDAOHelper
                    .getPollTypeDAO(transaction);
            loadDB("pollType", pollTypeDAO);

            // Chargement des types de dépouillement dans la table voteCounting
            VoteCountingDAO voteCountingDAO = PollenModelDAOHelper
                    .getVoteCountingDAO(transaction);
            loadDB("voteCounting", voteCountingDAO);

            transaction.closeContext();
        } catch (TopiaException e) {
            doCatch(e);
        }
    }

    /**
     * Chargement des types dans la base de données à partir du fichier de
     * configuration.
     *
     * @param element la clé du type dans le fichier de configuration
     * @param dao le DAO correspondant au type
     * @throws TopiaException
     */
    private void loadDB(String element, TopiaDAO dao) throws TopiaException {
        List<TopiaEntity> types = dao.findAll();
        if (types.size() == 0) {
            String strType = conf.getProperty(element);
            String[] tabTypes = strType.split(",");
            for (int i = 0; i < tabTypes.length; i++) {
                dao.create("name", tabTypes[i]);
            }
        }
        transaction.commitTransaction();
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}