/**
 * *##% Callao TransactionServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Account;
import org.chorem.callao.entity.Entry;
import org.chorem.callao.entity.EntryDAO;
import org.chorem.callao.entity.Journal;
import org.chorem.callao.entity.TimeSpan;
import org.chorem.callao.entity.Transaction;
import org.chorem.callao.entity.TransactionDAO;
import org.chorem.callao.service.convertObject.ConvertTransaction;
import org.chorem.callao.service.dto.EntryDTO;
import org.chorem.callao.service.dto.LogDTO;
import org.chorem.callao.service.dto.TransactionDTO;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Cette classe permet la création d'une transaction comptable dans l'application.
 * Toute action sur une transaction entraîne automatiquement une création de log.
 * Une transaction est composée d'entrées comptables.
 * Les actions sur les transactions sont soumises au statut de celle-ci, si elle
 * est dans une période bloquée ou non.
 *
 * @author Rémi Chapelet
 */
public class TransactionServiceImpl {

    private static final Log log = LogFactory.getLog(TransactionServiceImpl.class);

	private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private LogServiceImpl logServiceImpl = new LogServiceImpl();

    private ConvertTransaction convertTransaction = new ConvertTransaction();

    private EntryServiceImpl entryServiceImpl = new EntryServiceImpl();

    private TimeSpanServiceImpl timeSpanServiceImpl = new TimeSpanServiceImpl();

    /**
     * Permet de créer une transaction comptable. Pour ajouter une transaction,
     * le timeSpan doit être non bloqué.
     * @param entryDate
     * @param voucherRef
     * @param description
     * @param timespan
     * @param journal
     * @return
     */
    public String createTransaction (Date entryDate, String voucherRef, String description,TimeSpan timeSpan,Journal journal)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            String isCorrectTransaction = isCorrectTransaction(timeSpan,journal);
            if ( isCorrectTransaction.equals(ServiceHelper.RESPOND_SUCCESS) )
            {
                // Creation de la transaction
                Transaction newtransaction = transactionDAO.create();
                newtransaction.setTransDate(entryDate);
                newtransaction.setVoucherRef(voucherRef);
                newtransaction.setDescription(description);
                newtransaction.setTimeSpan(timeSpan);
                newtransaction.setJournal(journal);
                // Création BDD
                topiaContext.commitTransaction();
                if (log.isInfoEnabled()) {
                    log.info("Ajout avec succes de la transaction");
                }
                /**
                 * Il renvoie soit l'identifiant topiaId ou SUCCESS.
                 */
                result = newtransaction.getTopiaId();
                //result = ServiceHelper.RESPOND_SUCCESS;
            } else {
                result = isCorrectTransaction;
            }
            // Fermeture BDD
            topiaContext.closeContext();            
        }catch (TopiaException e) {
            log.error(e);
        }        
        return result;
    }

    /**
     * Permet de créer une transaction à partir d'une transaction DTO.
     * ATTENTION : si le journal ou timespan n'existe pas, l'ajout ne peut se
     * faire. Si on souhaite, on peut créer le journal automatiquement lors de
     * l'ajout de la transaction si il n'existe pas. Pour le timeSpan, il n'est
     * pas cohérent qu'il n'existe pas. On ne peut travailler que sur une période
     * comptable existante, et par conséquent il ne faut pas le créer.
     * @param transactionDTO
     * @return
     */
    public String createTransaction (TransactionDTO transactionDTO)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Vérifie que transactionDTO possède un journalDTO et un timeSpanDTO
        if (transactionDTO.getJournalDTO() == null || transactionDTO.getTimeSpanDTO() == null)
        {
            if (log.isErrorEnabled()) {
                log.error("L'objet transactionDTO ne possède pas de timeSpan ET/OU de journal.");
            }
            result = ServiceHelper.TRANSACTION_NOT_JOURNAL;
        } else
        {
            // Recherche du journal
            JournalServiceImpl journalServiceImpl = new JournalServiceImpl();
            Journal journal = journalServiceImpl.searchJournalWithLabel(transactionDTO.getJournalDTO().getLabel());
            // Recherche du timeSpan
            TimeSpan timeSpan = timeSpanServiceImpl.searchTimeSpanByDate(transactionDTO.getTimeSpanDTO().getBeginTimeSpan());

            // Vérification si timeSpan et Journal existe bien
            if (journal == null)
            {
                if (log.isErrorEnabled()) {
                    log.error("Création transaction : Le journal "+ transactionDTO.getJournalDTO().getLabel() +" n'existe pas.");
                }
                result = ServiceHelper.TRANSACTION_NOT_JOURNAL;
            } else {
                if (timeSpan == null)
                {
                    if (log.isErrorEnabled()) {
                        log.error("Création transaction : Le timeSpan "+ transactionDTO.getTimeSpanDTO().toString() +" n'existe pas.");
                    }
                    result = ServiceHelper.TRANSACTION_NOT_TIMESPAN;
                } else {
                    result = createTransaction(transactionDTO.getEntryDate(),transactionDTO.getVoucherRef(),transactionDTO.getDescription(),timeSpan,journal);
                }
            }
        }
        return result;
    }


    /**
     * Permet de modifier une transaction. Cette dernière est identifiée par la
     * clé topiaId.
     * @param topiaId
     * @param entryDate
     * @param voucherRef
     * @param description
     * @param timespan
     * @param journal
     * @return
     */
    public String modifyTransaction (String topiaId, Date entryDate, String voucherRef, String description,TimeSpan timeSpan,Journal journal)
    {
        String result = ServiceHelper.RESPOND_ERROR;        
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Vérification que la transaction existe bien
            Transaction transaction = transactionDAO.findByTopiaId(topiaId);
            if (transaction == null)
            {
                if (log.isInfoEnabled()) {
                        log.info("La transaction "+voucherRef+" n'existe pas. (identifiant " +
                                topiaId+")");
                }
                result = ServiceHelper.TRANSACTION_NOT_EXIST;
            } else {
                // Vérification si la transaction possède de bons paramètres
                String correct = isCorrectTransaction(timeSpan,journal);
                // Si la transaction est correcte
                if (correct.equals(ServiceHelper.RESPOND_SUCCESS))
                {
                    // Modification
                    transaction.setTransDate(entryDate);
                    transaction.setVoucherRef(voucherRef);
                    transaction.setDescription(description);
                    transaction.setTimeSpan(timeSpan);
                    transaction.setJournal(journal);
                    transaction.update();
                    // Création BDD
                    topiaContext.commitTransaction();
                    result = ServiceHelper.RESPOND_SUCCESS;
                }
            }
            // Fermeture BDD
            topiaContext.closeContext();            
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }


    /**
     * Permet de modifier une transaction à partir d'une transaction DTO.
     * Elle va rechercher le journal et le timespan correspondants.
     * @param transactionDTO
     * @return
     */
    public String modifyTransaction (TransactionDTO transactionDTO)
    {
        // Recherche du journal
        JournalServiceImpl journalServiceImpl = new JournalServiceImpl();
        Journal journal = journalServiceImpl.searchJournalWithLabel(transactionDTO.getJournalDTO().getLabel());
        // Recherche du timeSpan
        TimeSpan timeSpan = timeSpanServiceImpl.searchTimeSpanByDate(transactionDTO.getTimeSpanDTO().getBeginTimeSpan());
        // Appel pour la modification
        String result = modifyTransaction (transactionDTO.getId(),transactionDTO.getEntryDate(),
                transactionDTO.getVoucherRef(),transactionDTO.getDescription()
                ,timeSpan,journal);        
        return result;
    }


    /**
     * Recherche une transaction unique avec son identifiant TopiaId.
     * @param topiaId
     * @return
     */
    public Transaction searchTransactionWithTopiaId (String topiaId)
    {
        Transaction transaction = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Recherche
            transaction = transactionDAO.findByTopiaId(topiaId);
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return transaction;
    }


    /**
     * Recherche une transaction unique à partir d'une transaction DTO.
     * @param transactionDTO
     * @return
     */
    public Transaction searchTransactionWithTopiaId (TransactionDTO transactionDTO)
    {
        Transaction transaction = searchTransactionWithTopiaId(transactionDTO.getId());
        return transaction;
    }


    /**
     * Recherche une transaction dans la base de données, et renvoie celle-ci
     * sous format DTO. Nous ne pouvons pas appeler la méthode searchTransactionWithTopiaId
     * du fait que lors de la convertion, nous avons besoin du journal et du
     * timeSpan. Or il ne faut donc pas de fermeture de connexion à la base de
     * données pour celà.
     * @param topiaId
     * @return
     */
    public TransactionDTO searchTransactionDTO (String topiaId)
    {
        TransactionDTO transactionDTO = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Recherche
            Transaction transaction = transactionDAO.findByTopiaId(topiaId);
            // Converti la transaction en DTO
            convertTransaction.setTransaction(rootContext);
            transactionDTO = convertTransaction.transactionEntityToDto(transaction);
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}        
        return transactionDTO;
    }


    /**
     * Recherche toutes les transactions qui correspondent à un timespan.
     * @param timeSpan
     * @return
     */
    public List<Transaction> searchListTransactionWithTimeSpan (TimeSpan timeSpan)
    {
        List<Transaction> listTransaction = null;
        if ( timeSpan!= null )
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
                // Recherche
                listTransaction = transactionDAO.findAllByTimeSpan(timeSpan);
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return listTransaction;
    }


    /**
     * Recherche les transactions DTO d'un timeSpan
     * @param timeSpan
     * @return
     */
    public List<TransactionDTO> searchListTransactionDTOWithTimeSpan (TimeSpan timeSpan)
    {
        List<TransactionDTO> listTransactionDTO = new ArrayList<TransactionDTO>();
        if ( timeSpan!= null )
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
                // Recherche
                List<Transaction> listTransaction = transactionDAO.findAllByTimeSpan(timeSpan);               
                // Converti la transaction en DTO
                convertTransaction.setTransaction(rootContext);
                for (Transaction transaction : listTransaction)
                {
                    TransactionDTO transactionDTO = convertTransaction.transactionEntityToDto(transaction);
                    listTransactionDTO.add(transactionDTO);
                }
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return listTransactionDTO;
    }


    /**
     * Recherche toutes les transactions appartenant à un même journal donné
     * en paramètre.
     * @param journal
     * @return
     */
    public List<Transaction> searchListTransactionWithJournal (Journal journal)
    {
        List<Transaction> listTransaction = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Recherche
            listTransaction = transactionDAO.findAllByJournal(journal);
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return listTransaction;
    }


    /**
     * Permet de rechercher toutes les transactions dans la base de données,
     * avec en retour une liste de transactions au format DTO.
     * @return
     */
    public List<TransactionDTO> getAllTransactionDTO ()
    {
        List<TransactionDTO> listTransactionDTO = new ArrayList<TransactionDTO>();
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Recherche
            List<Transaction> listTransaction = transactionDAO.findAll();
            // Converti la transaction en DTO
            convertTransaction.setTransaction(rootContext);
            for (Transaction transaction : listTransaction)
            {
                TransactionDTO transactionDTO = convertTransaction.transactionEntityToDto(transaction);
                listTransactionDTO.add(transactionDTO);
            }
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return listTransactionDTO;
    }


    /**
     * Cette méthode permet de vérifier si une transaction possède des paramètres
     * corrects.
     * Un journal doit exister.
     * Un timeSpan doit exister et être non bloqué.
     * @param timeSpan
     * @param journal
     * @return
     */
    public String isCorrectTransaction (TimeSpan timeSpan, Journal journal)
    {
        String result = ServiceHelper.RESPOND_SUCCESS;
        // Vérification si timeSpan et Journal existe bien
        if (journal == null)
        {
            if (log.isErrorEnabled()) {
                log.error("Création transaction : Le journal n'existe pas.");
            }
            result = ServiceHelper.TRANSACTION_NOT_JOURNAL;
        }
        if (timeSpan == null)
        {
            if (log.isErrorEnabled()) {
                log.error("Création transaction : Le timeSpan n'existe pas.");
            }
            result = ServiceHelper.TRANSACTION_NOT_TIMESPAN;
        } else {
            // Vérifie si le timeSpan est bloqué
            if (timeSpan.getLocked())
            {
                if (log.isErrorEnabled()) {
                        log.error("Le timeSpan est bloqué ! Il est donc impossible" +
                                " d'ajouter une transaction sur cette période. (" +
                                "Période du "+timeSpan+")");
                }
                result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
            }
        }
        return result;
    }


    /**
     * Permet d'effacer une transaction dans la base de données.
     * ATTENTION : Elle supprime également toutes les entées comptables
     * associées. Une transaction appartenant à un timeSpan bloqué ne peut être
     * supprimée.
     * @return
     */
    public String removeTransaction (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);
            // Recherche de l'entry
            Transaction transaction = transactionDAO.findByTopiaId(topiaId);
            // Transaction doit exister
            if (transaction != null)
            {
                // Vérifie si la transaction est non bloquée.
                if (!isTransactionBlocked(transaction))
                {
                    // Supprime tous les entrys de la transactions
                    List<Entry> listEntry = entryServiceImpl.searchEntryWithTransaction(transaction);
                    for (Entry entry : listEntry)
                    {
                        result = removeEntry(entry.getTopiaId());
                    }
                    // Supprime la transaction
                    transactionDAO.delete(transaction);
                    // Création BDD
                    topiaContext.commitTransaction();
                    if (log.isInfoEnabled()) {
                            log.info("La transaction a été supprimée avec succès.");
                    }
                    result = ServiceHelper.RESPOND_SUCCESS;
                } else {
                    // Transaction bloquée
                    result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
                }
            } else {
                result = ServiceHelper.TRANSACTION_NOT_EXIST;
            }
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return result;
    }


    /**
     * Supprime une transaction au format DTO.
     * @param transactionDTO
     * @return
     */
    public String removeTransaction (TransactionDTO transactionDTO)
    {
        String result = removeTransaction(transactionDTO.getId());
        return result;
    }


    /**
     * Permet d'ajouter une entrée comptable pour une transaction donnée.
     * ATTENTION : la transaction doit être NON bloquée.
     *
     * @param description
     * @param amount
     * @param debit
     * @param lettering
     * @param detail
     * @param transaction
     * @return
     */
    public String addEntry (String description,String amount, boolean debit, 
            String lettering,String detail,Transaction transaction,Account account)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Si la transaction est non bloquée
        if (!isTransactionBlocked(transaction))
        {
            /**
             * Si il renvoie succes et non l'identifiant (à commenter dans la classe correspondante)
             */
            result = entryServiceImpl.createEntry(description, amount, debit, lettering, detail, transaction, account);
        } else {
            result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
        }
        return result;
    }

    /**
     * Permet d'ajouter une entrée comptable pour une transaction donnée.
     * L'entrée est sous format DTO.
     *
     * @param entryDTO
     * @return
     */
    public String addEntry (EntryDTO entryDTO)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Recherche TransactionDTO
        TransactionDTO transactonDTO = entryDTO.getTransactionDTO();
        Transaction transaction = searchTransactionWithTopiaId(transactonDTO);
        // Si la transaction est non bloquée
        if (!isTransactionBlocked(transaction))
        {
            /**
             * Si il renvoie succes et non l'identifiant (à commenter dans la classe correspondante)
             */
            result = entryServiceImpl.createEntry(entryDTO);
        } else {
            result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
        }
        return result;
    }


    /**
     * Permet de modifier une entrée comptable.
     * ATTENTION : la transaction doit être NON bloquée.
     *
     * @param topiaId
     * @param description
     * @param amount
     * @param debit
     * @param lettering
     * @param detail
     * @param transaction
     * @param account
     * @return
     */
    public String modifyEntry (String topiaId,String description,String amount, boolean debit,
            String lettering,String detail,Transaction transaction,Account account)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        if (!isTransactionBlocked(transaction))
        {
            result = entryServiceImpl.modifyEntry(topiaId,description, amount,
                    debit, lettering, detail, account);
        }  else {
            result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
        }
        return result;
    }


    /**
     * Permet de modifier une entrée.
     * @param entryDTO
     * @return
     */
    public String modifyEntry (EntryDTO entryDTO)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Recherche TransactionDTO
        TransactionDTO transactonDTO = entryDTO.getTransactionDTO();
        Transaction transaction = searchTransactionWithTopiaId(transactonDTO);
        if (!isTransactionBlocked(transaction))
        {
            result = entryServiceImpl.modifyEntry(entryDTO);
        }  else {
            result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
        }
        return result;
    }


    /**
     * Permet d'effacer une entrée comptable. Elle est identifiée par son identifiant
     * topiaID. Une entrée peut être supprimée seulement si elle est non bloquée.
     * @param topiaId
     * @return
     */
    public String removeEntry (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
            // Recherche de l'entry
            Entry entryDelete = entryDAO.findByTopiaId(topiaId);
            // Entry doit exister
            if (entryDelete != null)
            {
                // Prend la transaction dont appartient entry
                Transaction transaction = entryDelete.getTransaction();
                // Vérifie si la transaction est non bloquée.
                if (!isTransactionBlocked(transaction))
                {
                    // Création d'un log DTO
                    LogDTO logDTO = new LogDTO ("", new Date(),ServiceHelper.LOG_REMOVE,transaction.getTransDate(),
                        transaction.getVoucherRef(), transaction.getDescription(), entryDelete.getDescription(),entryDelete.getAmount(),
                        entryDelete.getDebit(), entryDelete.getLettering());
                    // Appel pour supprimer l'entrée
                    result = entryServiceImpl.removeEntry(topiaId);
                    // Si la suppression s'est bien déroulée, création log
                    if ( result.equals(ServiceHelper.RESPOND_SUCCESS) )
                    {
                        // Ajout du log
                        logServiceImpl.addLog(logDTO);
                    }
                } else {
                    result = ServiceHelper.TRANSACTION_TIMESPAN_BLOCKED;
                }
            } else {
                result = ServiceHelper.ENTRY_NOT_EXIST;
            }
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return result;
    }


    /**
     * Permet de supprimer une entrée à partir d'un DTO.
     * @param entryDTO
     * @return
     */
    public String removeEntry (EntryDTO entryDTO)
    {
        String result = removeEntry(entryDTO.getId());
        return result;
    }

    
    /**
     * Permet de vérifier si une transaction est dans une période bloquée ou non
     * @param transaction
     * @return
     */
    public boolean isTransactionBlocked (Transaction transaction)
    {
        boolean isTransactionBlocked = true;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TransactionDAO transactionDAO = CallaoDAOHelper.getTransactionDAO(topiaContext);                       
            // Recherche
            Transaction transactionSearch = transactionDAO.findByTopiaId(transaction.getTopiaId());
            isTransactionBlocked = timeSpanServiceImpl.isTimeSpanBlocked(transactionSearch.getTimeSpan());
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}        
        return isTransactionBlocked;
    }

    
    /**
     * Permet de vérifier si une transaction est équilibrée.
     * @param transaction
     * @return
     */
    public boolean isTransactionBalanced (Transaction transaction)
    {
        List<EntryDTO> ListEntryDTO = entryServiceImpl.searchEntryDTOWithTransaction(transaction);
        float debit = 0;
        float credit = 0;
        // Pour toutes les entrées comptables trouvées
        for ( EntryDTO entryDTO : ListEntryDTO )
        {
            // Si entrée au debit
            if ( entryDTO.isDebit() )
            {
                debit = debit + Float.parseFloat(entryDTO.getAmount().replaceFirst(",", "."));
            } else {
                credit = credit + Float.parseFloat(entryDTO.getAmount().replaceFirst(",", "."));
            }
        }
        // Vérifie si le debit == credit
        boolean isTransactionBalanced = debit == credit;
        return isTransactionBalanced;
    }

}