/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.api.jmx;

/**
 * MBean interface for exposing information about a registered observation
 * listener.
 *
 * @see <a href="https://issues.apache.org/jira/browse/JCR-3608">JCR-3608</a>
 */
public interface EventListenerMBean {

    /** Class name of the event listener */
    String getClassName();

    /** Stack trace of where the listener was registered */
    String getInitStackTrace();

    /** Event types of the listener registration */
    int getEventTypes();

    /** Absolute path of the listener registration */
    String getAbsPath();

    /** Whether the listener registration is deep */
    boolean isDeep();

    /** UUIDs of the listener registration */
    String[] getUuid();

    /** Node types of the listener registration */
    String[] getNodeTypeName();

    /** Whether the listener registration is non-local */
    boolean isNoLocal();

    /** Number of {@code onEvent()} calls made on the listener */
    long getEventDeliveries();

    /** Average number of {@code onEvent()} calls per hour */
    long getEventDeliveriesPerHour();

    /** Average time (in microseconds) taken per {@code onEvent()} call */
    long getMicrosecondsPerEventDelivery();

    /** Number of individual events delivered to the listener */
    long getEventsDelivered();

    /** Average number of individual events delivered per hour */
    long getEventsDeliveredPerHour();

    /** Average time (in microseconds) taken per event delivered */
    long getMicrosecondsPerEventDelivered();

    /** Ratio of time spent in event processing */
    double getRatioOfTimeSpentProcessingEvents();

    /** Is user information accessed without checking if an event is external? */
    boolean isUserInfoAccessedWithoutExternalsCheck();

    /** Is user information accessed from an external event? */
    boolean isUserInfoAccessedFromExternalEvent();

    /** Is date information accessed without checking if an event is external? */
    boolean isDateAccessedWithoutExternalsCheck();

    /** Is date information accessed from an external event? */
    boolean isDateAccessedFromExternalEvent();

}
