/*
 * Decompiled with CFR 0.152.
 */
package net.timewalker.ffmq4.utils.pool;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.jms.JMSException;
import net.timewalker.ffmq4.utils.pool.ObjectPoolException;
import net.timewalker.ffmq4.utils.pool.ObjectPoolMBean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class ObjectPool<T>
implements ObjectPoolMBean {
    private static final Log log = LogFactory.getLog(ObjectPool.class);
    public static final int WHEN_EXHAUSTED_FAIL = 0;
    public static final int WHEN_EXHAUSTED_BLOCK = 1;
    public static final int WHEN_EXHAUSTED_WAIT = 2;
    public static final int WHEN_EXHAUSTED_RETURN_NULL = 3;
    public static final int WHEN_EXHAUSTED_WAIT_RETURN_NULL = 4;
    private int maxIdle;
    private int minSize;
    private int maxSize;
    private int exhaustionPolicy;
    private long waitTimeout;
    private Set<T> all;
    private List<T> available;
    private boolean closed;
    private int pendingWaits;
    private Object closeLock = new Object();

    public ObjectPool(int minSize, int maxIdle, int maxSize, int exhaustionPolicy, long waitTimeout) throws JMSException {
        if (minSize < 0) {
            throw new ObjectPoolException("minSize cannot be negative");
        }
        if (minSize > maxSize) {
            throw new ObjectPoolException("minSize should be <= maxSize");
        }
        if (maxIdle < minSize || maxIdle > maxSize) {
            throw new ObjectPoolException("maxIdle should be between minSize and maxSize");
        }
        switch (exhaustionPolicy) {
            case 0: 
            case 1: 
            case 2: 
            case 3: 
            case 4: {
                break;
            }
            default: {
                throw new ObjectPoolException("Invalid exhaustion policy : " + exhaustionPolicy);
            }
        }
        if (waitTimeout < 0L) {
            throw new ObjectPoolException("waitTimeout cannot be negative");
        }
        this.minSize = minSize;
        this.maxIdle = maxIdle;
        this.maxSize = maxSize;
        this.exhaustionPolicy = exhaustionPolicy;
        this.waitTimeout = waitTimeout;
        this.all = new HashSet<T>(maxSize);
        this.available = new ArrayList<T>(maxSize);
    }

    protected void init() throws JMSException {
        for (int i = 0; i < this.minSize; ++i) {
            T poolObject = this.extendPool();
            this.available.add(poolObject);
        }
    }

    public synchronized T borrow() throws JMSException {
        if (this.closed) {
            throw new ObjectPoolException("Object pool is closed");
        }
        int availableCount = this.available.size();
        if (availableCount > 0) {
            return this.available.remove(availableCount - 1);
        }
        if (this.all.size() < this.maxSize) {
            return this.extendPool();
        }
        switch (this.exhaustionPolicy) {
            case 0: {
                throw new ObjectPoolException("Pool is exhausted (maxSize=" + this.maxSize + ")");
            }
            case 1: {
                return this.waitForAvailability();
            }
            case 2: {
                return this.waitForAvailability(this.waitTimeout, true);
            }
            case 3: {
                return null;
            }
            case 4: {
                return this.waitForAvailability(this.waitTimeout, false);
            }
        }
        throw new ObjectPoolException("Invalid exhaustion policy : " + this.exhaustionPolicy);
    }

    private T waitForAvailability() throws JMSException {
        ++this.pendingWaits;
        try {
            while (!this.closed) {
                int availableCount = this.available.size();
                if (availableCount > 0) {
                    T t = this.available.remove(availableCount - 1);
                    return t;
                }
                if (this.all.size() < this.maxSize) {
                    T t = this.extendPool();
                    return t;
                }
                try {
                    this.wait();
                }
                catch (InterruptedException e) {
                    log.error((Object)"waitForAvailability() was interrupted", (Throwable)e);
                }
            }
            throw new ObjectPoolException("Object pool was closed");
        }
        finally {
            --this.pendingWaits;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private T waitForAvailability(long waitTimeout, boolean throwExceptionOnTimeout) throws JMSException {
        ++this.pendingWaits;
        try {
            long now;
            long startTime = now = System.currentTimeMillis();
            while (!this.closed && now - startTime < waitTimeout) {
                int availableCount = this.available.size();
                if (availableCount > 0) {
                    T t = this.available.remove(availableCount - 1);
                    return t;
                }
                if (this.all.size() < this.maxSize) {
                    T t = this.extendPool();
                    return t;
                }
                try {
                    this.wait(waitTimeout - now + startTime);
                }
                catch (InterruptedException e) {
                    log.error((Object)"waitForAvailability() was interrupted", (Throwable)e);
                }
                now = System.currentTimeMillis();
            }
            if (this.closed) {
                throw new ObjectPoolException("Object pool was closed");
            }
            if (throwExceptionOnTimeout) {
                throw new ObjectPoolException("Timeout waiting for an available object");
            }
            T t = null;
            return t;
        }
        finally {
            --this.pendingWaits;
        }
    }

    public synchronized void release(T poolObject) {
        if (this.closed) {
            return;
        }
        if (this.pendingWaits > 0) {
            this.available.add(poolObject);
            this.notifyAll();
        } else if (this.available.size() >= this.maxIdle) {
            this.all.remove(poolObject);
            this.internalDestroyPoolObject(poolObject);
        } else {
            this.available.add(poolObject);
        }
    }

    private T extendPool() throws JMSException {
        T newPoolObject;
        try {
            newPoolObject = this.createPoolObject();
        }
        catch (JMSException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ObjectPoolException("Cannot create new pooled object", e);
        }
        this.all.add(newPoolObject);
        return newPoolObject;
    }

    private void internalDestroyPoolObject(T poolObject) {
        try {
            this.destroyPoolObject(poolObject);
        }
        catch (Exception e) {
            log.error((Object)"Cannot destroy new pooled object", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() {
        Object object = this.closeLock;
        synchronized (object) {
            if (this.closed) {
                return;
            }
            this.closed = true;
        }
        object = this;
        synchronized (object) {
            for (T poolObject : this.all) {
                this.internalDestroyPoolObject(poolObject);
            }
            this.all.clear();
            this.available.clear();
            this.notifyAll();
        }
    }

    protected abstract T createPoolObject() throws Exception;

    protected abstract void destroyPoolObject(T var1) throws Exception;

    @Override
    public int getThreadPoolMaxIdle() {
        return this.maxIdle;
    }

    @Override
    public int getThreadPoolMinSize() {
        return this.minSize;
    }

    @Override
    public int getThreadPoolMaxSize() {
        return this.maxSize;
    }

    @Override
    public int getThreadPoolExhaustionPolicy() {
        return this.exhaustionPolicy;
    }

    @Override
    public long getThreadPoolWaitTimeout() {
        return this.waitTimeout;
    }

    @Override
    public int getThreadPoolAvailableCount() {
        return this.available.size();
    }

    @Override
    public int getThreadPoolPendingWaits() {
        return this.pendingWaits;
    }

    @Override
    public int getThreadPoolSize() {
        return this.all.size();
    }

    public static String exhaustionPolicyAsString(int exhaustionPolicy) {
        switch (exhaustionPolicy) {
            case 0: {
                return "Fail";
            }
            case 1: {
                return "Block";
            }
            case 2: {
                return "Wait";
            }
            case 3: {
                return "Return null";
            }
            case 4: {
                return "Wait then return null";
            }
        }
        throw new IllegalArgumentException("Invalid exhaustion policy : " + exhaustionPolicy);
    }
}

