package fr.ifremer.tutti.ui.swing.util.attachment;

/*
 * #%L
 * Tutti :: UI
 * $Id: AttachmentEditorUIHandler.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/attachment/AttachmentEditorUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.nuiton.jaxx.application.ApplicationTechnicalException;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.Attachments;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.lang3.StringUtils;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JRootPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <morin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class AttachmentEditorUIHandler extends AbstractTuttiUIHandler<TuttiUIContext, AttachmentEditorUI> {

    public static final String CLOSE_DIALOG_ACTION = "closeDialog";

    public static final String SHOW_DIALOG_ACTION = "showDialog";

    @Override
    public void afterInit(AttachmentEditorUI ui) {

        ui.getFile().setDialogOwner(ui);
        ui.pack();
        ui.setResizable(true);
//        ui.setSize(800, 300);

        ComponentResizer cr = new ComponentResizer();
        cr.registerComponent(ui);
        ComponentMover cm = new ComponentMover();
        cm.setDragInsets(cr.getDragInsets());
        cm.registerComponent(ui);

        JRootPane rootPane = ui.getRootPane();

        KeyStroke shortcutClosePopup = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                shortcutClosePopup, CLOSE_DIALOG_ACTION);

        closeAction = new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                AttachmentEditorUIHandler.this.ui.dispose();
                AttachmentEditorUIHandler.this.ui.setVisible(false);
            }
        };

        openAction = new AbstractAction() {

            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                AttachmentEditorUIHandler.this.ui.setVisible(true);
            }
        };

        ImageIcon actionIcon = SwingUtil.createActionIcon("close-dialog");
        closeAction.putValue(Action.SMALL_ICON, actionIcon);
        closeAction.putValue(Action.LARGE_ICON_KEY, actionIcon);
        closeAction.putValue(Action.ACTION_COMMAND_KEY, "close");
        closeAction.putValue(Action.NAME, "close");
        closeAction.putValue(Action.SHORT_DESCRIPTION, t("tutti.attachmentEditor.action.closeAttachment.tip"));

        rootPane.getActionMap().put(CLOSE_DIALOG_ACTION, closeAction);
        rootPane.getActionMap().put(SHOW_DIALOG_ACTION, openAction);

        JButton closeButton = new JButton(closeAction);
        closeButton.setText(null);
        closeButton.setFocusPainted(false);
        closeButton.setRequestFocusEnabled(false);
        closeButton.setFocusable(false);

        JToolBar jToolBar = new JToolBar();
        jToolBar.setOpaque(false);
        jToolBar.add(closeAction);
        jToolBar.setBorderPainted(false);
        jToolBar.setFloatable(false);
        ui.getAttachmentBody().setRightDecoration(jToolBar);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getFile();
    }

    @Override
    public void onCloseUI() {
    }

    @Override
    public SwingValidator<TuttiUIContext> getValidator() {
        return null;
    }

    protected Action closeAction;

    protected Action openAction;

    public void closeEditor() {

        closeAction.actionPerformed(null);
    }

    public void openEditor(JComponent component) {

        if (component != null) {
            place(component);
        }
        openAction.actionPerformed(null);
    }

    public void init() {
        resetFields();
        ui.getAttachments().removeAll();
        AttachmentModelAware bean = ui.getBean();
        if (bean != null) {
            List<Attachment> list = bean.getAttachment();
            if (list != null) {
                for (Attachment attachment : list) {
                    addAttachment(attachment);
                }
            }
        }
    }

    public void place(JComponent component) {
        // Computes the location of bottom left corner of the cell
        Component comp = component;
        int x = 0;
        int y = component.getHeight();
        while (comp != null) {
            x += comp.getX();
            y += comp.getY();
            comp = comp.getParent();
        }

        ui.pack();
        // if the editor is too big on the right,
        // then align its right side to the right side of the cell
        if (x + ui.getWidth() > ui.getOwner().getX() + ui.getOwner().getWidth()) {
            x = x - ui.getWidth() + component.getWidth();
        }
        ui.setLocation(x, y);
    }

    public void addAttachment() {
        AttachmentModelAware bean = ui.getBean();
        Attachment attachment = Attachments.newAttachment();

        File file = ui.getFile().getSelectedFile();
        if (file != null) {
            String name = ui.getFileName().getText();
            if (StringUtils.isEmpty(name)) {
                name = file.getName();
            }
            attachment.setObjectType(bean.getObjectType());
            attachment.setObjectId(bean.getObjectId());
            attachment.setName(name);
            attachment.setComment(ui.getFileComment().getText());

            attachment = getPersistenceService().createAttachment(attachment, file);
            bean.addAttachment(attachment);

            resetFields();
            addAttachment(attachment);

            ui.pack();
        }
    }

    protected void addAttachment(final Attachment attachment) {
        AttachmentItem item = new AttachmentItem();
        item.setAttachment(attachment);
        item.setHandler(this);
        ui.getAttachments().add(item);
    }

    protected void resetFields() {
        ui.getFile().setSelectedFilePath(null);
        ui.getFileName().setText("");
        ui.getFileComment().setText("");
    }

    public void openAttachment(Attachment attachment) {
        File file = getPersistenceService().getAttachmentFile(attachment.getId());

        if (!file.exists()) {
            throw new ApplicationTechnicalException(t("tutti.attachmentEditor.fileNotFound", file.getAbsolutePath()));
        }

        Desktop desktop = TuttiUIUtil.getDesktopForBrowse();
        try {
            desktop.browse(file.toURI());

        } catch (IOException e) {
            getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
        }
    }

    public void saveAttachment(Attachment attachment) {

        SaveAttachmentAction action = getContext().getActionFactory().createLogicAction(
                this, SaveAttachmentAction.class);
        action.setAttachment(attachment);
        getContext().getActionEngine().runAction(action);
    }

    public void removeAttachment(AttachmentItem attachmentItem) {
        Attachment attachment = attachmentItem.getAttachment();
        boolean hackDialog = ui.isAlwaysOnTop();
        if (hackDialog) {
            ui.setAlwaysOnTop(false);
        }
        int answer = JOptionPane.showConfirmDialog(ui,
                                                   t("tutti.attachmentEditor.deleteAttachment.message", attachment.getName()),
                                                   t("tutti.attachmentEditor.deleteAttachment.title"),
                                                   JOptionPane.YES_NO_OPTION);
        if (hackDialog) {
            ui.setAlwaysOnTop(true);
        }

        if (answer == JOptionPane.YES_OPTION) {
            AttachmentModelAware bean = ui.getBean();

            getPersistenceService().deleteAttachment(attachment.getId());
            bean.removeAttachment(attachment);
            ui.getAttachments().remove(attachmentItem);

            ui.pack();
        }
    }
}
