package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosFrequencyUIHandler.java 1247 2013-09-28 12:15:25Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.6.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/frequency/BenthosFrequencyUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.service.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.UIManager;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyUIHandler extends AbstractTuttiTableUIHandler<BenthosFrequencyRowModel, BenthosFrequencyUIModel, BenthosFrequencyUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosFrequencyUIHandler.class);

    private BenthosFrequencyCellComponent.FrequencyCellEditor frequencyEditor;

//    private TuttiProtocol protocol;

    private Map<Integer, SpeciesProtocol> speciesProtocol;

    private Map<String, Caracteristic> lengthStepCaracteristics;

    protected Set<BenthosFrequencyRowModel> withWeightRows = Sets.newHashSet();

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public BenthosFrequencyUIHandler(TuttiUIContext context,
                                     BenthosFrequencyUI ui) {
        super(context, ui,
              BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
              BenthosFrequencyRowModel.PROPERTY_NUMBER,
              BenthosFrequencyRowModel.PROPERTY_WEIGHT);

        this.weightUnit = context.getConfig().getBenthosWeightUnit();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosFrequencyTableModel getTableModel() {
        return (BenthosFrequencyTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(BenthosFrequencyRowModel row) {
        return row.getLengthStepCaracteristic() != null &&
               row.getLengthStep() != null &&
               ((row.getNumber() == null && row.getWeight() == null) ||
                (row.getNumber() != null && row.getNumber() > 0 &&
                 (withWeightRows.isEmpty() || row.getWeight() != null && row.getWeight() > 0)));
    }

    @Override
    protected void onModelRowsChanged(List<BenthosFrequencyRowModel> rows) {
        super.onModelRowsChanged(rows);

        recomputeTotalNumberAndWeight();
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 BenthosFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        boolean recomputeAllRows;
        BenthosFrequencyUIModel model = getModel();

        if (row.getWeight() != null) {
            // check if no row had a weight, then if one of them now has a weight,
            // the other ones must have one too to be valid
            recomputeAllRows = withWeightRows.isEmpty();
            withWeightRows.add(row);

        } else {
            withWeightRows.remove(row);
            // check if no row has a weight, then if none of them now has a weight,
            // the other ones do not need to have a weight to be valid
            recomputeAllRows = withWeightRows.isEmpty();
        }


        if (recomputeAllRows) {
            for (BenthosFrequencyRowModel r : model.getRows()) {
                recomputeRowValidState(r);
            }
        }

        recomputeTotalNumberAndWeight();

        if (!recomputeAllRows) {
            recomputeRowValidState(row);
        }
        if (row.isValid() && row.getNumber() == null && row.getWeight() == null) {
            model.addEmptyRow(row);
        } else {
            model.removeEmptyRow(row);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<BenthosFrequencyRowModel> rowMonitor,
                                             BenthosFrequencyRowModel row) {
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          BenthosFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);
        ui.getValidator().doValidate();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<BenthosFrequencyUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        SampleCategoryModel sampleCategoryModel =
                getDataContext().getSampleCategoryModel();

        BenthosFrequencyUIModel model =
                new BenthosFrequencyUIModel(weightUnit, sampleCategoryModel);

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Caracteristic> lengthStepCaracterics =
                Lists.newArrayList(getDataContext().getLengthStepCaracteristics());

        lengthStepCaracteristics = TuttiEntities.splitById(lengthStepCaracterics);

        BenthosFrequencyUIModel model = getModel();

        if (context.isProtocolFilled()) {

//            // get loaded protocol
//            protocol = getDataContext().getProtocol();
//            Preconditions.checkNotNull(protocol,
//                                       "Could not find protocol in ui context");

            // FIXME 20130128 kmorin: the species have no technical id
            speciesProtocol =
                    getContext().getPersistenceService().toBenthosProtocolMap();

//            speciesProtocol = Maps.newHashMap();
//            for (SpeciesProtocol sp : protocol.getBenthos()) {
//                speciesProtocol.put(sp.getSpeciesReferenceTaxonId(), sp);
//            }

        } else {
            speciesProtocol = Maps.newHashMap();
        }

        Caracteristic modelCaracteristic = model.getLengthStepCaracteristic();
        initBeanFilterableComboBox(ui.getLengthStepCaracteristicComboBox(),
                                   lengthStepCaracterics,
                                   modelCaracteristic);

        // get precision from the pmfm
        Float precision = null;
        if (modelCaracteristic != null) {
            precision = model.getLengthStepCaracteristic().getPrecision();
        }
        if (precision == null) {
            precision = 1f;
        }
        model.setStep(precision);

        model.setMinStep(null);
        model.setMaxStep(null);

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) ui.getRafaleStepField().getModel();

                    applyRafaleStep(step);

                    //select text
                    JTextField field = (JTextField) e.getSource();
                    field.selectAll();
                }
            }
        });

        // when lengthStepCaracteristic changed, let's updates all row with the new value
        model.addPropertyChangeListener(BenthosFrequencyUIModel.PROPERTY_LENGHT_STEP_CARACTERISTIC, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Caracteristic newValue = (Caracteristic) evt.getNewValue();
                for (BenthosFrequencyRowModel rowModel : getModel().getRows()) {
                    rowModel.setLengthStepCaracteristic(newValue);
                    // get precision from the pmfm
                    Float precision = null;
                    if (newValue != null) {
                        precision = newValue.getPrecision();
                    }
                    if (precision == null) {
                        precision = .5f;
                    }
                    getModel().setStep(precision);
                    recomputeRowValidState(rowModel);
                }
                ui.getValidator().doValidate();
            }
        });

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    BenthosFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  BenthosFrequencyTableModel.WEIGHT,
                                  weightUnit);
        }

        // create table model
        BenthosFrequencyTableModel tableModel =
                new BenthosFrequencyTableModel(
                        weightUnit,
                        columnModel,
                        model);

        JXTable table = getTable();

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        installTableKeyListener(columnModel, table);

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    protected void addHighlighters(JXTable table) {
        super.addHighlighters(table);

        // highlight only the lengthstep column if the row is selected
        Highlighter selectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.IS_SELECTED,
                        new HighlightPredicate.IdentifierHighlightPredicate(BenthosFrequencyTableModel.LENGTH_STEP)),
                UIManager.getColor("Table[Enabled+Selected].textBackground"));

        table.addHighlighter(selectedHighlighter);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getLengthStepCaracteristicComboBox();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        frequencyEditor = null;

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        getModel().setSimpleCount(null);

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isDebugEnabled()) {
            log.debug("Cancel UI " + ui);
        }

        // close dialog
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addLengthStepCaracteristic() {

        // compute list of possible caracteristics (all but the one in the select box)
        List<Caracteristic> allNumericCaracteristic = persistenceService.getAllNumericCaracteristic();
        List<Caracteristic> toSelect = Lists.newArrayList(allNumericCaracteristic);
        List<Caracteristic> knownCaracteristics = getUI().getLengthStepCaracteristicComboBox().getData();
        toSelect.removeAll(knownCaracteristics);

        // open a dialog to select it

        BeanFilterableComboBox<Caracteristic> editor =
                new BeanFilterableComboBox<Caracteristic>();
        editor.setBeanType(Caracteristic.class);
        editor.setShowReset(true);

        initBeanFilterableComboBox(editor, toSelect, null);

        int response = JOptionPane.showConfirmDialog(
                getTopestUI(),
                editor,
                _("tutti.editBenthosFrequencies.title.addLengthStepCaracteristic"),
                JOptionPane.OK_CANCEL_OPTION);

        Caracteristic selectedItem;
        if (response == JOptionPane.OK_OPTION) {
            selectedItem = (Caracteristic) editor.getSelectedItem();

//            // FIXME ? Should we add it to the combo box universe?
//            ui.getLengthStepCaracteristicComboBox().getData().add(selectedItem);
        } else {
            // user cancel selection
            selectedItem = null;
        }
        // set to model
        getModel().setLengthStepCaracteristic(selectedItem);
    }

    public void generateLengthSteps() {

        BenthosFrequencyUIModel model = getModel();
        BenthosFrequencyTableModel tableModel = getTableModel();

        Map<Float, BenthosFrequencyRowModel> rowsByStep =
                getTableModel().getRowCache();

        Float minStep = model.getLengthStep(model.getMinStep());
        Float maxStep = model.getLengthStep(model.getMaxStep());
        Caracteristic lengthStepCaracteristic = model.getLengthStepCaracteristic();

        for (float i = minStep, step = model.getStep(); i <= maxStep;
             i = TuttiUIUtil.getRoundedLengthStep(i + step, true)) {

            if (!rowsByStep.containsKey(i)) {

                // add it
                BenthosFrequencyRowModel newRow = tableModel.createNewRow();
                newRow.setLengthStep(i);
                newRow.setLengthStepCaracteristic(lengthStepCaracteristic);
                rowsByStep.put(i, newRow);
            }
        }

        List<BenthosFrequencyRowModel> rows =
                Lists.newArrayList(rowsByStep.values());

        model.setRows(rows);

        // select first cell in table (see http://forge.codelutin.com/issues/2496)
        AbstractSelectTableAction.doSelectCell(getUI().getTable(), 0, 1);
    }

    public void applyRafaleStep(Float step) {

        if (log.isDebugEnabled()) {
            log.debug("Will apply rafale step: " + step);
        }
        BenthosFrequencyUIModel model = getModel();
        BenthosFrequencyTableModel tableModel = getTableModel();

        Map<Float, BenthosFrequencyRowModel> rowsByStep = tableModel.getRowCache();

        float aroundLengthStep = model.getLengthStep(step);

        BenthosFrequencyRowModel row = rowsByStep.get(aroundLengthStep);

        int rowIndex;

        if (row != null) {

            // increments current row
            Integer number = row.getNumber();
            row.setNumber((number == null ? 0 : number) + 1);
            rowIndex = tableModel.updateRow(row);

        } else {

            // create a new row

            row = tableModel.createNewRow();
            row.setLengthStep(aroundLengthStep);
            row.setNumber(1);
            row.setValid(true);

            // get new index
            List<Float> steps = Lists.newArrayList(rowsByStep.keySet());
            steps.add(aroundLengthStep);

            Collections.sort(steps);

            rowIndex = steps.indexOf(aroundLengthStep);

            tableModel.addNewRow(rowIndex, row);
        }
        Integer totalNumber = model.getTotalNumber();
        if (totalNumber == null) {
            totalNumber = 0;
        }
        model.setTotalNumber(totalNumber + 1);

        getTable().scrollRowToVisible(rowIndex);
    }

    public void editBatch(BenthosFrequencyCellComponent.FrequencyCellEditor editor) {
        withWeightRows.clear();

        BenthosBatchRowModel speciesBatch = editor.getEditRow();

        BenthosFrequencyUIModel model = getModel();
        model.setNextEditableRowIndex(editor.getNextEditableRowIndex());
        model.setTotalNumber(null);
        model.setTotalWeight(null);
        model.setSimpleCount(null);

        frequencyEditor = editor;

        Caracteristic lengthStepCaracteristic = null;
        Float lengthStep;

        List<BenthosFrequencyRowModel> editFrequency = Lists.newArrayList();

        if (speciesBatch != null) {

            List<BenthosFrequencyRowModel> frequency =
                    speciesBatch.getFrequency();

            // try to load existing frequency

            if (CollectionUtils.isNotEmpty(frequency)) {

                Integer totalNumber = 0;
                Float totalWeight = 0f;

                BenthosFrequencyTableModel tableModel = getTableModel();

                for (BenthosFrequencyRowModel rowModel : frequency) {

                    BenthosFrequencyRowModel newRow = tableModel.createNewRow();
                    newRow.setLengthStepCaracteristic(rowModel.getLengthStepCaracteristic());
                    newRow.setLengthStep(rowModel.getLengthStep());
                    newRow.setNumber(rowModel.getNumber());
                    newRow.setWeight(rowModel.getWeight());
                    editFrequency.add(newRow);

                    if (newRow.getWeight() != null) {
                        withWeightRows.add(newRow);
                        totalWeight += newRow.getWeight();
                    } else {
                        totalWeight = null;
                    }
                    totalNumber += newRow.getNumber();
                }
                model.setTotalNumber(totalNumber);
                model.setTotalWeight(totalWeight);

                // use first frequency row length step caracteristics

                BenthosFrequencyRowModel rowModel = frequency.get(0);
                lengthStepCaracteristic = rowModel.getLengthStepCaracteristic();
                lengthStep = rowModel.getLengthStep();

                if (log.isInfoEnabled()) {
                    log.info("Use existing lengthStep " +
                             "caracteristic / step " +
                             decorate(lengthStepCaracteristic) + " / " +
                             lengthStep);
                }
            }

            BenthosBatchRowModel previousSiblingRow =
                    frequencyEditor.getPreviousSiblingRow();

            if (lengthStepCaracteristic == null && previousSiblingRow != null) {

                // try to get it from his previous brother row
                List<BenthosFrequencyRowModel> previousFrequency =
                        previousSiblingRow.getFrequency();

                if (CollectionUtils.isNotEmpty(previousFrequency)) {

                    // use the first frequency length step caracteristic / step
                    BenthosFrequencyRowModel rowModel = previousFrequency.get(0);
                    lengthStepCaracteristic =
                            rowModel.getLengthStepCaracteristic();
                    lengthStep = rowModel.getLengthStep();
                    if (log.isInfoEnabled()) {
                        log.info("Use previous sibling existing lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }

//            if (lengthStepCaracteristic == null && protocol != null) {
            if (lengthStepCaracteristic == null) {

                Species species = speciesBatch.getSpecies();

                SpeciesProtocol sProtocol =
                        speciesProtocol.get(species.getReferenceTaxonId());

                if (sProtocol != null) {


                    String lengthStepPmfmId = sProtocol.getLengthStepPmfmId();

                    lengthStepCaracteristic =
                            this.lengthStepCaracteristics.get(lengthStepPmfmId);
                    lengthStep = sProtocol.getLengthStep();

                    if (log.isInfoEnabled()) {
                        log.info("Use existing from protocol lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will edit batch row: " + speciesBatch + " with " +
                      editFrequency.size() + " frequency");
        }

        BenthosFrequencyUIModel.ConfigurationMode mode = BenthosFrequencyUIModel.ConfigurationMode.SIMPLE;
//        if (lengthStepCaracteristic == null && protocol != null) {
        if (lengthStepCaracteristic == null) {
            Integer taxonId = speciesBatch.getSpecies().getReferenceTaxonId();
            SpeciesProtocol speciesProtocol = this.speciesProtocol.get(taxonId);
            if (speciesProtocol == null || speciesProtocol.getLengthStepPmfmId() == null) {
                mode = BenthosFrequencyUIModel.ConfigurationMode.SIMPLE_COUNTING;
            }
        }
        Integer number = speciesBatch.getNumber();
        if (number != null && editFrequency.isEmpty()) {
            mode = BenthosFrequencyUIModel.ConfigurationMode.SIMPLE_COUNTING;
            model.setSimpleCount(number);
        }
        model.setConfigurationMode(mode);

        // connect model to validator
        ui.getValidator().setBean(model);

        // always sort row by their length
        // see http://forge.codelutin.com/issues/2482
        Collections.sort(editFrequency);

        model.setRows(editFrequency);
        model.setLengthStepCaracteristic(lengthStepCaracteristic);

        // keep batch (will be used to push back editing entry)
        model.setBatch(speciesBatch);
    }

    public void reset() {

        // remove all frequencies
        getModel().setRows(Lists.<BenthosFrequencyRowModel>newArrayList());
    }

    public void saveAndClose() {

        if (log.isDebugEnabled()) {
            log.debug("Save And Close UI " + ui);
        }

        // check that we do not have doublon in length
        // see http://forge.codelutin.com/issues/2499
        Set<Float> lengths = Sets.newHashSet();

        Float doublon = null;
        int index = 0;
        for (BenthosFrequencyRowModel row : getModel().getRows()) {
            Float lengthStep = row.getLengthStep();
            if (!lengths.add(lengthStep)) {

                // already exist
                doublon = lengthStep;
                break;
            }
            index++;
        }
        if (doublon != null) {

            // can't save mensurations (found doublon)
            String message =
                    _("tutti.editBenthosFrequencies.error.length.doublon",
                      doublon, index + 1);
            TuttiUIContext.getErrorHelper().showErrorDialog(
                    message);

            // focus to first error row
            AbstractSelectTableAction.doSelectCell(getTable(), index, 0);
            return;
        }

        frequencyEditor.save(getModel(), true);

        closeUI(ui);
    }

    public void saveAndContinue() {

        if (log.isDebugEnabled()) {
            log.debug("Save And Continue UI " + ui);
        }

        // check that we do not have doublon in length
        // see http://forge.codelutin.com/issues/2499
        Set<Float> lengths = Sets.newHashSet();

        Float doublon = null;
        int index = 0;
        for (BenthosFrequencyRowModel row : getModel().getRows()) {
            Float lengthStep = row.getLengthStep();
            if (!lengths.add(lengthStep)) {

                // already exist
                doublon = lengthStep;
                break;
            }
            index++;
        }
        if (doublon != null) {

            // can't save mensurations (found doublon)
            String message =
                    _("tutti.editBenthosFrequencies.error.length.doublon",
                      doublon, index + 1);
            TuttiUIContext.getErrorHelper().showErrorDialog(
                    message);

            // focus to first error row
            AbstractSelectTableAction.doSelectCell(getTable(), index, 0);
            return;
        }

        frequencyEditor.save(getModel(), false);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void recomputeTotalNumberAndWeight() {
        BenthosFrequencyUIModel model = getModel();
        List<BenthosFrequencyRowModel> rows = model.getRows();
        Integer totalNumber = 0;
        Float totalWeight = 0f;
        for (BenthosFrequencyRowModel r : rows) {
            if (r.isValid()) {
                Integer number = r.getNumber();
                if (number != null) {
                    totalNumber += number;
                }
                Float weight = r.getWeight();
                if (weight != null) {
                    totalWeight += weight;
                }
            }
        }
        model.setTotalNumber(totalNumber);
        model.setTotalWeight(totalWeight);
    }
}
