package fr.ifremer.tutti.ui.swing.content.db;

/*
 * #%L
 * Tutti :: UI
 * $Id: ExportAndCleanDbAction.java 673 2013-03-24 13:03:42Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.4.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/db/ExportAndCleanDbAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiIOUtil;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.config.TuttiPersistenceConfig;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.content.AbstractMainUITuttiAction;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;

import static org.nuiton.i18n.I18n._;

/**
 * To export a db attached to Tutti.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class ExportAndCleanDbAction extends AbstractMainUITuttiAction {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ExportAndCleanDbAction.class);

    protected File file;

    public ExportAndCleanDbAction(MainUIHandler handler) {
        super(handler, true);
    }

    @Override
    protected boolean prepareAction() throws Exception {
        file = null;
        boolean doAction = super.prepareAction();

        if (doAction) {

            // ask user file where to export db

            // choose file to import
            file = TuttiUIUtil.chooseFile(
                    getContext().getMainUI(),
                    _("tutti.dbManager.title.choose.dbExportFile"),
                    _("tutti.dbManager.action.chooseDbExportFile"),
                    "^.*\\.zip", _("tutti.common.file.zip")
            );
            if (file == null) {

                // user cancel
                doAction = false;
            } else {

                // add extension if missing
                file = TuttiIOUtil.addExtensionIfMissing(file, ".zip");

                // ask user to confirm overwrite.
                doAction = getHandler().askOverwriteFile(file);
            }
        }
        return doAction;
    }

    @Override
    protected void doAction() {
        Preconditions.checkNotNull(file);
        if (log.isInfoEnabled()) {
            log.info("Will export db to " + file);
        }

        ProgressionModel progressionModel = new ProgressionModel();
        setProgressionModel(progressionModel);
        progressionModel.setTotal(3);

        // close db

        progressionModel.setMessage("Fermeture de la base courante");


        TuttiActionHelper.runInternalAction(getHandler(), CloseDbAction.class);

        PersistenceService persistenceService =
                getContext().getPersistenceService();

        // export db

        progressionModel.increments(1);
        progressionModel.setMessage("Création de l'archive " + file);

        persistenceService.exportDb(file);

        // clean files

        progressionModel.increments(1);
        progressionModel.setMessage("Suppression des fichiers dans Tutti");

        TuttiPersistenceConfig persistenceConfig =
                getConfig().getServiceConfig().getPersistenceConfig();

        File dbDirectory = persistenceConfig.getDbDirectory();
        TuttiIOUtil.cleanDirectory(dbDirectory, "Could not delete directory " + dbDirectory);

        File attachmentDirectory = persistenceConfig.getDbAttachmentDirectory();
        TuttiIOUtil.cleanDirectory(attachmentDirectory, "Could not delete directory " + attachmentDirectory);

        File protocolDirectory = persistenceConfig.getProtocolDirectory();
        TuttiIOUtil.cleanDirectory(protocolDirectory, "Could not delete directory " + protocolDirectory);

        getModel().setDbExist(false);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        sendMessage(_("tutti.flash.info.db.exported.and.clean", file));

        // make sure title is reloaded
        getUI().getHandler().changeTitle();
    }

}