package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIAction.java 572 2013-03-11 10:01:36Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/TuttiUIAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.SwingUtilities;
import java.awt.event.ActionEvent;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Abstract tutti ui action which launch a {@link AbstractTuttiAction}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.2
 */
public class TuttiUIAction<A extends AbstractTuttiAction> extends AbstractAction {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIAction.class);

    public static final ExecutorService waitingThread =
            Executors.newSingleThreadExecutor();

    private static final String LOGIC_ACTION = "logicAction";

    public TuttiUIAction(AbstractButton button, A action) {

        putValue(LOGIC_ACTION, action);

        // fill the ui action from the button
        setActionKey(action.getClass().getName());
        if (button != null) {
            setActionIcon(button.getIcon());
            setActionName(button.getText());
            setActionDescription(button.getToolTipText());
            setActionMnemonic(button.getMnemonic());
        }
    }

    @Override
    public final void actionPerformed(final ActionEvent event) {

        if (log.isInfoEnabled()) {
            log.info("Task [" + this + "] starting");
        }

        // prepare action
        boolean doAction;

        A action = getLogicAction();

        try {
            doAction = action.prepareAction();
        } catch (Exception e) {
            action.releaseAction();
            throw TuttiActionException.propagateError(action, e);
        }

        if (doAction) {

            final TuttiActionSwingWorker<A> worker =
                    new TuttiActionSwingWorker<A>(action);

            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {

                    // make ui busy
                    worker.updateBusyState(true);

                }
            });

            if (log.isInfoEnabled()) {
                log.info("Before execute of action " + action);
            }

            // perform and release action
            worker.execute();

            // wait until action is done
            waitingThread.execute(
                    new Runnable() {
                        @Override
                        public void run() {

                            A action = getLogicAction();

                            try {
                                worker.get();
                            } catch (ExecutionException e) {
                                // don't care .
                            } catch (CancellationException e) {
                                // dont care ?

                            } catch (InterruptedException e) {
                                // don't care ?
                            }
                            if (log.isInfoEnabled()) {
                                log.info("After execute of action " + action + " (worker done? " + worker.isDone() + ")");
                            }

                            if (worker.isFailed()) {

                                throw TuttiActionException.propagateError(action, worker.getError());
                            }
                        }
                    }
            );

        } else {

            // release action
            action.releaseAction();
        }
    }

    public void setActionIcon(Icon actionIcon) {
        putValue(SMALL_ICON, actionIcon);
        putValue(LARGE_ICON_KEY, actionIcon);
    }

    public void setActionKey(String actionKey) {
        putValue(ACTION_COMMAND_KEY, actionKey);
    }

    public void setActionName(String actionName) {
        putValue(NAME, actionName);
    }

    public void setActionDescription(String actionDescription) {
        putValue(SHORT_DESCRIPTION, actionDescription);
        getLogicAction().setActionDescription(actionDescription);
    }

    public void setActionMnemonic(int key) {
        putValue(MNEMONIC_KEY, key);
    }

    public A getLogicAction() {
        return (A) getValue("logicAction");
    }

}
