package fr.ifremer.tutti.ui.swing.util.table;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiTableUIHandler.java 349 2013-02-06 10:54:11Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/table/AbstractTuttiTableUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchTableModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.JAXXWidgetUtil;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.swing.editor.cell.NumberCellEditor;
import jaxx.runtime.swing.renderer.DecoratorTableCellRenderer;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JComboBox;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @param <R> type of a row
 * @param <M> type of the ui model
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public abstract class AbstractTuttiTableUIHandler<R extends AbstractTuttiBeanUIModel, M extends AbstractTuttiTableUIModel<?, R, M>, UI extends TuttiUI<M, ?>> extends AbstractTuttiUIHandler<M, UI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiTableUIHandler.class);

    /**
     * @return the table model handled byt the main table.
     * @since 0.2
     */
    public abstract AbstractTuttiTableModel<R> getTableModel();

    /**
     * @return the main table of the ui.
     * @since 0.2
     */
    public abstract JXTable getTable();

    /**
     * Validates the given row.
     *
     * @param row row to validate
     * @return {@code true} if row is valid, {@code false} otherwise.
     * @since 0.2
     */
    protected abstract boolean isRowValid(R row);

    /**
     * Invoke each time the {@link AbstractTuttiBeanUIModel#modify} state on
     * the current selected row changed.
     *
     * @param rowIndex     row index of the modified row
     * @param row          modified row
     * @param propertyName name of the modified property of the row
     * @param oldValue     old value of the modified property
     * @param newValue     new value of the modified property
     * @since 0.3
     */
    protected void onRowModified(int rowIndex,
                                 R row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        getModel().setModify(true);
    }

    /**
     * Given the row monitor and his monitored row, try to save it if required.
     * <p/>
     * Coming in this method, we are sure that row is not null.
     *
     * @param rowMonitor the row monitor (see {@link #rowMonitor})
     * @param row        the row to save if necessary
     * @since 0.3
     */
    protected abstract void saveSelectedRowIfRequired(TuttiBeanMonitor<R> rowMonitor, R row);

    /**
     * Monitor the selected row (save it only if something has changed).
     *
     * @since 0.2
     */
    private final TuttiBeanMonitor<R> rowMonitor;

    /**
     * Persistence service.
     *
     * @since 0.2
     */
    protected final PersistenceService persistenceService;

    protected AbstractTuttiTableUIHandler(TuttiUIContext context,
                                          UI ui,
                                          String... properties) {
        super(context, ui);

        this.persistenceService = context.getService(PersistenceService.class);

        rowMonitor = new TuttiBeanMonitor<R>(properties);

        // listen when bean is changed
        rowMonitor.addPropertyChangeListener(TuttiBeanMonitor.PROPERTY_BEAN, new PropertyChangeListener() {

            final Set<String> propertiesToSkip =
                    Sets.newHashSet(getRowPropertiesToIgnore());

            final PropertyChangeListener l = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    String propertyName = evt.getPropertyName();

                    R row = (R) evt.getSource();

                    Object oldValue = evt.getOldValue();
                    Object newValue = evt.getNewValue();

                    int rowIndex = getTableModel().getRowIndex(row);

                    if (AbstractTuttiBeanUIModel.PROPERTY_VALID.equals(propertyName)) {
                        onRowValidStateChanged(rowIndex, row,
                                               (Boolean) oldValue,
                                               (Boolean) newValue);
                    } else if (AbstractTuttiBeanUIModel.PROPERTY_MODIFY.equals(propertyName)) {
                        onRowModifyStateChanged(rowIndex, row,
                                                (Boolean) oldValue,
                                                (Boolean) newValue);
                    } else if (!propertiesToSkip.contains(propertyName)) {

                        if (log.isInfoEnabled()) {
                            log.info("row [" + rowIndex + "] property " +
                                     propertyName + " changed from " + oldValue +
                                     " to " + newValue);
                        }
                        onRowModified(rowIndex, row,
                                      propertyName,
                                      oldValue,
                                      newValue);
                    }
                }
            };

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                R oldValue = (R) evt.getOldValue();
                R newValue = (R) evt.getNewValue();
                if (log.isInfoEnabled()) {
                    log.info("Monitor row changed from " +
                             oldValue + " to " + newValue);
                }
                if (oldValue != null) {
                    oldValue.removePropertyChangeListener(l);
                }
                if (newValue != null) {
                    newValue.addPropertyChangeListener(l);
                }
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- Internal methods (row methods)                                     --//
    //------------------------------------------------------------------------//

    protected String[] getRowPropertiesToIgnore() {
        return ArrayUtils.EMPTY_STRING_ARRAY;
    }

    protected void onModelRowsChanged(List<R> rows) {
        if (log.isInfoEnabled()) {
            log.info("Will set " + (rows == null ? 0 : rows.size()) +
                     " rows on model.");
        }
        if (CollectionUtils.isNotEmpty(rows)) {

            // compute valid state for each row
            for (R row : rows) {
                recomputeRowValidState(row);
            }
        }
        getTableModel().setRows(rows);
    }

    protected void onRowModifyStateChanged(int rowIndex,
                                           R row,
                                           Boolean oldValue,
                                           Boolean newValue) {
        if (log.isInfoEnabled()) {
            log.info("row [" + rowIndex + "] modify state changed from " +
                     oldValue + " to " + newValue);
        }
    }

    protected void onRowValidStateChanged(int rowIndex,
                                          R row,
                                          Boolean oldValue,
                                          Boolean newValue) {

        if (log.isInfoEnabled()) {
            log.info("row [" + rowIndex + "] valid state changed from " +
                     oldValue + " to " + newValue);
        }

        if (rowIndex > -1) {
            getTableModel().fireTableRowsUpdated(rowIndex, rowIndex);
        }
    }

    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             R oldRow,
                                             int newRowIndex,
                                             R newRow) {
        if (log.isInfoEnabled()) {
            log.info("Selected row changed from [" + oldRowIndex + "] to [" +
                     newRowIndex + "]");
        }
    }

    //------------------------------------------------------------------------//
    //-- Internal methods (init methods)                                    --//
    //------------------------------------------------------------------------//

    protected void initTable(JXTable table) {

        // by default do not authorize to change column orders
        table.getTableHeader().setReorderingAllowed(false);

        // paint in a special color read only cells
        Highlighter readOnlyHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.READ_ONLY, getConfig().getColorRowReadOnly());
        table.addHighlighter(readOnlyHighlighter);

        Color cellWithValueColor = getConfig().getColorCellWithValue();
        Highlighter commentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(SpeciesBatchTableModel.COMMENT),
                        new HighlightPredicate.NotHighlightPredicate(new HighlightPredicate.EqualsHighlightPredicate())
                ), cellWithValueColor);
        table.addHighlighter(commentHighlighter);

        Highlighter attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(SpeciesBatchTableModel.ATTACHMENTS),
                        new HighlightPredicate.NotHighlightPredicate(new HighlightPredicate.EqualsHighlightPredicate())
                ), cellWithValueColor);
        table.addHighlighter(attachmentHighlighter);

        // paint in a special color inValid rows
        Highlighter validHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(HighlightPredicate.EDITABLE, new HighlightPredicate() {
                    @Override
                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {

                        boolean result = false;
                        if (adapter.isEditable()) {
                            int rowIndex = adapter.convertRowIndexToModel(adapter.row);
                            R row = getTableModel().getEntry(rowIndex);
                            result = !row.isValid();
                        }
                        return result;
                    }
                }), getConfig().getColorRowInvalid());
        table.addHighlighter(validHighlighter);

        // when model data change let's propagate it table model
        getModel().addPropertyChangeListener(AbstractTuttiTableUIModel.PROPERTY_ROWS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                onModelRowsChanged((List<R>) evt.getNewValue());
            }
        });

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(getTable());

        // always force to uninstall listener
        uninstallTableSaveOnRowChangedSelectionListener();

        // save when row chaged and was modified
        installTableSaveOnRowChangedSelectionListener();
    }

    protected void addColumnToModel(TableColumnModel model,
                                    TableCellEditor editor,
                                    TableCellRenderer renderer,
                                    ColumnIdentifier<R> identifier) {

        TableColumnExt col = new TableColumnExt(model.getColumnCount());
        col.setCellEditor(editor);
        col.setCellRenderer(renderer);
        col.setHeaderValue(_(identifier.getHeaderI18nKey()));
        col.setToolTipText(_(identifier.getHeaderTipI18nKey()));

        col.setIdentifier(identifier);
        model.addColumn(col);
    }

    protected void addColumnToModel(TableColumnModel model,
                                    ColumnIdentifier<R> identifier) {

        addColumnToModel(model, null, null, identifier);
    }

    protected void addFloatColumnToModel(TableColumnModel model,
                                         ColumnIdentifier<R> identifier,
                                         String numberPattern) {

        NumberCellEditor<Float> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier);

    }

    protected void addIntegerColumnToModel(TableColumnModel model,
                                           ColumnIdentifier<R> identifier,
                                           String numberPattern) {

        NumberCellEditor<Integer> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Integer.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier);
    }

    protected void addBooleanColumnToModel(TableColumnModel model,
                                           ColumnIdentifier<R> identifier,
                                           JTable table) {

        addColumnToModel(model,
                         table.getDefaultEditor(Boolean.class),
                         table.getDefaultRenderer(Boolean.class),
                         identifier);
    }

    protected <B> void addComboDataColumnToModel(TableColumnModel model,
                                                 ColumnIdentifier<R> identifier,
                                                 Decorator<B> decorator,
                                                 List<B> data) {
        JComboBox comboBox = new JComboBox();
        comboBox.setRenderer(newListCellRender(decorator));

        List<B> dataToList = Lists.newArrayList(data);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);

        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         identifier);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type) {

        return newTableCellRender(type, null);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);

        TableCellRenderer result = newTableCellRender(decorator);
        return result;
    }

    protected <O> TableCellRenderer newTableCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        DecoratorTableCellRenderer result = new DecoratorTableCellRenderer(decorator);
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods (listener methods)                                --//
    //------------------------------------------------------------------------//

    private ListSelectionListener tableSelectionListener;

    private KeyAdapter keyAdapter;

    protected void installTableSaveOnRowChangedSelectionListener() {

        Preconditions.checkState(
                tableSelectionListener == null,
                "There is already a tableSelectionListener registred, " +
                "remove it before invoking this method.");

        // create new listener
        // save when row chaged and was modified

        tableSelectionListener = new ListSelectionListener() {
            /**
             * Current selected row index.
             *
             * @since 0.3
             */
            protected int selectedRowIndex;

            @Override
            public void valueChanged(ListSelectionEvent e) {

                if (!e.getValueIsAdjusting()) {
                    ListSelectionModel source = (ListSelectionModel) e.getSource();

                    int oldRowIndex = selectedRowIndex;
                    R oldRow = rowMonitor.getBean();

                    int newRowIndex = source.getLeadSelectionIndex();
                    R newRow;

                    if (source.isSelectionEmpty()) {

                        newRow = null;
                    } else {
                        newRow = getTableModel().getEntry(newRowIndex);
                    }

                    // save selected entry if required
                    saveSelectedRowIfNeeded();

                    if (log.isDebugEnabled()) {
                        log.debug("Will monitor entry: " + newRow);
                    }
                    rowMonitor.setBean(newRow);

                    selectedRowIndex = newRowIndex;

                    onAfterSelectedRowChanged(oldRowIndex,
                                              oldRow,
                                              selectedRowIndex,
                                              rowMonitor.getBean());
                }
            }
        };

        if (log.isInfoEnabled()) {
            log.info("Intall " + tableSelectionListener + " on tableModel " + getTableModel());
        }

        getTable().getSelectionModel().addListSelectionListener(tableSelectionListener);
    }

    protected void uninstallTableSaveOnRowChangedSelectionListener() {

        if (tableSelectionListener != null) {

            if (log.isInfoEnabled()) {
                log.info("Desintall " + tableSelectionListener);
            }

            // there was a previous selection listener, remove it
            getTable().getSelectionModel().removeListSelectionListener(tableSelectionListener);
            tableSelectionListener = null;
        }
    }

    protected void installTableKeyListener(TableColumnModel columnModel,
                                           final JTable table) {

        Preconditions.checkState(
                keyAdapter == null,
                "There is already a tableSelectionListener registred, " +
                "remove it before invoking this method.");

        final AbstractTuttiTableModel<R> model = getTableModel();
        final MoveToNextEditableCellAction nextCellAction =
                MoveToNextEditableCellAction.newAction(model, table);
        final MoveToPreviousEditableCellAction previousCellAction =
                MoveToPreviousEditableCellAction.newAction(model, table);

        final MoveToNextEditableRowAction nextRowAction =
                MoveToNextEditableRowAction.newAction(model, table);
        final MoveToPreviousEditableRowAction previousRowAction =
                MoveToPreviousEditableRowAction.newAction(model, table);

        keyAdapter = new KeyAdapter() {

            @Override
            public void keyPressed(KeyEvent e) {
                TableCellEditor editor = table.getCellEditor();

                int keyCode = e.getKeyCode();
                if (keyCode == KeyEvent.VK_LEFT ||
                    (keyCode == KeyEvent.VK_TAB && e.isShiftDown())) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    previousCellAction.actionPerformed(null);

                } else if (//e.getKeyCode() == KeyEvent.VK_ENTER ||
                        keyCode == KeyEvent.VK_RIGHT ||
                        keyCode == KeyEvent.VK_TAB) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    nextCellAction.actionPerformed(null);

                } else if (keyCode == KeyEvent.VK_UP) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    previousRowAction.actionPerformed(null);

                } else if (keyCode == KeyEvent.VK_DOWN) {
                    e.consume();
                    if (editor != null) {
                        editor.stopCellEditing();
                    }
                    nextRowAction.actionPerformed(null);
                }
            }
        };

        if (log.isInfoEnabled()) {
            log.info("Intall " + keyAdapter);
        }

        table.addKeyListener(keyAdapter);

        Enumeration<TableColumn> columns = columnModel.getColumns();
        while (columns.hasMoreElements()) {
            TableColumn tableColumn = columns.nextElement();
            TableCellEditor cellEditor = tableColumn.getCellEditor();
            if (cellEditor instanceof NumberCellEditor) {
                NumberCellEditor editor = (NumberCellEditor) cellEditor;
                editor.getNumberEditor().getTextField().addKeyListener(keyAdapter);
            }
        }
    }

    protected void uninstallTableKeyListener() {

        if (keyAdapter != null) {

            if (log.isInfoEnabled()) {
                log.info("Desintall " + keyAdapter);
            }

            getTable().removeKeyListener(keyAdapter);

            TableColumnModel columnModel = getTable().getColumnModel();
            Enumeration<TableColumn> columns = columnModel.getColumns();
            while (columns.hasMoreElements()) {
                TableColumn tableColumn = columns.nextElement();
                TableCellEditor cellEditor = tableColumn.getCellEditor();
                if (cellEditor instanceof NumberCellEditor) {
                    NumberCellEditor editor = (NumberCellEditor) cellEditor;
                    editor.getNumberEditor().getTextField().removeKeyListener(keyAdapter);
                }
            }
            keyAdapter = null;
        }
    }

    protected final void saveSelectedRowIfNeeded() {

        R row = rowMonitor.getBean();

        if (row != null) {

            saveSelectedRowIfRequired(rowMonitor, row);
        }
    }

    protected final void recomputeRowValidState(R row) {

        // recompute row valid state
        boolean valid = isRowValid(row);

        // apply it to row
        row.setValid(valid);

        if (valid) {
            getModel().removeRowInError(row);
        } else {
            getModel().addRowInError(row);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void autoSelectRowInTable(MouseEvent e, JPopupMenu popup) {

        boolean rightClick = SwingUtilities.isRightMouseButton(e);

        if (rightClick || SwingUtilities.isLeftMouseButton(e)) {

            // get the coordinates of the mouse click
            Point p = e.getPoint();

            JXTable source = (JXTable) e.getSource();

            // get the row index at this point
            int rowIndex = source.rowAtPoint(p);

            if (log.isDebugEnabled()) {
                log.debug("At point [" + p + "] found Row " + rowIndex);
            }

            boolean canContinue = true;

            if (source.isEditing()) {

                // stop editing
                boolean stopEdit = source.getCellEditor().stopCellEditing();
                if (!stopEdit) {
                    if (log.isWarnEnabled()) {
                        log.warn("Could not stop edit cell...");
                    }
                    canContinue = false;
                }
            }

            if (canContinue) {

                // select row (could empty selection)
                if (rowIndex == -1) {
                    source.clearSelection();
                } else {
                    source.setRowSelectionInterval(rowIndex, rowIndex);
                }

                if (rightClick) {

                    // on right click show popup
                    popup.show(source, e.getX(), e.getY());
                }
            }
        }
    }
}
