/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.intarsys.cwt.font.FontQuery;
import de.intarsys.cwt.font.FontStyle;
import de.intarsys.cwt.font.FontTools;
import de.intarsys.cwt.font.IFont;
import de.intarsys.cwt.font.IFontProgram;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.font.PDFontAny;
import de.intarsys.pdf.font.PDFontDescriptor;
import de.intarsys.pdf.font.PDFontStyle;
import de.intarsys.pdf.font.PDFontTrueType;
import de.intarsys.pdf.font.PDFontType1;
import de.intarsys.tools.attribute.Attribute;
import de.intarsys.tools.locator.ByteArrayLocator;
import de.intarsys.tools.locator.ILocator;

public class PlatformFontTools {

	private static final Attribute ATTR_CWTFONT = new Attribute("cwtfont");

	private static final Logger Log = PACKAGE.Log;

	static public boolean embedFontFile(PDFont pdFont) throws IOException {
		if (pdFont.isEmbedded()) {
			return true;
		}
		IFont font = PlatformFontTools.getCWTFont(pdFont);
		if (font == null) {
			return false;
		}
		if (pdFont instanceof PDFontTrueType) {
			if (!"TrueType".equals(font.getFontType())) {
				return false;
			}
		} else if (pdFont instanceof PDFontType1) {
			if (!"Type1".equals(font.getFontType())) {
				return false;
			}
		} else {
			return false;
		}
		ILocator locator = font.getFontProgram().getLocator();
		if (locator == null) {
			return false;
		}
		return embedFontFile(pdFont, locator);
	}

	public static boolean embedFontFile(PDFont pdFont, ILocator locator)
			throws IOException {
		if (pdFont instanceof PDFontTrueType) {
			PDFontTrueTypeTools.embedFontFile(pdFont, locator);
			return true;
		} else if (pdFont instanceof PDFontType1) {
			PDFontType1Tools.embedFontFile(pdFont, locator);
			return true;
		} else {
			return false;
		}
	}

	public static IFont getCWTFont(PDFont pdFont) {
		if (pdFont == null) {
			return null;
		}
		if (pdFont instanceof PDFontAny) {
			return null;
		}
		IFont cwtFont = (IFont) pdFont.getAttribute(ATTR_CWTFONT);
		if (cwtFont != null) {
			return cwtFont;
		}
		//
		Log.log(Level.FINEST, "load font program for " + pdFont.toString());
		//
		FontQuery query = new FontQuery();
		query.setFontType(pdFont.getFontType());
		query.setFontName(pdFont.getLookupFontName());
		query.setFontFamilyName(pdFont.getLookupFontFamilyName());
		PDFontStyle fontStyle = pdFont.getLookupFontStyle();
		FontStyle cwtFontStyle = FontStyle.getFontStyle(fontStyle.getLabel());
		query.setFontStyle(cwtFontStyle);
		cwtFont = FontTools.lookupFont(query);
		//
		if (cwtFont != null) {
			setCWTFont(pdFont, cwtFont);
			return cwtFont;
		}
		if (cwtFontStyle != FontStyle.REGULAR) {
			query.setFontStyle(FontStyle.REGULAR);
			cwtFont = FontTools.lookupFont(query);
			if (cwtFont != null) {
				setCWTFont(pdFont, cwtFont);
				return cwtFont;
			}
		}
		// some programs create invalid type entries
		// !! -> font file type will not correspond to font type - fix
		query.setFontType("Any");
		query.setFontStyle(cwtFontStyle);
		cwtFont = FontTools.lookupFont(query);
		if (cwtFont != null) {
			setCWTFont(pdFont, cwtFont);
			return cwtFont;
		}
		if (cwtFontStyle != FontStyle.REGULAR) {
			query.setFontStyle(FontStyle.REGULAR);
			cwtFont = FontTools.lookupFont(query);
			if (cwtFont != null) {
				setCWTFont(pdFont, cwtFont);
				return cwtFont;
			}
		}
		if (pdFont instanceof PDFontTrueType) {
			// use Arial ....
			query = new FontQuery();
			query.setFontType("TrueType");
			query.setFontName("Arial");
			query.setFontFamilyName("Arial");
			query.setFontStyle(cwtFontStyle);
			cwtFont = FontTools.lookupFont(query);
			if (cwtFont != null) {
				setCWTFont(pdFont, cwtFont);
				return cwtFont;
			}
		}
		// use Helvetica....
		query = new FontQuery();
		query.setFontType("Type1");
		// must provide font name as mapping is defined on font name only
		query.setFontName("Helvetica");
		query.setFontFamilyName("Helvetica");
		query.setFontStyle(cwtFontStyle);
		cwtFont = FontTools.lookupFont(query);
		setCWTFont(pdFont, cwtFont);
		return cwtFont;
	}

	static public ILocator getFontFile(PDFont pdFont) {
		PDFontDescriptor pdDescriptor = pdFont.getFontDescriptor();
		if (pdDescriptor != null) {
			byte[] bytes = null;
			bytes = pdDescriptor.getFontFile();
			if (bytes == null) {
				bytes = pdDescriptor.getFontFile2();
			}
			if (bytes == null) {
				bytes = pdDescriptor.getFontFile3();
			}
			if (bytes != null) {
				return new ByteArrayLocator(bytes, pdFont.getFontName(), "dat");
			}
		}
		IFont font = PlatformFontTools.getCWTFont(pdFont);
		if (font != null) {
			IFontProgram fontProgram = font.getFontProgram();
			return fontProgram.getLocator();
		}
		return null;
	}

	public static void setCWTFont(PDFont pdFont, IFont cwtFont) {
		pdFont.setAttribute(ATTR_CWTFONT, cwtFont);
	}

}
