/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.font;

import java.awt.geom.Rectangle2D;

import de.intarsys.pdf.content.GraphicsState;
import de.intarsys.pdf.font.PDFont;
import de.intarsys.pdf.platform.cwt.rendering.ICSPlatformDevice;

/**
 * The abstraction of a platform font implementation.
 */
public interface IPlatformFont extends IPlatformGlyphsFactory {

	/**
	 * The ascent of this font over the baseline.
	 * 
	 * @return The ascent of this font over the baseline.
	 */
	public int getAscent();

	/**
	 * A bounding box for all glyphs in the font.
	 * 
	 * @return A bounding box for all glyphs in the font.
	 */
	public Rectangle2D getBBox();

	/**
	 * The index into the font characters for the <code>unicode</code>
	 * character.
	 * <p>
	 * This is needed for example when creating Type0 fonts with TrueType CID
	 * fonts.
	 * 
	 * @param unicode
	 * @return The index into the font characters for the <code>unicode</code>
	 *         character.
	 */
	public int getCharIndex(char unicode);

	/**
	 * The descent of this font below the baseline.
	 * 
	 * @return The descent of this font below the baseline.
	 */
	public int getDescent();

	/**
	 * The height of this font. This will be put in the /CapHeight entry in the
	 * font descriptor.
	 * 
	 * @return The height of this font
	 */
	public int getHeight();

	/**
	 * The associated {@link PDFont}.
	 * 
	 * @return The associated {@link PDFont}.
	 */
	public PDFont getPDFont();

	/**
	 * The underline position in PDF glyph space.
	 * 
	 * @return The underline position in PDF glyph space.
	 */
	public int getUnderlinePosition();

	/**
	 * The underline thickness in PDF glyph space.
	 * 
	 * @return The underline thickness in PDF glyph space.
	 */
	public int getUnderlineThickness();

	/**
	 * <code>true </code> if this font is in a bold style.
	 * 
	 * @return <code>true </code> if this font is in a bold style.
	 */
	public boolean isBoldStyle();

	/**
	 * <code>true </code> if this font is in in italic style.
	 * 
	 * @return <code>true </code> if this font is in in italic style.
	 */
	public boolean isItalicStyle();

	/**
	 * <code>true</code> if this font has fixed width characters.
	 * 
	 * @return <code>true</code> if this font has fixed width characters.
	 */
	public boolean isMonospaced();

	/**
	 * <code>true</code> if this font has an internal encoding and shouldn't
	 * be accessed using normal characters codes.
	 * 
	 * @return <code>true</code> if this font has an internal encoding
	 */
	public boolean isSymbolFont();

	/**
	 * This is forwarded from the {@link ICSPlatformDevice} in case the receiver
	 * needs some special actions upon saving the graphics state (operator "q").
	 * 
	 * @param device
	 * @param oldState
	 * @param newState
	 */
	public void saveState(ICSPlatformDevice device, GraphicsState oldState,
			GraphicsState newState);

	/**
	 * This is forwarded from the {@link ICSPlatformDevice} to indicate a change
	 * in the device text state.
	 * <p>
	 * Upon font change, this is called twice, one time for the old font and
	 * onetime for the new font.
	 * 
	 * @param device
	 */
	public void textStateChanged(ICSPlatformDevice device);

}
