/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.color.awt;

import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.color.ICC_ColorSpace;
import java.awt.color.ICC_Profile;
import java.awt.color.ProfileDataException;
import java.awt.image.ColorModel;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.IndexColorModel;

import de.intarsys.pdf.pd.PDCSICCBased;
import de.intarsys.pdf.pd.PDColorSpace;
import de.intarsys.pdf.pd.PDImage;

/**
 * 
 */
public class AwtCSICCBased extends AwtCSCIEBased {
	public AwtCSICCBased(PDColorSpace pdColorSpace) {
		super(pdColorSpace);
	}

	@Override
	protected ColorSpace createColorSpace() {
		PDColorSpace alternate;

		ICC_Profile profile = resolveProfile();
		if (profile != null) {
			try {
				ColorSpace colorSpace;
				float[] testValues;

				colorSpace = new ICC_ColorSpace(profile);
				testValues = new float[colorSpace.getNumComponents()];
				for (int index = 0; index < testValues.length; index++) {
					testValues[index] = colorSpace.getMinValue(index);
				}
				colorSpace.toRGB(testValues);
				return colorSpace;
			} catch (ProfileDataException ex) {
				// fall through
			}
		}

		alternate = myPDColorSpace().getAlternate();
		if (alternate != null) {
			return getColorSpace(alternate).getColorSpace();
		}
		return FALLBACK;
	}

	@Override
	public ColorModel getColorModel(PDImage pdImage) {
		int bits;
		ColorSpace colorSpace;

		bits = pdImage.getBitsPerComponent();
		colorSpace = getColorSpace();
		/*
		 * ComponentColorModel cannot be used when multiple pixels are packed in
		 * 1 byte. Create different color model here if possible to avoid rather
		 * inefficient conversion later.
		 */
		if (bits < 8 && colorSpace.getNumComponents() == 1) {
			byte[] r = new byte[(int) Math.pow(2, bits)];
			byte[] g = new byte[r.length];
			byte[] b = new byte[r.length];
			for (int index = 0; index < r.length; index++) {
				float[] values;

				values = colorSpace.toRGB(new float[] { (float) index
						/ (r.length - 1) });
				r[index] = (byte) ((int) (values[0] * 255) & 0xFF);
				g[index] = (byte) ((int) (values[1] * 255) & 0xFF);
				b[index] = (byte) ((int) (values[2] * 255) & 0xFF);
			}
			ColorModel colormodel = new IndexColorModel(bits, r.length, r, g, b);
			return colormodel;
		}
		return new ComponentColorModel(colorSpace, false, false,
				Transparency.OPAQUE, DataBuffer.TYPE_BYTE);
	}

	protected PDCSICCBased myPDColorSpace() {
		return (PDCSICCBased) getPDColorSpace();
	}

	public ICC_Profile resolveProfile() {
		try {
			return ICC_Profile.getInstance(myPDColorSpace()
					.cosGetProfileStream().getDecodedBytes());
		} catch (IllegalArgumentException ex) {
			return null;
		}
	}

}
