package org.planx.xmlstore.routing.messaging;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

/**
 * A Streamable object is able to write its state to an output stream and
 * a class implementing Streamable must be able to recreate an instance of
 * the class from an input stream. No information about class name is written
 * to the output stream so it must be known what class type is expected when
 * reading objects back in from an input stream. This gives a space
 * advantage over Serializable but essentially kills polymorphy.
 * <p>
 * Since the exact class must be known anyway prior to reading, it is encouraged
 * that classes implementing Streamable also provide a constructor of the form:
 * <p>
 * <code>Streamable(DataInput in) throws IOException;</code>
 *
 * @author Thomas Ambus
 **/
public interface Streamable {
    /**
     * Writes the internal state of the Streamable object to the output stream
     * in a format that can later be read by the same Streamble class using
     * the {@link #fromStream} method.
     **/
    public void toStream(DataOutput out) throws IOException;

    /**
     * Reads the internal state of the Streamable object from the input stream.
     * Immutable objects should throw an <code>UnsupportedOperationException</code>,
     * and instead implement a constructor as described above.
     **/
    public void fromStream(DataInput in) throws IOException;
}
