package org.planx.xmlstore.io;

import java.io.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import org.planx.util.Array;

/**
 * Tools for <code>Streamer</code>s and writing to <code>DataOutput</code> and reading
 * from <code>DataInput</code>.
 *
 * @author Thomas Ambus
 */
public final class Streamers {
    private Streamers() {}

    private static Map<Class,PolymorphicStreamer> pstreamers =
        new HashMap<Class,PolymorphicStreamer>();

    /**
     * Returns a {@link PolymorphicStreamer} for the specified class.
     * This is useful for maintaining the ability to stream the subtypes
     * of various implementations of classes. Each implementation should
     * make sure to register its sub-classes with the
     * <code>PolymorphicStreamer</code> of the appropriate super class
     * before any other part of the system needs to stream its sub-classes.
     */
    public static <E> PolymorphicStreamer<E> getPolymorphicStreamer(Class<E> cls) {
        synchronized (pstreamers) {
            PolymorphicStreamer<E> ps = pstreamers.get(cls);
            if (ps == null) {
                ps = new PolymorphicStreamer<E>();
                pstreamers.put(cls, ps);
            }
            return ps;
        }
    }

    public static Streamer<String> stringStreamer() {
        return new Streamer<String>() {
            public void toStream(DataOutput out, String s) throws IOException {
                writeUTF(out, s);
            }
            public String fromStream(DataInput in) throws IOException {
                return readUTF(in);
            }
        };
    }

    /**
     * Return the number of bytes that will be written by the {@link #writeUTF} method
     * for the specified <code>String</code>.
     */
    public static int utfSize(String s) {
        int size = rawUTFSize(s);
        return shortIntSize(size) + size;
    }

    /**
     * Returns the number of bytes needed to represent the <code>String</code> in
     * Java modified UTF format.
     */
    private static int rawUTFSize(String s) {
        int size = 0;
        for (int i=0, max=s.length(); i<max; i++) {
            int c = s.charAt(i);
            if (c == 0x0000) size += 2;
            else if (c <= 0x007f) size += 1;
            else if (c <= 0x07ff) size += 2;
            else size += 3;
        }
        return size;
    }

    /**
     * Writes a <code>String</code> of any size in Java modified UTF format.
     * Note, that {@link DataOutput#writeUTF} has a maximum UTF size of
     * 65535 bytes.
     */
    public static void writeUTF(DataOutput out, String s) throws IOException {
        int utflen = rawUTFSize(s);
        writeShortInt(out, utflen);
        writeUTF(out, s, utflen);
    }

    /**
     * Writes a <code>String</code> of any size in Java modified UTF format.
     * Note, that {@link DataOutput#writeUTF} has a maximum UTF size of
     * 65535 bytes.
     */
    public static void writeUTF(DataOutput out, String s, int utflen)
                                            throws IOException {
        int strlen = s.length();
        int c, count = 0, i=0;
        byte[] bytearr = new byte[utflen];

        for (; i<strlen; i++) {
            c = s.charAt(i);
            if (!((c >= 0x0001) && (c <= 0x007F))) break;
            bytearr[count++] = (byte) c;
        }
        for (;i < strlen; i++) {
            c = s.charAt(i);
            if ((c >= 0x0001) && (c <= 0x007F)) {
                bytearr[count++] = (byte) c;
            } else if (c > 0x07FF) {
                bytearr[count++] = (byte) (0xE0 | ((c >> 12) & 0x0F));
                bytearr[count++] = (byte) (0x80 | ((c >>  6) & 0x3F));
                bytearr[count++] = (byte) (0x80 | ((c >>  0) & 0x3F));
            } else {
                bytearr[count++] = (byte) (0xC0 | ((c >>  6) & 0x1F));
                bytearr[count++] = (byte) (0x80 | ((c >>  0) & 0x3F));
            }
        }
        out.write(bytearr, 0, utflen);
    }

    /**
     * Reads a <code>String</code> written by
     * {@link #writeUTF(DataOutput,String)}.
     */
    public static String readUTF(DataInput in) throws IOException {
        int utflen = readShortInt(in);
        return readUTF(in, utflen);
    }

    /**
     * Reads a <code>String</code> written by
     * {@link #writeUTF(DataOutput,String,int)}.
     */
    public static String readUTF(DataInput in, int utflen) throws IOException {
        byte[] bytearr = new byte[utflen];
        char[] chararr = new char[utflen];
        int c, char2, char3;
        int count = 0;
        int chararr_count = 0;

        in.readFully(bytearr, 0, utflen);
        while (count < utflen) {
            c = (int) bytearr[count] & 0xff;
            if (c > 127) break;
            count++;
            chararr[chararr_count++] = (char) c;
        }

        while (count < utflen) {
            c = (int) bytearr[count] & 0xff;
            switch (c >> 4) {
                case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
                    count++;
                    chararr[chararr_count++] = (char) c;
                    break;
                case 12: case 13:
                    count += 2;
                    if (count > utflen)
                        throw new UTFDataFormatException
                            ("malformed input: partial character at end");
                    char2 = (int) bytearr[count-1];
                    if ((char2 & 0xC0) != 0x80)
                        throw new UTFDataFormatException
                            ("malformed input around byte " + count);
                    chararr[chararr_count++]=(char)(((c & 0x1F) << 6) |
                                                    (char2 & 0x3F));
                    break;
                case 14:
                    count += 3;
                    if (count > utflen)
                        throw new UTFDataFormatException
                            ("malformed input: partial character at end");
                    char2 = (int) bytearr[count-2];
                    char3 = (int) bytearr[count-1];
                    if (((char2 & 0xC0) != 0x80) || ((char3 & 0xC0) != 0x80))
                        throw new UTFDataFormatException
                            ("malformed input around byte "+(count-1));
                    chararr[chararr_count++]=(char)(((c     & 0x0F) << 12) |
                                                    ((char2 & 0x3F) << 6)  |
                                                    ((char3 & 0x3F) << 0));
                    break;
                default:
                    throw new UTFDataFormatException
                        ("malformed input around byte "+count);
            }
        }
        return new String(chararr, 0, chararr_count);
    }

    public static <E> byte[] toByteArray(E obj, Streamer<E> streamer)
                            throws IOException {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        DataOutput out = new DataOutputStream(bout);
        streamer.toStream(out, obj);
        return bout.toByteArray();
    }

    public static <E> E fromByteArray(byte[] b, Streamer<E> streamer)
                            throws IOException {
        DataInput in = new DataInputStream(new ByteArrayInputStream(b));
        return streamer.fromStream(in);
    }

    /**
     * Returns the number of bytes that will be written by the
     * {@link #writeShortInt} method for the specified <code>int</code>.
     */
    public static int shortIntSize(int i) {
        return (i < 128) ? 1 : 4;
    }

    /**
     * Writes an <code>int</code>. If <code>0 <= i < 128</code> then only
     * a single byte is written. If <code>i >= 128</code> 4 bytes are
     * written. If <code>i < 0</code> an <code>IllegalArgumentException</code>
     * is thrown.
     */
    public static void writeShortInt(DataOutput out, int i) throws IOException {
        if (i < 0) throw new IllegalArgumentException
            ("Integer may not be negative");
        if (i < 128) out.writeByte((byte) i);
        else out.writeInt(i | 0x80000000);
    }

    /**
     * Reads an <code>int</code> written by the <code>writeShortInt</code>
     * method.
     */
    public static int readShortInt(DataInput in) throws IOException {
        byte b0 = in.readByte();
        if (0 <= b0 && b0 < 128) {
            return (int) b0;
        } else {
            byte b1 = in.readByte();
            byte b2 = in.readByte();
            byte b3 = in.readByte();
            return (((b0 & 0x7f) << 24) | ((b1 & 0xff) << 16) |
                    ((b2 & 0xff) << 8) | (b3 & 0xff));
        }
    }

    /**
     * The <code>int</code>s in the array must be positive, and if an
     * <code>int</code> is smaller than 128, only a single byte is written
     * to represent it.
     */
    public static void writePosArray(DataOutput out, int[] arr)
                            throws IOException {
        writeShortInt(out, arr.length);
        for (int i=0; i<arr.length; i++) {
            writeShortInt(out, arr[i]);
        }
    }

    /**
     * Reads data written by {@link #writePosArray}.
     */
    public static int[] readPosArray(DataInput in) throws IOException {
        int size = readShortInt(in);
        int[] arr = new int[size];
        for (int i=0; i<size; i++) {
            arr[i] = readShortInt(in);
        }
        return arr;
    }

    /**
     * Writes a list of objects where the specified <code>Streamer</code> is
     * capable of writing each object contained in the <code>List</code>.
     * A <code>NullPointerException</code> is thrown if the list is
     * <code>null</code>.
     **/
    public static <E> void writeList(DataOutput out, List<E> l, Streamer<E> s)
                                                           throws IOException {
        writeShortInt(out, l.size());
        for (E elm : l) {
            s.toStream(out, elm);
        }
    }

    /**
     * Reads an immutable <code>List</code> of objects, where the specified
     * <code>Streamer</code> is capable of reading each object in the list.
     **/
    public static <E> List<E> readList(DataInput in, Streamer<E> s)
                                                throws IOException {
        int size = readShortInt(in);
        Object[] ss = new Object[size];
        for (int i=0; i<size; i++) {
            ss[i] = s.fromStream(in);
        }
        return Array.asUnmodifiableList((E[]) ss);
    }

    /**
     * Writes <code>false</code> to the output if the object was
     * <code>null</code> and <code>true</code> if the object was
     * non-<code>null</code>.
     * @return The truth value written
     */
    public static boolean wrapNull(DataOutput out, Object o)
                                         throws IOException {
        if (o == null) {
            out.writeBoolean(false);
            return false;
        } else {
            out.writeBoolean(true);
            return true;
        }
    }

    /**
     * Reads a truth value written by {@link #wrapNull} and returns it.
     * If <code>false</code> is returned the object was <code>null</code> and
     * should therefore not be read from the input.
     */
    public static boolean unwrapNull(DataInput in) throws IOException {
        return in.readBoolean();
    }
}
