package org.planx.xmlstore.regions;

import java.io.*;
import java.util.*;
import org.planx.msd.Discriminator;
import org.planx.msd.graph.Compactor;
import org.planx.msd.graph.Compactor.Statistics;
import org.planx.msd.graph.Navigator;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.util.*;
import static org.planx.xmlstore.regions.RegionConfiguration.*;

public class MSDSharer extends Sharer {
    // Multiset Discrimination fields
    private Compactor<SystemNode> compactor;
    private CanonicPolicy nodeNav;

    // Iteration fields
    private LinkedList<Region> regions; // All regions ordered by age (eldest first)
    private LinkedList<Region> rem;     // Regions removed from consideration
    private LinkedList<Region> live;    // Live regions
    private LinkedList<Region> q1;      // Regions pending discrimination
    private LinkedList<Region> q2;      // Regions pending discrimination

    MSDSharer(RegionManager manager) {
        super(manager);
        RegionConfiguration conf = manager.configuration();

        // Init multiset discrimination
        if (conf.POLICY_INSTANCE == null) {
            switch (manager.configuration().POLICY) {
            case POLICY_FIRST:
                conf.POLICY_INSTANCE = new FirstPolicy();
                break;
            case POLICY_LEAST_INTER_REGION:
                conf.POLICY_INSTANCE = new LeastInterRegionPolicy();
                break;
            default:
                throw new IllegalArgumentException("Unknown policy");
            }
        }
        nodeNav = manager.configuration().POLICY_INSTANCE;
        Discriminator<SystemNode> nodeDisc = new NodeDiscriminator();
        compactor = new Compactor<SystemNode>(nodeNav, nodeDisc, true);

        regions = new LinkedList<Region>();
        live = new LinkedList<Region>();
        rem = new LinkedList<Region>();
        initiateIteration();
    }

    synchronized void addRegion(Region newRegion) throws IOException,
                                           UnknownReferenceException {
        if (newRegion == null) throw new NullPointerException();
        share(newRegion);

        regions.addLast(newRegion);

        // Optimized check of simple, complete discrimination
        if (regions.size() == 1) {
            isDiscriminated = true;
            isIterated = true;
            rem.add(newRegion);
        } else {
            isDiscriminated = false;
            isIterated = false;
            live.addLast(newRegion);
            revive();
        }
        initiateIteration();
    }

    synchronized void removeRegion(Region region) {
        regions.remove(region);
        rem.remove(region);
        live.remove(region);
        q1.remove(region);
        q2.remove(region);
    }

    private synchronized void revive() {
        // Make all regions removed from consideration belong to the
        // same group
        Group group = new Group();
        Iterator<Region> it = rem.iterator();
        while (it.hasNext()) {
            Region r = it.next();
            r.setGroup(group);
            it.remove();
        }
        // Add regions to live list in order of age (eldest first)
        live.clear();
        for (Region r : regions) {
            live.add(r);
        }
    }

    private synchronized void initiateIteration() {
        q1 = new LinkedList();
        q2 = new LinkedList();
        for (Region r : live) {
            r.setDirty(false);
            q1.offer(r);
            q2.offer(r);
        }
    }

    /**
     * Incremental sharing algorithm.
     */
    public synchronized void share() throws IOException, UnknownReferenceException {
        Region r1 = null, r2 = null;

        // Find regions to weak-discriminate
        do {
            // Check if the combination queues are empty
            if (q2.peek() == null) {
                // Current region has been weak-discriminated with
                // all other live regions
                if (q1.peek() == null) {
                    // Iteration finished
                    isIterated = true;
                    Iterator<Region> it = live.iterator();
                    while (it.hasNext()) {
                        Region r = it.next();
                        // Remove from consideration?
                        if (!r.isDirty()) {
                            it.remove();
                            rem.add(r);
                        }
                    }
                    // Were all removed from consideration?
                    if (live.size() <= 1) {
                        // Store is discriminated
                        isDiscriminated = true;
                        // But keep iterating to achieve garbage collection
                        revive();
                        initiateIteration();
                        if (live.size() == 1) {
                            // Only one region exists so share it for garbage collection
                            share(live.getFirst());
                        }
                    }
                    // Prepare for all combinations of live regions
                    initiateIteration();
                    return; // Wait with the next iteration
                } else {
                    // Iteration not finished, prepare for next region to be
                    // weak-discriminated with all other live regions
                    q1.remove(); // Next region, please
                    for (Region r : q1) q2.offer(r);
                }
            }
            // Fetch the next combination
            r1 = q1.peek(); // Current region being weak-discriminated with all live
            r2 = q2.poll();

            // Combinations within the same group are ignored,
            // specifically, this includes weak-discriminating
            // a region with itself
        } while (r1 == null || r2 == null || r1.getGroup() == r2.getGroup());

        // Perform two-region weak-discrimination
        invocations++;
        int nodesBefore = r1.currentNodes + r2.currentNodes;
        Compactor.Statistics stats = share(r1, r2);
        int nodesAfter = r1.currentNodes + r2.currentNodes;

        // Any merges?
        if (nodesBefore != nodesAfter) {
            r1.setDirty(true);
            r2.setDirty(true);
        }
    }

    /**
     * Request weak-discrimination of the specified <code>Region</code>.
     */
    public synchronized Compactor.Statistics share(Region r)
              throws IOException, UnknownReferenceException {
        boolean isConsistent;
        Compactor.Statistics stats;

        synchronized (r) {
            if (!r.isClosed()) throw new IllegalArgumentException("Region is not closed");
            SystemNode root = r.getRegionRoot(null);

            nodeNav.setBounds(r.getBound(), null);
            nodeNav.setRegions(r, null);
            compactor.share(root);
            stats = compactor.getStatistics();

            r.persist();
            r.prepareMend(null);
            isConsistent = r.mend(null);
            assert r.assertInterRegion() : "inter-region inconsistency";
            r.releaseRegionRoot();
        }

        if (!isConsistent) {
            share(r); // global roots changed, re-run
        }
        return stats;
    }

    /**
     * Request weak-discrimination of the specified <code>Region</code>s.
     */
    public synchronized Compactor.Statistics share(Region r1, Region r2)
                          throws IOException, UnknownReferenceException {
        boolean isConsistent;
        Compactor.Statistics stats;

        synchronized (r1) {
            synchronized (r2) {
                if (!r1.isClosed()) throw new IllegalArgumentException("First Region is not closed");
                if (!r2.isClosed()) throw new IllegalArgumentException("Second Region is not closed");

                SystemNode root1 = r1.getRegionRoot(null);
                SystemNode root2 = r2.getRegionRoot(r1);
                SystemNode root = new DVMNode
                    (Node.ELEMENT, "SharerRoot", new SystemNode[] {root1,root2}, null);

                nodeNav.setBounds(r1.getBound(), r2.getBound());
                nodeNav.setRegions(r1, r2);
                compactor.share(root);
                stats = compactor.getStatistics();

                r1.persist();
                r2.persist();
                r1.prepareMend(r2);
                r2.prepareMend(r1);
                isConsistent = r1.mend(r2);
                isConsistent = r2.mend(r1) & isConsistent;
                assert r1.assertInterRegion() : "first region inter-region inconsistency";
                assert r2.assertInterRegion() : "second region inter-region inconsistency";
                r1.releaseRegionRoot();
                r2.releaseRegionRoot();
            }
        }
        if (!isConsistent) {
            share(r1, r2); // global roots changed, re-run
        }
        return stats;
    }
}
