package org.nuiton.topia;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaServiceSupportImpl.java 2843 2013-10-11 22:40:03Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-4/topia-persistence/src/main/java/org/nuiton/topia/TopiaServiceSupportImpl.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.framework.TopiaService;

import com.google.common.collect.ImmutableMap;

/**
 * FIXME AThimel 11/10/13 All this life-cycle has to be reviewed
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class TopiaServiceSupportImpl implements TopiaServiceSupport {

    private static final Log log = LogFactory.getLog(TopiaServiceSupportImpl.class);

    protected ImmutableMap<String, TopiaService> services;
    protected AbstractTopiaApplicationContext applicationContext;

    public TopiaServiceSupportImpl(AbstractTopiaApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
        this.services = loadServices(applicationContext.getConfiguration());
    }

    public void init() {
        preInitServices(this.services);
        applicationContext.getHibernateProvider().getHibernateConfiguration(); // force mapping loading
        postInitServices(this.services);
    }

    protected void preInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            if (!service.preInit(null)) { // TODO AThimel 11/10/13 was: this (aka TopiaContext)
                log.warn(String.format("The service named '%1$s' could not be post-initialized (service not activated)",
                        service.getServiceName()));
            }
        }
    }

    protected void postInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            if (!service.postInit(null)) { // TODO AThimel 11/10/13 was: this (aka TopiaContext)
                log.warn(String.format("The service named '%1$s' could not be pre-initialized (service not activated)",
                        service.getServiceName()));
            }
        }
    }

    /**
     * Retrieve service name using SERVICE_NAME static field on service
     * interface.
     *
     * @param interfaceService class of the service
     * @param <E>              type of the service that extends {@link
     *                         TopiaService}
     * @return the service name
     * @throws IllegalAccessException if field SERVICE_NAME can't be accessed
     * @throws NoSuchFieldException   if no field SERVICE_NAME is defined
     */
    protected <E extends TopiaService> String getServiceName(
            Class<E> interfaceService)
            throws IllegalAccessException, NoSuchFieldException {
        Field f = interfaceService.getField("SERVICE_NAME");
        String name = (String) f.get(null);
        return name;
    }

    protected String getProperExceptionMessage(Throwable eee) {
        return eee.getClass().getSimpleName() + " : " + eee.getMessage();
    }

    protected ImmutableMap<String, TopiaService> loadServices(ImmutableMap<String, String> configuration) {
        Map<String, TopiaService> result = new HashMap<String, TopiaService>();
        // recherche des services present dans la config
        for (String key : result.keySet()) {
            if (key.matches("^topia\\.service\\.\\w+$")) {
                String serviceClass = configuration.get(key);
                try {
                    Class<?> forName = Class.forName(serviceClass);
                    Object newInstance = forName.getConstructor().newInstance();
                    TopiaService service = (TopiaService) newInstance;
                    if (key.equals("topia.service." + service.getServiceName())) {
                        result.put(service.getServiceName(), service);
                        log.info(String.format("Service '%1$s' loaded (implementation %2$s)",
                                key, serviceClass));
                    } else {
                        log.warn(String.format("The service with key '%1$s' has a different name '%2$s'! (service not activated)",
                                key, service.getServiceName()));
                    }
                } catch (Throwable eee) {
                    String message =
                            String.format("The service %1$s of type %2$s was not found.",
                                    key, serviceClass);
                    if (log.isDebugEnabled()) {
                        log.debug(message, eee);
                    } else if (log.isErrorEnabled()) {
                        log.error(message);
                    }
                }
            }
        }
        return ImmutableMap.copyOf(result);
    }


    @Override
    public <E extends TopiaService> boolean serviceEnabled(Class<E> interfaceService) {
        boolean result = false;
        try {
            String name = getServiceName(interfaceService);
            result = getServices().containsKey(name);
        } catch (Exception eee) {
            String format = "The service named '%1$s' could not be found for following reason: %2$s";
            String message = String.format(format, interfaceService, getProperExceptionMessage(eee));
            if (log.isDebugEnabled()) {
                log.debug(message, eee);
            } else if (log.isWarnEnabled()) {
                log.warn(message);
            }
        }
        return result;
    }

    @Override
    public <E extends TopiaService> E getService(Class<E> interfaceService) throws TopiaNotFoundException {
        E result;
        try {
            String name = getServiceName(interfaceService);
            result = (E) getServices().get(name);
        } catch (Exception eee) {
            String format = "Could not retreave service %1$s for following reason: %2$s";
            String message = String.format(format, interfaceService, getProperExceptionMessage(eee));
            throw new TopiaNotFoundException(message, eee);
        }
        if (result == null) {
            String message = String.format("The service %1$s was not found.", interfaceService);
            throw new TopiaNotFoundException(message);
        }
        return result;
    }

    @Override
    public Map<String, TopiaService> getServices() {
        return services;
    }

}
