/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaDAOImpl.java 2789 2013-08-05 08:22:58Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-3.1/topia-persistence/src/main/java/org/nuiton/topia/persistence/TopiaDAOImpl.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * TopiaDAOAbstract.java
 *
 * Created: 31 déc. 2005 13:10:34
 *
 * @author poussin <poussin@codelutin.com>
 * @version $Revision: 2789 $
 *
 * Last update: $Date: 2013-08-05 10:22:58 +0200 (Mon, 05 Aug 2013) $
 * by : $Author: tchemit $
 */

package org.nuiton.topia.persistence;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.FlushMode;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.metadata.ClassMetadata;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaFiresSupport;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.PagerBeanUtil;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.security.Permission;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * Cette classe permet d'avoir un ensemble de méthode implantée de façon
 * standard et plus spécifiquement pour Hibernate.
 * <p/>
 * Certains accès à Hibernate sont tout de même fait ici, car on a pris le choix
 * de se baser entièrement sur hibernate pour la persistence, et il est ainsi
 * possible d'accèder au meta information hibernate sur les classes lorque l'on
 * en a besoin.
 *
 * @param <E> le type de l'entite
 * @author bpoussin <poussin@codelutin.com>
 * @version $Id: TopiaDAOImpl.java 2789 2013-08-05 08:22:58Z tchemit $
 */

public class TopiaDAOImpl<E extends TopiaEntity> implements
        TopiaDAO<E> { // TopiaDAOImpl

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(TopiaDAOImpl.class);

    /**
     * Type of entity managed by this dao.
     *
     * @since ever
     */
    protected Class<E> entityClass;

    /**
     * Underlying context used by this dao to do actions on db.
     *
     * @since ever
     */
    protected TopiaContext context;

    protected TopiaFiresSupport topiaFiresSupport;

    /**
     * Default batch size used to iterate on data.
     *
     * @since 2.6.14
     */
    private int batchSize = 1000;

    @Override
    public TopiaEntityEnum getTopiaEntityEnum() {
        throw new UnsupportedOperationException(
                "This method must be overided in generated DAO");
    }

    @Override
    public Class<E> getEntityClass() {
        throw new UnsupportedOperationException(
                "This method must be overided in generated DAO");
    }

    @Override
    public int getBatchSize() {
        return batchSize;
    }

    @Override
    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    private E query(Criterion criterion) throws TopiaException {
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            criteria.add(criterion);
            criteria.setMaxResults(1);
            List<E> result = (List<E>) criteria.list();
            int sizeBefore = result != null ? result.size() : 0;
            result = topiaFiresSupport.fireEntitiesLoad(context, result);
            int sizeAfter = result != null ? result.size() : 0;
            if (sizeAfter < sizeBefore) {
                if (log.isDebugEnabled()) {
                    log.debug((sizeBefore - sizeAfter)
                              + " element(s) removed. Filter entity: "
                              + entityClass.getName() + " - criterion: "
                              + criterion);
                }
            }
            if (result != null && result.size() > 0) {
                E elem = result.get(0);
                return elem;
            }
            return null;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    /**
     * Renvoie un Criteria créé avec l'entityClass
     *
     * @param mode le FlushMode du Criteria
     * @return le Criteria nouvellement créé
     * @throws TopiaException if any pb
     */
    private Criteria createCriteria(FlushMode mode) throws TopiaException {
        Criteria criteria = getSession().createCriteria(entityClass);
        criteria.setFlushMode(mode);
        return criteria;
    }

    @Override
    public Iterator<E> iterator() {

        Iterator<E> iterator = new FindAllIterator<E, E>(
                this,
                getEntityClass(),
                batchSize,
                "FROM " + getTopiaEntityEnum().getImplementationFQN() + " ORDER BY id");

        return iterator;
    }

    /**
     * Retourne l'id de l'entity
     *
     * @param e l'entity
     * @return l'id de l'entity ou null si pas trouvé
     * @throws TopiaException Si une erreur survient durant la recherche
     */
    protected Serializable getId(E e) throws TopiaException {
        ClassMetadata meta = getClassMetadata();
        String idPropName = meta.getIdentifierPropertyName();

        try {
            Serializable result;
            result = (Serializable) PropertyUtils.getSimpleProperty(e,
                                                                    idPropName);
            return result;
        } catch (Exception eee) {
            throw new TopiaException("Impossible de récuperer l'identifiant "
                                     + idPropName + " de l'entite: " + e);
        }
    }

    /**
     * Retourne l'id de l'entity representer comme une map
     *
     * @param map l'entity en representation map
     * @return l'id de l'entity ou null si pas trouvé
     * @throws TopiaException Si une erreur survient durant la recherche
     */
    protected Serializable getId(Map map) throws TopiaException {
        try {
            ClassMetadata meta = getClassMetadata();
            String idPropName = meta.getIdentifierPropertyName();

            Serializable id = (Serializable) map.get(idPropName);
            return id;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    /**
     * When AbstractTopiaContext create the TopiaDAOHibernate, it must call this
     * method just after.
     *
     * @param entityClass
     */
    public void init(TopiaContext context, Class<E> entityClass, TopiaFiresSupport topiaFiresSupport)
            throws TopiaException {
        log.debug("init dao for " + entityClass.getName());
        this.context = context;
        this.entityClass = entityClass;
        this.topiaFiresSupport = topiaFiresSupport;
    }

    @Override
    public TopiaContextImplementor getContext() {
        return (TopiaContextImplementor) getTopiaContext();
    }

    @Override
    public TopiaContext getTopiaContext() {
        return context;
    }

    protected TopiaFiresSupport getTopiaFiresSupport() {
        return topiaFiresSupport;
    }

    @Override
    public String createSimpleQuery(String alias) {
        String hql = "FROM " + getTopiaEntityEnum().getImplementationFQN();
        if (StringUtils.isNotBlank(alias)) {
            hql += " " + alias;
        }
        return hql;
    }

    @SuppressWarnings("unchecked")
    @Override
    public E newInstance() throws TopiaException {
        if (log.isDebugEnabled()) {
            log.debug("entityClass = " + entityClass);
        }
        Class<E> implementation = (Class<E>)
                getTopiaEntityEnum().getImplementation();

        try {
            E result = implementation.newInstance();
            return result;
        } catch (Exception e) {
            throw new TopiaException(
                    "Impossible de trouver ou d'instancier la classe "
                    + implementation);
        }
    }

    @Override
    public <U extends TopiaEntity> List<U> findUsages(Class<U> type, E e)
            throws TopiaException {
        // must be implemented by specialized dao
        throw new UnsupportedOperationException(
                "This method must be overided in generated DAO");
    }

    @Override
    public Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> findAllUsages(E e)
            throws TopiaException {
        // must be implemented by specialized dao
        throw new UnsupportedOperationException(
                "This method must be overided in generated DAO");
    }

    @Override
    public List<Permission> getRequestPermission(String topiaId, int actions)
            throws TopiaException {
        return null;
    }

    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        getContext().addTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        getContext().addTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        getContext().removeTopiaEntityListener(entityClass, listener);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        getContext().removeTopiaEntityVetoable(entityClass, vetoable);
    }

    @Override
    public E create(E entity) throws TopiaException {

        try {
            // first set topiaId
            if (StringUtils.isBlank(entity.getTopiaId())) {

                // only set id if not already on
                String topiaId = getContext().getTopiaIdFactory().newTopiaId(entityClass, entity);
                entity.setTopiaId(topiaId);
            }

            if (entity instanceof TopiaEntityContextable) {
                TopiaEntityContextable contextable = (TopiaEntityContextable)entity;
                contextable.setTopiaContext(getContext());
            }

            // save entity
            getSession().save(entity);
            topiaFiresSupport.warnOnCreateEntity(entity);
            return entity;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public E create(Object... propertyNamesAndValues) throws TopiaException {

        int propertiesLength = propertyNamesAndValues.length;
        Preconditions.checkArgument(propertiesLength % 2 == 0,
                                    "Wrong number of argument "
                                    + propertiesLength
                                    + ", you must have even number.");

        Map<String, Object> map = new HashMap<String, Object>();
        for (int i = 0; i < propertiesLength; ) {
            Object propertyName = propertyNamesAndValues[i++];
            Object value = propertyNamesAndValues[i++];
            Preconditions.checkArgument(
                    propertyName instanceof String,
                    "Argument at position [" + (i - 1) + "] " +
                    "shoud be a property name (says a String) but was " +
                    propertyName);
            map.put((String) propertyName, value);
        }

        E result = create(map);
        return result;
    }

    /**
     * Cette methode appelle fireVetoableCreate et fireOnCreated Si vous la
     * surchargé, faites attention a appeler le super ou a appeler vous aussi
     * ces deux methodes.
     */
    @Override
    public E create(Map<String, Object> properties) throws TopiaException {

        E result = newInstance();

        try {
            for (Map.Entry<String, Object> e : properties.entrySet()) {
                String propertyName = e.getKey();
                Object value = e.getValue();
                PropertyUtils.setProperty(result, propertyName, value);
            }
        } catch (IllegalAccessException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (InvocationTargetException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (NoSuchMethodException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        }

        create(result);

        return result;
    }

    @Override
    public E update(E e) throws TopiaException {
        try {
            getSession().saveOrUpdate(e);
            topiaFiresSupport.warnOnUpdateEntity(e);
            return e;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public void delete(E e) throws TopiaException {
        try {
            getSession().delete(e);
            e.notifyDeleted();
            topiaFiresSupport.warnOnDeleteEntity(e);
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public void deleteAll(Iterable<E> entities) throws TopiaException {
        for (E entity : entities) {
            delete(entity);
        }
    }

    @Override
    public E findByTopiaId(String id) throws TopiaException {
        E result = query(Restrictions.idEq(id));
        return result;
    }

    @Override
    public E findByProperty(String propertyName, Object value)
            throws TopiaException {
        Map<String, Object> properties = new HashMap<String, Object>();
        properties.put(propertyName, value);
        E result = findByProperties(properties);
        return result;
    }

    @Override
    public E findByProperties(String propertyName, Object value,
                              Object... propertyNamesAndValues) throws TopiaException {
        Map<String, Object> properties =
                convertPropertiesArrayToMap(propertyName, value, propertyNamesAndValues);
        E result = findByProperties(properties);
        return result;
    }

    @Override
    public E findByProperties(Map<String, Object> properties)
            throws TopiaException {
        E result = query(Restrictions.allEq(properties));
        return result;
    }

    @Override
    public List<E> findAll() throws TopiaException {
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            List<E> result = (List<E>) criteria.list();
            result = topiaFiresSupport.fireEntitiesLoad(context,
                                                                     result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public List<String> findAllIds() throws TopiaException {
        List<String> find = context.findAll("select src.topiaId from " + getEntityClass().getName() + " src");
        return find;
    }

    @Override
    public List<E> findAllByProperty(String propertyName, Object value)
            throws TopiaException {
        Map<String, Object> properties =
                convertPropertiesArrayToMap(propertyName, value);
        List<E> result = findAllByProperties(properties);
        return result;
    }

    @Override
    public List<E> findAllByProperties(String propertyName, Object value,
                                       Object... propertyNamesAndValues) throws TopiaException {
        Map<String, Object> properties =
                convertPropertiesArrayToMap(propertyName, value, propertyNamesAndValues);
        List<E> result = findAllByProperties(properties);
        return result;
    }

    @Override
    public List<E> findAllByProperties(Map<String, Object> properties)
            throws TopiaException {
        List<E> result = queryAll(Restrictions.allEq(properties));
        return result;
    }

    private List<E> queryAll(Criterion criterion) throws TopiaException {
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            criteria.add(criterion);
            List<E> result = (List<E>) criteria.list();
            result = topiaFiresSupport.fireEntitiesLoad(context,
                                                                     result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public List<E> findAllWithOrder(String... propertyNames)
            throws TopiaException {
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            for (String propertyName : propertyNames) {
                criteria.addOrder(Order.asc(propertyName));
            }
            List<E> result = (List<E>) criteria.list();
            result = topiaFiresSupport.fireEntitiesLoad(context,
                                                                     result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public E findContains(String propertyName,
                          Object value) throws TopiaException {
        E find = context.findUnique("from " + getEntityClass().getName() +
                " WHERE :property in elements(" + propertyName + ")", "property", value);
        return find;
    }

    @Override
    public List<E> findAllContains(String propertyName,
                                   Object value) throws TopiaException {
        List<E> find = context.findAll("from " + getEntityClass().getName() +
                " WHERE :property in elements(" + propertyName + ")", "property", value);
        return find;
    }

    @Override
    public boolean existByTopiaId(String id) throws TopiaException {
        boolean result = existByProperties(TopiaEntity.PROPERTY_TOPIA_ID, id);
        return result;
    }

    @Override
    public boolean existByProperties(String propertyName, Object propertyValue,
                                     Object... propertyNamesAndValues) throws TopiaException {
        Map<String, Object> properties =
                convertPropertiesArrayToMap(propertyName, propertyValue, propertyNamesAndValues);
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            criteria.add(Restrictions.allEq(properties));
            criteria.setProjection(Projections.rowCount());
            Number count = (Number) criteria.uniqueResult();
            boolean result = count.intValue() > 0;
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    @Override
    public long count() throws TopiaException {
        try {
            Criteria criteria = createCriteria(FlushMode.AUTO);
            criteria.setProjection(Projections.rowCount());
            long result  = ((Number)criteria.uniqueResult()).longValue();
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
    }

    /**
     * Convert a properties array to a proper map used to find entities in
     * methods {@link #findByProperties(String, Object, Object...)} and {@link
     * #findAllByProperties(String, Object, Object...)}.
     *
     * @param propertyName  first property name to test existence
     * @param propertyValue first property value to test existence
     * @param others        altern propertyName and propertyValue
     * @return a Map with properties, propertyName as key.
     * @throws IllegalArgumentException for ClassCast or ArrayIndexOutOfBounds
     *                                  errors
     */
    private Map<String, Object> convertPropertiesArrayToMap(String propertyName,
                                                            Object propertyValue,
                                                            Object... others)
            throws IllegalArgumentException {
        Map<String, Object> properties = new HashMap<String, Object>();
        properties.put(propertyName, propertyValue);
        Object name = null;
        for (int i = 0; i < others.length; ) {
            try {
                name = others[i++];
                propertyValue = others[i++];
                properties.put((String) name, propertyValue);
            } catch (ClassCastException eee) {
                throw new IllegalArgumentException(
                        "Les noms des propriétés doivent être des chaines et " +
                        "non pas " + propertyName.getClass().getName(),
                        eee);
            } catch (ArrayIndexOutOfBoundsException eee) {
                throw new IllegalArgumentException(
                        "Le nombre d'argument n'est pas un nombre pair: "
                        + (others.length + 2)
                        + " La dernière propriété était: " + name, eee);
            }
        }
        return properties;
    }

    @Override
    public E findByPrimaryKey(Map<String, Object> keys)
            throws TopiaException {
        try {
            // we used hibernate meta information for all persistence type
            // it's more easy than create different for all persistence
            ClassMetadata meta = getClassMetadata();
            if (meta.hasNaturalIdentifier()) {
                E result = findByProperties(keys);
                return result;
            }
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
        throw new TopiaException("La classe " + entityClass.getName()
                                 + " n'a pas de cle primaire naturelle");

    }

    @Override
    public E findByPrimaryKey(Object... k) throws TopiaException {
        // TODO pour une meilleur gestion des problemes a la compilation
        // mettre un premier couple (propName, value) en argument ca evitera
        // de pouvoir appeler cette methode sans argument
        try {
            ClassMetadata meta = getClassMetadata();
            if (meta.hasNaturalIdentifier()) {
                int[] ikeys = meta.getNaturalIdentifierProperties();
                String[] pnames = meta.getPropertyNames();

                Map<String, Object> keys = new HashMap<String, Object>();
                for (int ikey : ikeys) {
                    keys.put(pnames[ikey], k[ikey]);
                }

                E result = findByProperties(keys);
                return result;
            }
        } catch (HibernateException eee) {
            throw new TopiaException(eee);
        }
        throw new TopiaException("La classe " + entityClass.getName()
                                 + " n'a pas de cle primaire naturelle");
    }

    @Override
    public boolean existsByQuery(String hql,
                                 Object... propertyNamesAndValues) throws TopiaException {
        long count = countByQuery(hql, propertyNamesAndValues);
        return count > 0;
    }

    @Override
    public long countByQuery(String hql,
                             Object... propertyNamesAndValues) throws TopiaException {

        Preconditions.checkNotNull(StringUtils.isNotBlank(hql));
        Preconditions.checkArgument(hql.toUpperCase().trim().startsWith("SELECT COUNT("));

        Long result = findByQuery(Long.class, hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public E findByQuery(String hql,
                         Object... propertyNamesAndValues) throws TopiaException {
        E result = findByQuery(getEntityClass(), hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public <R> R findByQuery(Class<R> type,
                             String hql,
                             Object... params) throws TopiaException {

        Preconditions.checkNotNull(type);
        Preconditions.checkNotNull(hql);

        Object unique = getContext().findUnique(hql, params);
        Preconditions.checkState(unique == null ||
                                 type.isAssignableFrom(unique.getClass()));
        return (R) unique;
    }

    @Override
    public List<E> findAllByQuery(String hql,
                                  Object... propertyNamesAndValues) throws TopiaException {

        List<E> result = findAllByQuery(getEntityClass(), hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public <R> List<R> findAllByQuery(Class<R> type,
                                      String hql,
                                      Object... propertyNamesAndValues) throws TopiaException {

        Preconditions.checkNotNull(type);
        Preconditions.checkNotNull(hql);

        List<R> result = getContext().findAll(hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public Iterable<E> findAllLazyByQuery(String hql,
                                          Object... propertyNamesAndValues) throws TopiaException {
        Iterable<E> result = findAllLazyByQuery(batchSize, hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public <R> Iterable<R> findAllLazyByQuery(Class<R> type,
                                              String hql,
                                              Object... propertyNamesAndValues) throws TopiaException {
        Iterable<R> result = findAllLazyByQuery(type, batchSize, hql, propertyNamesAndValues);
        return result;
    }

    @Override
    public Iterable<E> findAllLazyByQuery(int batchSize,
                                          String hql,
                                          Object... propertyNamesAndValues) throws TopiaException {
        return findAllLazyByQuery(getEntityClass(), batchSize, hql, propertyNamesAndValues);
    }

    @Override
    public <R> Iterable<R> findAllLazyByQuery(Class<R> type,
                                              int batchSize,
                                              String hql,
                                              Object... propertyNamesAndValues) throws TopiaException {

        final Iterator<R> iterator = new FindAllIterator<E, R>(this,
                                                               type,
                                                               batchSize,
                                                               hql,
                propertyNamesAndValues);
        Iterable<R> result = new Iterable<R>() {
            @Override
            public Iterator<R> iterator() {
                return iterator;
            }
        };
        return result;
    }

    @Override
    public <R> List<R> findAllByQueryWithBound(Class<R> type,
                                               String hql,
                                               int startIndex,
                                               int endIndex,
                                               Object... propertyNamesAndValues) throws TopiaException {
        Preconditions.checkNotNull(type);
        Preconditions.checkNotNull(hql);

        List<R> result = getContext().find(hql, startIndex, endIndex, propertyNamesAndValues);
        return result;
    }

    @Override
    public List<E> findAllByQueryWithBound(String hql,
                                           int startIndex,
                                           int endIndex,
                                           Object... propertyNamesAndValues) throws TopiaException {
        List<E> result = findAllByQueryWithBound(getEntityClass(),
                hql,
                startIndex,
                endIndex,
                propertyNamesAndValues);
        return result;
    }

    @Override
    public <R> List<R> findAllByQueryAndPager(Class<R> type,
                                              String hql,
                                              TopiaPagerBean pager,
                                              Object... propertyNamesAndValues) throws TopiaException {
        Preconditions.checkNotNull(pager);
        Preconditions.checkNotNull(hql);

        if (StringUtils.isNotBlank(pager.getSortColumn())) {
            hql += " ORDER BY " + pager.getSortColumn();
            if (!pager.isSortAscendant()) {
                hql += " DESC";
            }
        }
        List<R> result = findAllByQueryWithBound(type, hql,
                (int) pager.getRecordStartIndex(),
                (int) pager.getRecordEndIndex() - 1,
                propertyNamesAndValues);
        return result;
    }

    @Override
    public List<E> findAllByQueryAndPager(String hql,
                                          TopiaPagerBean pager,
                                          Object... propertyNamesAndValues) throws TopiaException {

        List<E> result = findAllByQueryAndPager(getEntityClass(),
                hql,
                pager,
                propertyNamesAndValues);
        return result;
    }

    @Override
    public void computeAndAddRecordsToPager(String hql,
                                            TopiaPagerBean pager,
                                            Object... propertyNamesAndValues) throws TopiaException {

        long records = countByQuery(hql, propertyNamesAndValues);

        pager.setRecords(records);
        PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);
    }

    /**
     * Renvoie la Session contenue dans le contexte
     *
     * @return hibernate session
     * @throws TopiaException if any pb
     */
    protected Session getSession() throws TopiaException {
        Session result = getContext().getHibernateSession();
        return result;
    }

    /**
     * package locale method because this is hibernate specific method and
     * we don't want expose it.
     *
     * @return the meta-data of the entity
     * @throws TopiaException if any pb
     */
    protected ClassMetadata getClassMetadata() throws TopiaException {
        ClassMetadata meta = getContext().getHibernateFactory()
                .getClassMetadata(entityClass);
        if (meta == null) {
            meta = getContext().getHibernateFactory().getClassMetadata(
                    getTopiaEntityEnum().getImplementationFQN());
        }
        return meta;
    }

    public static class FindAllIterator<E extends TopiaEntity, R> implements Iterator<R> {

        protected Iterator<R> data;

        protected final TopiaDAO<E> dao;

        protected final Class<R> type;

        protected final String hql;

        protected final Object[] params;

        protected TopiaPagerBean pager;

        public FindAllIterator(TopiaDAO<E> dao,
                               Class<R> type,
                               int batchSize,
                               String hql,
                               Object... params) {
            this.dao = dao;
            this.type = type;
            this.hql = hql;
            this.params = params;

            String hql2 = hql.toLowerCase();
            int i = hql2.indexOf("order by");
            if (i == -1) {
                throw new IllegalStateException(
                        "must have a *order by* in hql, " +
                        "but did not find it in " + hql);
            }

            // get the count (removing the order-by)
            long count2 = dao.countByQuery("SELECT COUNT(*) " +
                                           hql.substring(0, i), params);
            pager = new TopiaPagerBean();
            pager.setRecords(count2);
            pager.setPageSize(batchSize);
            PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

            // empty iterator (will be changed at first next call)
            data = Iterators.emptyIterator();
        }

        @Override
        public boolean hasNext() {
            boolean result = data.hasNext() || // no more data
                    pager.getPageIndex() < pager.getPagesNumber();
            return result;
        }

        @Override
        public R next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }

            if (!data.hasNext()) {

                // must load iterator

                // increments page index
                pager.setPageIndex(pager.getPageIndex() + 1);
                PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

                // load new window of data
                data = dao.findAllByQueryAndPager(type,
                                                  hql,
                                                  pager,
                                                  params).iterator();
            }

            R next = data.next();
            return next;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException(
                    "This iterator does not support remove operation.");
        }
    }
} //TopiaDAOImpl
