package org.nuiton.topia;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaPersistenceContext.java 2787 2013-08-03 01:14:27Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-3.1/topia-persistence/src/main/java/org/nuiton/topia/TopiaPersistenceContext.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.util.List;
import java.util.Properties;

import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaIdFactory;

/**
 * This API provides all methods related to persistence :
 * <ul>
 * <li>Generic entity find</li>
 * <li>Persistence customization</li>
 * <li>DAO factory</li>
 * <li>Schema management</li>
 * </ul>
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 * @since 3.0
 */
public interface TopiaPersistenceContext {

    /**
     * Retrieve {@link org.nuiton.topia.persistence.TopiaEntity} using its unique {@code topiaId}.
     *
     * @param topiaId unique identifier of the entity in all the application.
     * @return the entity found or null
     */
    <E extends TopiaEntity> E findByTopiaId(String topiaId);

    /**
     * Add into this TopiaContext an entity created by another TopiaContext
     *
     * @param entity the entity to add
     */
    void update(TopiaEntity entity);

    /**
     * Get the currently configured {@link TopiaIdFactory}.
     *
     * @return the {@link TopiaIdFactory} in use
     * @see {@link TopiaIdFactory}
     * @since 3.0
     */
    TopiaIdFactory getTopiaIdFactory();

    /**
     * @return the list of classes managed by this TopiaPersistenceContext
     */
    List<Class<?>> getPersistenceClasses();

    /**
     * Tells to the context if it has to use a flush mode before each query.
     * <p/>
     * By default, we use a flush mode, but in some case it costs to much doing
     * this, that's why you can disable it setting the value to {@code false}.
     *
     * @param useFlushMode the new value to set
     * @since 2.5
     */
    void setUseFlushMode(boolean useFlushMode);

    /**
     * @return Returns the config.
     */
    Properties getConfig(); // TODO AThimel 02/08/13 Change type ?

    /**
     * Get DAO for specified class. If Specialized DAO exists then it returned
     * otherwize TopiaDAO&lt;entityClass&gt; is returned
     *
     * @param entityClass type of entity
     * @return the expected dao
     */
    <E extends TopiaEntity> TopiaDAO<E> getDAO(Class<E> entityClass);

    /**
     * Get DAO for specified class. If Specialized DAO exists then it returned
     * otherwize TopiaDAO&lt;entityClass&gt; is returned
     *
     * @param entityClass type of entity
     * @param daoClass    the concrete dao class to use
     * @return the expected dao
     */
    <E extends TopiaEntity, D extends TopiaDAO<E>> D getDAO(Class<E> entityClass, Class<D> daoClass);

    /**
     * Detects if the db associated to the current context contains at least one
     * entity table.
     *
     * @return {@code true} if there is no schema for any of the dealed entities,
     *         {@code false} otherwise.
     * @since 3.0
     */
    boolean isSchemaEmpty();

    /**
     * Detects if the table is created on storage for a given persistent class.
     *
     * @param clazz the researched class
     * @return true if the table exists on storage
     * @since 3.0
     */
    boolean isTableExists(Class<?> clazz);

    /**
     * @return the current schema name
     * @since 3.0
     */
    String getSchemaName();

    /**
     * Triggers database schema creation
     */
    void createSchema();

    /**
     * Displays the SQL queries that would be used for a schema creation
     */
    void showCreateSchema();

    /**
     * Triggers database schema update
     */
    void updateSchema();

    /**
     * Triggers database schema drop
     *
     * @since 3.0
     */
    void dropSchema();

}
