/*
 * #%L
 * Maven helper plugin
 * 
 * $Id: Xpp3Reader.java 701 2010-04-15 14:01:44Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-helper-plugin/tags/maven-helper-plugin-1.2.7/src/main/java/org/nuiton/io/xpp3/Xpp3Reader.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.io.xpp3;

import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

import java.io.IOException;
import java.io.Reader;

/**
 * A simple contract to mark all xpp readers.
 * <p/>
 * a such reader is associated to a {@link #getType()}.
 * <p/>
 * The implementations of such readers must be register in a file
 * <pre>
 * META-INF/services.org.nuiton.io.xpp3.Xpp3Reader
 * </pre>
 * <p/>
 * to make possible auto-discovering of availables reader at runtime.
 * <p/>
 * See {@link Xpp3Helper#getReader(Class)} and
 * {@link Xpp3Helper#getReaders()}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @param <O> the type of object to be build by the reader.
 * @since 1.0.3
 */
public interface Xpp3Reader<O> {

    /** @return the type of main object to read */
    Class<O> getType();

    /**
     * Read a single instance of the typed object and return it.
     * <p/>
     * Note : this is a convinient method to call
     * {@link #read(Reader, boolean)} in strict mode.
     * <p/>
     * In the xml stream, the root tag must be the {@link #getRootTagName()}.
     * <p/>
     * Example :
     * <pre>
     * &lt;issue&gt:
     *    ...
     * &lt;/issue&gt:
     * </pre>
     *
     * @param reader the xml input reader
     * @return Settings
     * @throws IOException            if any io pb
     * @throws XmlPullParserException if parsing error
     */
    O read(Reader reader) throws IOException, XmlPullParserException;

    /**
     * Read a single instance of the typed object and return it.
     * <p/>
     * In the xml stream, the root tag must be the {@link #getRootTagName()}.
     * <p/>
     * Example :
     * <pre>
     * &lt;issue&gt:
     *    ...
     * &lt;/issue&gt:
     * </pre>
     *
     * @param reader the xml input reader
     * @param strict flag to be strict while parsing
     * @return the read object
     * @throws IOException            if any io pb
     * @throws XmlPullParserException if any parsing pb
     */
    O read(Reader reader, boolean strict) throws IOException,
            XmlPullParserException;

    /**
     * Read some instances of the typed object and return it.
     * <p/>
     * In the xml stream, the root tag must be the
     * {@link #getArrayRootTagName()}.
     * <p/>
     * Note : this is a convinient method to call :
     * {@link #readArray(Reader, boolean)} in stritc mode.
     * <p/>
     * Example :
     * <pre>
     * &lt;issues&gt:
     *   &lt;issue&gt:
     *    ...
     *   &lt;/issue&gt:
     * &lt;/issues&gt:
     * </pre>
     *
     * @param reader the xml input reader
     * @return the array of read objects.
     * @throws IOException            if any io pb
     * @throws XmlPullParserException if any parsing pb
     */
    O[] readArray(Reader reader) throws IOException, XmlPullParserException;

    /**
     * Read some instances of the typed object and return it.
     * <p/>
     * In the xml stream, the root tag must be the
     * {@link #getArrayRootTagName()}.
     * <p/>
     * Example :
     * <pre>
     * &lt;issues&gt:
     *   &lt;issue&gt:
     *    ...
     *   &lt;/issue&gt:
     * &lt;/issues&gt:
     * </pre>
     *
     * @param reader the xml input reader
     * @param strict flag to be strict while parsing
     * @return the array of read objects.
     * @throws IOException            if any io pb
     * @throws XmlPullParserException if any parsing pb
     */
    O[] readArray(Reader reader, boolean strict) throws IOException,
            XmlPullParserException;

    /** @return the name of the root tag of a object to read */
    String getRootTagName();

    /**
     * Set the name of the root tag of an object to read.
     *
     * @param rootTagName the name of the tag
     */
    void setRootTagName(String rootTagName);

    /** @return the name of the root tag of an array of objets to read */
    String getArrayRootTagName();

    /**
     * Set the name of the root tag for an array of object to read.
     *
     * @param parentRootTagName the name of the tag
     */
    void setParentRootTagName(String parentRootTagName);

    /**
     * @return <code>true</code> if parser will load the default entities,
     *         <code>false</code> otherwise.
     */
    boolean isAddDefaultEntities();

    /**
     * Set the new value of the {@code defaultEntities} flag.
     *
     * @param addDefaultEntities the new value.
     */
    void setAddDefaultEntities(boolean addDefaultEntities);
}
