/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: PublishNewsMojo.java 157 2010-10-08 10:23:16Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.1.4/maven-jredmine-plugin/src/main/java/org/nuiton/jredmine/plugin/PublishNewsMojo.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import org.apache.maven.plugin.MojoExecutionException;
import org.nuiton.jredmine.model.News;
import org.nuiton.plugin.PluginHelper;

import java.io.File;

/**
 * Publish a news on redmine server.
 *
 * @author tchemit
 * @goal publish-news
 * @since 1.0.0
 */
public class PublishNewsMojo extends AbstractRedmineMojo {
    /**
     * Flag to know if anonymùous connexion to redmine server is required.
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @parameter expression="${redmine.anonymous}" default-value="false"
     * @since 1.1.3
     */
    protected boolean anonymous;

    /**
     * The content file of the news.
     *
     * @parameter expression="${redmine.newsContentFile}"
     * @required
     * @since 1.0.0
     */
    protected File newsContentFile;

    /**
     * Short description or introduction of the released artifact.
     *
     * @parameter expression="${redmine.newsSummary}"
     * @since 1.0.0
     */
    protected String newsSummary;

    /**
     * The title of the news to create on redmine server.
     * <p/>
     * <b>Note : </b> the size can not be more than 60 caracters (due to a
     * redmine limitation).
     *
     * @parameter expression="${redmine.newsTitle}"
     * @required
     * @since 1.0.0
     */
    protected String newsTitle;

    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${redmine.skipPublishNews}" default-value="false"
     * @since 1.0.0
     */
    protected boolean skipPublishNews;

    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @parameter expression="${redmine.dryRun}" default-value="false"
     * @since 1.0.0
     */
    protected boolean dryRun;

    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @parameter expression="${redmine.runOnce}" default-value="true"
     * @since 1.0.0
     */
    protected boolean runOnce;

    public PublishNewsMojo() {
        super(true, false, true);
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    @Override
    protected boolean isGoalSkip() {
        return skipPublishNews;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected void init() throws Exception {

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return;
            }
        }

        if (newsSummary == null || newsSummary.trim().isEmpty()) {
            newsSummary = project.getUrl();
        }

        if (!newsContentFile.exists()) {
            // no file to publish...
            throw new MojoExecutionException("could not find the template " + newsContentFile);
        }

        newsTitle = newsTitle.trim();
        if (newsTitle.length() > 60) {
            getLog().warn("News title can not be longer than 60 caracters, but was " + newsTitle.length());
            newsTitle = newsTitle.substring(0, 59);
            getLog().warn("will use the restricted title : " + newsTitle);
        }

        super.init();
    }

    @Override
    protected void doAction() throws Exception {

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        // create news to publish

        News news = new News();
        news.setAuthorId(releaseUser.getId());
        news.setProjectId(releaseProject.getId());
        news.setTitle(newsTitle);
        news.setSummary(newsSummary);

        String newsContent = PluginHelper.readAsString(newsContentFile, encoding);

        news.setDescription(newsContent);

        if (dryRun) {
            getLog().info("news title   : " + news.getTitle());
            getLog().info("news summary : " + news.getSummary());
            getLog().info("news content :\n" + newsContent);
            return;
        }

        // publish news

        getLog().info("publish news " + news.getTitle());

        if (verbose) {
            getLog().info("redmine announcement :\n" + newsContent);
        }

        news = service.addNews(projectId, news);

        if (verbose) {
            getLog().info("done. news id : " + news.getId());
        }
    }
}
