/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin;

import java.io.File;
import org.nuiton.jredmine.model.News;
import org.nuiton.plugin.PluginHelper;

/**
 * Publish a news on redmine server.
 *
 * @goal publish-news
 *
 * @author tchemit
 * @since 1.0.0
 */
public class PublishNewsMojo extends AbstractRedmineMojo {

    /**
     * The content file of the news.
     *
     * @parameter expression="${redmine.newsContentFile}"
     * @required
     * @since 1.0.0
     */
    private File newsContentFile;
    /**
     * Short description or introduction of the released artifact.
     *
     * @parameter expression="${redmine.newsSummary}"
     * @since 1.0.0
     */
    private String newsSummary;
    /**
     * The title of the news to create on redmine server.
     *
     * <b>Note : </b> the size can not be more than 60 caracters (due to a 
     * redmine limitation).
     *
     * @parameter expression="${redmine.newsTitle}"
     * @required
     * @since 1.0.0
     */
    private String newsTitle;
    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${redmine.skipPublishNews}" default-value="false"
     * @since 1.0.0
     */
    private boolean skipPublishNews;
    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @parameter expression="${redmine.dryRun}" default-value="false"
     * @since 1.0.0
     */
    protected boolean dryRun;
    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @parameter expression="${redmine.runOnce}" default-value="true"
     * @since 1.0.0
     */
    private boolean runOnce;

    public PublishNewsMojo() {
        super(true, false, true);
    }

    @Override
    protected boolean isGoalSkip() {
        return skipPublishNews;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean init() throws Exception {

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return true;
            }
        }
        if (!super.init()) {
            return false;
        }

        if (newsSummary == null || newsSummary.trim().isEmpty()) {
            newsSummary = project.getUrl();
        }

        if (!newsContentFile.exists()) {
            // no file to publish...
            getLog().warn("could not find the template " + newsContentFile);
            return false;
        }

        newsTitle = newsTitle.trim();
        if (newsTitle.length() > 60) {
            getLog().warn("News title can not be longer than 60 cars, but was " + newsTitle.length());
            newsTitle = newsTitle = newsTitle.substring(0, 59);
            getLog().warn("will use the restricted title : " + newsTitle);
        }

        return true;
    }

    @Override
    protected void doAction() throws Exception {

        if (isRunOnceDone()) {
            getLog().info("skip goal, runOnce flag is on, and was already executed.");
            return;
        }

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        // create news to publish

        News news = new News();
        news.setAuthorId(releaseUser.getId());
        news.setProjectId(releaseProject.getId());
        news.setTitle(newsTitle);
        news.setSummary(newsSummary);

        String newsContent = PluginHelper.readAsString(newsContentFile, encoding);

        news.setDescription(newsContent);

        if (dryRun) {
            getLog().info("news title   : " + news.getTitle());
            getLog().info("news summary : " + news.getSummary());
            getLog().info("news content :\n" + newsContent);
            return;
        }

        // publish news

        getLog().info("publish news " + news.getTitle());

        if (verbose) {
            getLog().info("redmine announcement :\n" + newsContent);
        }

        news = service.addNews(releaseProject.getName(), news);

        if (verbose) {
            getLog().info("done. news id : " + news.getId());
        }
    }
}
