package org.nuiton.jredmine.service;

/*
 * #%L
 * JRedmine :: Client
 * $Id: DefaultRedmineAnonymousService.java 407 2013-08-08 09:34:23Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.6/jredmine-client/src/main/java/org/nuiton/jredmine/service/DefaultRedmineAnonymousService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jredmine.client.RedmineRequest;
import org.nuiton.jredmine.client.RedmineRequestHelper;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.TimeEntry;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;
import org.nuiton.jredmine.model.VersionStatusEnum;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;


/**
 * Default implementation of {@link RedmineAnonymousService}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.jredmine.service.RedmineAnonymousService" role-hint="default"
 * @since 1.5
 */
public class DefaultRedmineAnonymousService extends AbstractRedmineService implements RedmineAnonymousService {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(DefaultRedmineAnonymousService.class);

    @Override
    public void init(RedmineServiceConfiguration configuration) throws RedmineServiceException {
        Preconditions.checkNotNull(configuration, "the client configuration was not be null!");

        // always clone the configuration to keep it in the service
        RedmineServiceConfiguration serviceConfiguration =
                RedmineConfigurationUtil.cloneConfiguration(configuration);

        // Force to not be loggued
        serviceConfiguration.setAnonymous(true);

        if (log.isDebugEnabled()) {
            log.debug("init service with configuration: " + serviceConfiguration);
        }
        if (!isInit()) {
            super.init(serviceConfiguration);
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineAnonymousService
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public IssueStatus[] getIssueStatuses() throws RedmineServiceException {
        RedmineRequest<IssueStatus> request =
                RedmineRequestHelper.action("get_issue_statuses.xml",
                                            IssueStatus.class);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public IssuePriority[] getIssuePriorities() throws RedmineServiceException {
        RedmineRequest<IssuePriority> request = RedmineRequestHelper.action(
                "get_issue_priorities.xml", IssuePriority.class);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Project[] getProjects() throws RedmineServiceException {
        RedmineRequest<Project> request = RedmineRequestHelper.action(
                "get_projects.xml", Project.class);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public IssueCategory[] getIssueCategories(String projectName) throws RedmineServiceException {
        RedmineRequest<IssueCategory> request =
                RedmineRequestHelper.actionWithProject(
                        "get_issue_categories.xml",
                        IssueCategory.class,
                        projectName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Project getProject(String projectName) throws RedmineServiceException {
        RedmineRequest<Project> request =
                RedmineRequestHelper.actionWithProject("get_project.xml",
                                                       Project.class,
                                                       projectName);
        return executeRequestAndReturnData(request
        );
    }

    @Override
    public Tracker[] getTrackers(String projectName) throws RedmineServiceException {
        RedmineRequest<Tracker> request =
                RedmineRequestHelper.actionWithProject("get_project_trackers.xml",
                                                       Tracker.class,
                                                       projectName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public News[] getNews(String projectName) throws RedmineServiceException {
        RedmineRequest<News> request =
                RedmineRequestHelper.actionWithProject("get_project_news.xml",
                                                       News.class,
                                                       projectName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public User[] getProjectMembers(String projectName) throws RedmineServiceException {
        RedmineRequest<User> request =
                RedmineRequestHelper.actionWithProject("get_project_users.xml",
                                                       User.class,
                                                       projectName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Version[] getVersions(String projectName) throws RedmineServiceException {
        RedmineRequest<Version> request =
                RedmineRequestHelper.actionWithProject("get_project_versions.xml",
                                                       Version.class,
                                                       projectName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Version getVersion(String projectName,
                              String versionName) throws RedmineServiceException {
        RedmineRequest<Version> request =
                RedmineRequestHelper.actionWithProjectAndVersion("get_version.xml",
                                                                 Version.class,
                                                                 projectName,
                                                                 versionName);
        return executeRequestAndReturnData(request);
    }

    @Override
    public Version getLastestClosedVersion(String projectName) throws RedmineServiceException {

        // get all versions
        Version[] versions = getVersions(projectName);
        List<Version> closedVersions = new ArrayList<Version>();
        for (Version version : versions) {
            if (version.getEffectiveDate() != null &&
                VersionStatusEnum.closed.name().equals(version.getStatus())) {
                closedVersions.add(version);
            }
        }

        Version result = null;
        if (!closedVersions.isEmpty()) {
            Collections.sort(closedVersions, new Comparator<Version>() {
                @Override
                public int compare(Version o1, Version o2) {
                    return o2.getEffectiveDate().compareTo(o1.getEffectiveDate());
                }
            });
            result = closedVersions.get(0);
        }
        return result;
    }

    @Override
    public Attachment[] getAttachments(String projectName,
                                       String versionName) throws RedmineServiceException {
        RedmineRequest<Attachment> request =
                RedmineRequestHelper.actionWithProjectAndVersion("get_version_attachments.xml",
                                                                 Attachment.class,
                                                                 projectName,
                                                                 versionName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Issue[] getIssues(String projectName,
                             String versionName) throws RedmineServiceException {
        RedmineRequest<Issue> request =
                RedmineRequestHelper.actionWithProjectAndVersion("get_version_issues.xml",
                                                                 Issue.class,
                                                                 projectName,
                                                                 versionName);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public TimeEntry[] getIssueTimeEntries(String projectName,
                                           String issueId) throws RedmineServiceException {
        RedmineRequest<TimeEntry> request =
                RedmineRequestHelper.actionWithProjectAndIssue("get_issue_times.xml",
                                                               TimeEntry.class,
                                                               projectName,
                                                               issueId);
        return executeRequestAndReturnDatas(request);
    }

    @Override
    public Issue[] getIssues(String projectName) throws RedmineServiceException {
        RedmineRequest<Issue> request =
                RedmineRequestHelper.actionWithProject("get_project_issues.xml",
                                                       Issue.class,
                                                       projectName);
        Issue[] result = executeRequestAndReturnDatas(request);
        return result;
    }

    @Override
    public Issue[] getOpenedIssues(String projectName) throws RedmineServiceException {
        RedmineRequest<Issue> request =
                RedmineRequestHelper.actionWithProject("get_project_opened_issues.xml",
                                                       Issue.class,
                                                       projectName);
        Issue[] result = executeRequestAndReturnDatas(request);
        return result;
    }

    @Override
    public Issue[] getClosedIssues(String projectName) throws RedmineServiceException {
        RedmineRequest<Issue> request =
                RedmineRequestHelper.actionWithProject("get_project_closed_issues.xml",
                                                       Issue.class,
                                                       projectName);
        Issue[] result = executeRequestAndReturnDatas(request);
        return result;
    }
}
