package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXInitialContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.navigation.NavigationTreeModel.NavigationTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JPanel;
import javax.swing.event.TreeSelectionEvent;
import java.awt.Component;

/**
 * Simple {@link NavigationTreeSelectionAdapter} implementation with a {@link jaxx.runtime.swing.CardLayout2} to manage components to
 * associated with tree's nodes.
 * <p/>
 * For each node, the ui associated has a constraints in a cardlayout which is the node context path.
 * <p/>
 * A single container managed by the cardlayout is used to display the components associated with tree's nodes.
 *
 * @author chemit
 */
public abstract class NavigationTreeSelectionAdapterWithCardLayout extends NavigationTreeSelectionAdapter {


    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(NavigationTreeSelectionAdapterWithCardLayout.class);

    /**
     * All components associated with a tree's node is displayed in a single container.
     *
     * @return the containter of components
     */
    protected abstract JPanel getContentContainer();

    /**
     * the cardlayout managing components associated with tree node. The constraints
     * of each component is the node contextPath.
     *
     * @return the layout used to display components associated with tree's nodes.
     */
    protected abstract CardLayout2 getContentLayout();

    public NavigationTreeSelectionAdapterWithCardLayout(Class<? extends JAXXObject> defaultUIClass, Class<? extends JAXXAction> defaultUIHandlerClass, JAXXObject context, Strategy strategy) {
        super(defaultUIClass, defaultUIHandlerClass, context,strategy);

        if (getContentContainer() == null) {
            throw new IllegalArgumentException("could not have a null 'contentContainer' in ui " + context);
        }
        if (getContentLayout() == null) {
            throw new IllegalArgumentException("could not have a null 'contentLayout' in ui " + context);
        }
    }

    protected Component getCurrentUI() {
        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        return layout.getVisibleComponent(container);
    }

    protected Component getUI(NavigationTreeNode node) {
        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        String path = getNodeConstraints(node);
        return layout.contains(path) ? layout.getComponent(container, path) : null;
    }

    protected void openUI(Component newUI, NavigationTreeNode node) throws Exception {

        CardLayout2 layout = getContentLayout();
        JPanel container = getContentContainer();
        // switch layout
        layout.show(container, getNodeConstraints(node));
    }

    protected boolean closeUI(TreeSelectionEvent event, Component component) throws Exception {
        // by default, we says that component was succesfull closed
        return true;
    }

    protected Component createUI(NavigationTreeNode node) throws Exception {
        JAXXObject newUI;

        if (node.getJaxxActionClass() != null) {
            JAXXAction action = node.getJaxxActionClass().newInstance();
            // init context with
            JAXXInitialContext context = action.init(this.context);
            // must instanciate the ui with an JAXXInitialContext
            newUI = node.getJaxxClass().getConstructor(JAXXContext.class).newInstance(context);
        } else {
            if (log.isWarnEnabled()) {
                log.warn("no action associated with ui " + node.getJaxxClass());
            }
            // no action associated, just
            newUI = node.getJaxxClass().getConstructor(JAXXContext.class).newInstance(this.context);
        }
        if (log.isDebugEnabled()) {
            log.debug("instanciate new ui " + newUI);
        }

        getContentContainer().add((Component) newUI, getNodeConstraints(node));
        return (Component) newUI;
    }
}

