/**
 * *##% jaxx-runtime-swing-widget
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package jaxx.runtime.swing.editor.config.model;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ApplicationConfig.OptionDef;
import static org.nuiton.i18n.I18n._;

/**
 * Le modele de l'ui des preferences.
 *
 * Ce modele contient les catégories des options.
 * 
 * @author chemit
 */
public class ConfigUIModel implements Iterable<CategoryModel> {

    public static final String CATEGORY_MODEL_PROPERTY_NAME = "categoryModel";
    /** le dictionnaire des options disponibles par categorie */
    protected final Map<String, CategoryModel> categories;
    /** La configuration de l'application */
    protected final ApplicationConfig config;
    /** la cateogrie en cours d'utilisation */
    protected CategoryModel categoryModel;
    /**
     * un drapeau pour savoir si la configuration a été modifiée au moins une
     * fois.
     *
     * On utilise ce drapeau lors de la sortie pour verifier s'il faut ou non
     * redemarer l'application (si non en mode standalone)
     */
    protected boolean saved;
    /**
     * un drapeau pour savoir si l'ui de configuration a été lancée en mode
     * standalone ou pas.
     *
     * Si pas lancée en mode standalone, et si la confi a été sauvé on vérifie
     * s'il ne faut pas relancer l'application.
     */
    protected boolean standalone;
    /** suport of modification */
    protected final PropertyChangeSupport pcs = new PropertyChangeSupport(this);

    public ConfigUIModel(ApplicationConfig config) {
        this.config = config;
        this.categories = new LinkedHashMap<String, CategoryModel>();
    }

    /**
     * Ajoute une categorie dans le modele.
     *
     * @param category l'id de la categorie (la clef de traduction du nom de la categorie)
     * @param categoryLabel la clef de traduction de la description de la categorie
     * @param keys les options de la categorie
     */
    public void addCategory(String category, String categoryLabel, OptionDef... keys) {
        if (categories.containsKey(category)) {
            throw new IllegalArgumentException(_("config.error.category.already.exists", category));
        }
        OptionModel[] entries = new OptionModel[keys.length];
        int index = 0;
        for (OptionDef d : keys) {
            Object value = config.getOption(d);
            OptionModel e = new OptionModel(d, value);
            entries[index++] = e;
        }
        CategoryModel m = new CategoryModel(category, categoryLabel, entries);
        categories.put(category, m);
    }

    /**
     * Change la categorie en cours d'édition.
     * 
     * @param category l'id de la categorie courante
     */
    public void setCategory(String category) {
        if (!categories.containsKey(category)) {
            throw new IllegalArgumentException(_("config.error.category.not.found", category));
        }
        CategoryModel newCategoryModel = categories.get(category);
        setCategoryModel(newCategoryModel);
        newCategoryModel.firePropertyChange(CategoryModel.MODIFIED_PROPERTY_NAME, false, getCategoryModel().isModified());
        newCategoryModel.firePropertyChange(CategoryModel.VALID_PROPERTY_NAME, false, getCategoryModel().isValid());
    }

    @Override
    public Iterator<CategoryModel> iterator() {
        return categories.values().iterator();
    }

    public CategoryModel getCategoryModel() {
        return categoryModel;
    }

    public void setCategoryModel(CategoryModel categoryModel) {
        CategoryModel old = this.categoryModel;
        this.categoryModel = categoryModel;
        firePropertyChange(CATEGORY_MODEL_PROPERTY_NAME, old, categoryModel);
    }

    public boolean isSaved() {
        return saved;
    }

    public void setSaved(boolean saved) {
        this.saved = saved;
    }

    public boolean isStandalone() {
        return standalone;
    }

    public void setStandalone(boolean standalone) {
        this.standalone = standalone;
    }

    public void saveModified() {
        // compute transients keys (to never be saved)
        List<String> transients = new ArrayList<String>();

        for (OptionModel option : categoryModel) {
            if (option.isModified()) {
                Object value = option.getValue();
                //TODO TC-20090245 : should try to seek for a mutator, since
                // mutator could have extra code to be done when modify an option
                config.setOption(option.getKey(), value == null ? null : value.toString());
                // l'option a été sauvegardée, on la marque
                option.setSaved(true);
                // this is the new original value
                option.initValue(value);
            }
            if (option.isTransient()) {
                transients.add(option.getKey());
            }
        }
        setSaved(true);
        // save config
        config.saveForUser(transients.toArray(new String[transients.size()]));
        // notify data has changed
        categoryModel.firePropertyChange(CategoryModel.MODIFIED_PROPERTY_NAME, categoryModel.isModified(), true);
        categoryModel.firePropertyChange(CategoryModel.VALID_PROPERTY_NAME, false, categoryModel.isValid());
        categoryModel.firePropertyChange(CategoryModel.RELOAD_PROPERTY_NAME, false, true);
    }

    public void reset() {
        // reset all modified options of the current category
        for (OptionModel key : categoryModel) {
            if (key.isModified()) {
                key.initValue(key.getOriginalValue());
            }
        }
        // notify data has changed
        categoryModel.firePropertyChange(CategoryModel.MODIFIED_PROPERTY_NAME, categoryModel.isModified(), true);
        categoryModel.firePropertyChange(CategoryModel.VALID_PROPERTY_NAME, false, categoryModel.isValid());
        categoryModel.firePropertyChange(CategoryModel.RELOAD_PROPERTY_NAME, false, true);
    }

    public int getCategoryIndex(String category) {
        int i = 0;
        for (CategoryModel m : this) {
            if (category.equals(m.getCategory())) {
                return i;
            }
            i++;
        }
        // not found
        return -1;
    }

    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(propertyName, listener);
    }

    public synchronized boolean hasListeners(String propertyName) {
        return pcs.hasListeners(propertyName);
    }

    public synchronized PropertyChangeListener[] getPropertyChangeListeners(String propertyName) {
        return pcs.getPropertyChangeListeners(propertyName);
    }

    public synchronized PropertyChangeListener[] getPropertyChangeListeners() {
        return pcs.getPropertyChangeListeners();
    }
}
