package org.nuiton.jaxx.application.swing.action;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.application.swing.ApplicationUIContext;
import org.nuiton.util.TimeLog;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.SwingUtilities;
import java.awt.event.ActionEvent;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Abstract tutti ui action which launch a {@link AbstractApplicationAction}.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.8.2
 */
public class ApplicationUIAction<A extends AbstractApplicationAction> extends AbstractAction {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ApplicationUIAction.class);

    private static final TimeLog TIME_LOG = new TimeLog(ApplicationUIAction.class);

    public static final ExecutorService waitingThread =
            Executors.newSingleThreadExecutor();

    private static final String LOGIC_ACTION = "logicAction";

    private final Object lock = new Object();

    private boolean wait;

    private long t0;

    public ApplicationUIAction(final AbstractButton button, A action) {

        putValue(LOGIC_ACTION, action);

        // fill the ui action from the button
        setActionKey(action.getClass().getName());
        if (button != null) {

            setActionIcon(button.getIcon());
            setActionName(button.getText());
            setActionDescription(button.getToolTipText());
            setActionMnemonic(button.getMnemonic());
            setEnabled(button.isEnabled());

            // see https://forge.nuiton.org/issues/3525
            button.addPropertyChangeListener(evt -> setEnabled(button.isEnabled()));

        }
    }

    public void launchActionAndWait() {
        wait = true;
        actionPerformed(null);
        lock();
    }


    @Override
    public final void actionPerformed(final ActionEvent event) {

        t0 = TimeLog.getTime();

        if (log.isInfoEnabled()) {
            log.info("Task [" + getLogicAction().getClass().getSimpleName() + "] starting");
        }

        // prepare action
        boolean doAction;

        A action = getLogicAction();

        ApplicationUIContext actionContext = action.getContext();
        if (actionContext.isActionInProgress(this)) {
            if (log.isInfoEnabled()) {
                log.info("Task [" + getLogicAction().getClass().getSimpleName() + "] stopped: action already in progress");
            }
            return;
        }

        actionContext.setActionInProgress(this, true);

        // reset status message
        action.sendMessage("");

        try {
            doAction = action.prepareAction();
        } catch (Exception e) {
            action.releaseAction();
            actionContext.setActionInProgress(this, false);
            throw ApplicationActionException.propagateError(action, e);
        }

        if (doAction) {

            final ApplicationActionSwingWorker<A> worker =
                    new ApplicationActionSwingWorker<>(action);

            SwingUtilities.invokeLater(() -> {

                // make ui busy
                worker.updateBusyState(true);

            });

            if (log.isDebugEnabled()) {
                log.debug("Before execute of action " + action);
            }

            // perform and release action
            worker.execute();

            // wait until action is done
            waitingThread.execute(
                    () -> {

                        A action1 = getLogicAction();
                        try {
                            try {
                                worker.get();
                            } catch (ExecutionException | InterruptedException | CancellationException e) {
                                // don't care .
                            }
                            if (log.isDebugEnabled()) {
                                log.debug("After execute of action " + action1 + " (worker done? " + worker.isDone() + ")");
                            }

                            if (worker.isFailed()) {

                                throw ApplicationActionException.propagateError(action1, worker.getError());
                            }
                        } finally {
                            unlock();
                        }
                    }
            );

        } else {

            try {
                // release action
                action.releaseAction();

            } finally {
                unlock();
            }
        }
    }

    public void setActionIcon(Icon actionIcon) {
        putValue(SMALL_ICON, actionIcon);
        putValue(LARGE_ICON_KEY, actionIcon);
    }

    public void setActionKey(String actionKey) {
        putValue(ACTION_COMMAND_KEY, actionKey);
    }

    public void setActionName(String actionName) {
        putValue(NAME, actionName);
    }

    public void setActionDescription(String actionDescription) {
        putValue(SHORT_DESCRIPTION, actionDescription);
        getLogicAction().setActionDescription(actionDescription);
    }

    public void setActionMnemonic(int key) {
        putValue(MNEMONIC_KEY, key);
    }

    public A getLogicAction() {
        return (A) getValue(LOGIC_ACTION);
    }

    protected void lock() {
        if (wait) {
            synchronized (lock) {
                try {

                    lock.wait();

                } catch (InterruptedException e) {
                    throw ApplicationActionException.propagateError(getLogicAction(), e);
                } finally {
                    wait = false;
                }
            }
        }
    }

    protected void unlock() {
        TIME_LOG.log(t0, "Task [" + getLogicAction().getClass().getSimpleName() + "] End");
        if (wait) {
            synchronized (lock) {
                lock.notifyAll();
            }
        }
        getLogicAction().getContext().setActionInProgress(this, false);
    }
}
