/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.nuiton.guix.tags;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.nuiton.guix.css.CSSParser;
import org.nuiton.guix.css.CSSParserConstants;
import org.nuiton.guix.css.CSSParserTreeConstants;
import org.nuiton.guix.css.SimpleNode;
import org.nuiton.guix.model.Rule;
import org.nuiton.guix.model.Selector;
import org.nuiton.guix.model.StyleSheet;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

//~--- JDK imports ------------------------------------------------------------
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.nuiton.guix.css.ParseException;
/**
 * Handles the <code>style</code> tag.
 *
 * @author morin
 */
public class StyleHandler {

    /** log */
    protected static final Log log = LogFactory.getLog(TagManager.class);
    /** List of the selectors */
    private List<Selector> selectors = new ArrayList<Selector>();

    /**
     * Loads the content of a file.
     *
     * @param styleFile     the file to load
     * @return              the content of the file
     * @throws java.io.FileNotFoundException if styleFile does not exist
     * @throws java.io.IOException if an error occurs while reading styleFile
     */
    private String loadStyleFile(File styleFile) throws FileNotFoundException, IOException {
        StringWriter styleBuffer = new StringWriter();
        FileReader in = new FileReader(styleFile);
        char[] readBuffer = new char[2048];
        int c;

        while ((c = in.read(readBuffer)) > 0) {
            styleBuffer.write(readBuffer, 0, c);
        }

        return styleBuffer.toString();
    }

    /**
     * Load the file with the name of the class
     * and creates a <code>StyleSheet</code> with the content of the file.
     *
     * @param styleFile   the file containing the css code
     * @return              a <code>StyleSheet</code> containing the content of the file
     * @throws FileNotFoundException  if the file with the same name as the class
     * does not exist
     * @throws IOException if an error occurs while reading the file
     * with the same name as the class
     */
    public StyleSheet autoDetectStyleFile(File styleFile) throws IOException {
        selectors = new ArrayList<Selector>();
        return processStylesheet(loadStyleFile(styleFile));
    }

    /**
     * Load the file specified by the <code>source</code> attribute
     * or the code between the style tags
     * and creates a <code>StyleSheet</code> with this content.
     *
     * @param xpp       the parser referencing the style tag
     * @param styleFile   the file containing the css code
     * @return          a <code>StyleSheet</code> containing the content of the file
     * @throws java.io.IOException if an error occurs while reading
     * the file specified by the <code>source</code> attribute
     * @throws org.xmlpull.v1.XmlPullParserException if an error occurs
     * while parsing the guix file
     */
    public StyleSheet compileStyle(XmlPullParser xpp, File styleFile) throws IOException, XmlPullParserException {
        StyleSheet result = null;
        StringBuffer style = new StringBuffer();
        selectors = new ArrayList<Selector>();

        if ((styleFile != null) && styleFile.exists()) {
            style.append(loadStyleFile(styleFile));
        }

        xpp.nextToken();

        do {
            switch (xpp.getEventType()) {
                case XmlPullParser.START_TAG:
                    if (log.isWarnEnabled()) {
                        log.warn("<style> tag may not contain child elements: " + xpp.getName());
                    }

                    break;

                case XmlPullParser.TEXT:
                    style.append(xpp.getText());
                    xpp.nextToken();
                    break;    // fall through

                case XmlPullParser.CDSECT:
                    style.append(xpp.getText());
                    xpp.nextToken();

                    break;

                default :
                    break;
            }
        } while ((xpp.getEventType() != XmlPullParser.END_TAG) && (xpp.getEventType() != XmlPullParser.END_DOCUMENT) && (!"style".equals(xpp.getName())));

        // analyze the content of the CSS
        result = processStylesheet(style.toString());

        return result;
    }

    /**
     * Analyses a selector node
     *
     * @param selector the selector node to analyse
     * @return a Selector with the class, styleclass, pseudoclass and/or id
     *          which rules will be applied to
     */
    protected Selector processSelector(SimpleNode selector) {
        // chack if it is a selector
        if (selector.getId() != CSSParserTreeConstants.JJTSELECTOR) {
            throw new IllegalArgumentException("argument node is not a Selector");
        }

        String javaClassName = null;
        String styleClass = null;
        String pseudoClass = null;
        String id = null;

        for (int i = 0; i < selector.jjtGetNumChildren(); i++) {
            SimpleNode child = selector.getChild(i);

            switch (child.getId()) {

                // class
                case CSSParserTreeConstants.JJTJAVACLASS:
                    if (!child.getText().trim().equals("*")) {
                        javaClassName = child.getText();
                    }

                    break;

                // styleclass
                case CSSParserTreeConstants.JJTCLASS:
                    styleClass = child.getText().substring(1);

                    break;

                // pseudoclass
                case CSSParserTreeConstants.JJTPSEUDOCLASS:
                    pseudoClass = child.getText().substring(1);

                    break;

                // id
                case CSSParserTreeConstants.JJTID:
                    id = child.getText().substring(1);

                    break;

                // error if other
                default:
                    throw new IllegalStateException("unexpected child of Selector node, type=" + child.getId());
            }
        }

        return new Selector(javaClassName, styleClass, pseudoClass, id);
    }

    /**
     * Analyses a CSS rule
     *
     * @param ruleNode the CSSParser node containing the rule to analyse
     * @return a Rule containing the properties and the selectors of the rule
     */
    protected Rule processRule(SimpleNode ruleNode) {

        // checks if it is a rule
        if (ruleNode.getId() != CSSParserTreeConstants.JJTRULE) {
            throw new IllegalArgumentException("argument node is not a Rule");
        }

        SimpleNode selectorsNode = ruleNode.getChild(0);

        assert selectorsNode.getId() == CSSParserTreeConstants.JJTSELECTORS : "expected node to be of type Selectors";

        List<Selector> ruleSelectors = new ArrayList<Selector>();

        // analyses each selector
        for (int i = 0; i < selectorsNode.jjtGetNumChildren(); i++) {
            SimpleNode selectorNode = selectorsNode.getChild(i);
            Selector selector = processSelector(selectorNode);

            ruleSelectors.add(selector);

            // records the selector if it has not been recorded yet
            if (!selectorRecorded(selector)) {
                selectors.add(selector);
            }
        }

        Map<String, String> properties = new HashMap<String, String>();

        // records the properties
        for (int i = 1; i < ruleNode.jjtGetNumChildren(); i++) {
            SimpleNode declarationNode = ruleNode.getChild(i);

            if (declarationNode.getId() == CSSParserTreeConstants.JJTDECLARATION) {
                String key = declarationNode.getChild(0).getText();
                SimpleNode valueNode = declarationNode.getChild(1);
                String value = valueNode.getText();

                if (valueNode.firstToken.kind == CSSParserConstants.STRING) {
                    value = value.substring(1, value.length() - 1);
                }

                properties.put(key, value);
            }
        }

        // creates the new rule
        Rule rule = new Rule();

        rule.setSelectors(ruleSelectors);

        for (Selector selector : ruleSelectors) {
            selector.getRules().add(rule);
        }

        rule.setProperties(properties);

        return rule;
    }

    /**
     * Analyses the CSS
     *
     * @param stylesheetText the content of the CSS file or the style tag
     * @return the StyleSheet containing the rules of the CSS file or the style tag
     */
    protected StyleSheet processStylesheet(String stylesheetText) throws IOException {
        // parses the CSS
        CSSParser p = new CSSParser(new StringReader(stylesheetText));

        try {
            SimpleNode node = p.Stylesheet();
            List<Rule> rules = new ArrayList<Rule>();

            // analyses each rule
            for (int i = 0; i < node.jjtGetNumChildren(); i++) {
                SimpleNode ruleNode = node.getChild(i);
                Rule rule = processRule(ruleNode);
                    rules.add(rule);

            }

            StyleSheet styleSheet = new StyleSheet();

            styleSheet.setRules(rules);
            styleSheet.setSelectors(selectors);

            return styleSheet;
            
        } catch(ParseException eee) {
            if(log.isWarnEnabled()) {
                log.warn(eee);
            }
        }
        return null;
    }

    /**
     * Checks if a new Selector has already been recorded or not
     * We chack it to add each selector only once in the StyleSheet
     * @param newSelector the Selector to check
     * @return true if the Selector has already been recorded
     */
    private boolean selectorRecorded(Selector newSelector) {
        for (Selector selector : selectors) {
            boolean areJavaClassNamesNull = (newSelector.getJavaClassName() == null) && (selector.getJavaClassName() == null);
            boolean areJavaClassNamesNotNull = (newSelector.getJavaClassName() != null) && (selector.getJavaClassName() != null);
            boolean areStyleClassesNull = (newSelector.getStyleClass() == null) && (selector.getStyleClass() == null);
            boolean areStyleClassesNotNull = (newSelector.getStyleClass() != null) && (selector.getStyleClass() != null);
            boolean arePseudoClassesNull = (newSelector.getPseudoClass() == null) && (selector.getPseudoClass() == null);
            boolean arePseudoClassesNotNull = (newSelector.getPseudoClass() != null) && (selector.getPseudoClass() != null);
            boolean areIdsNull = (newSelector.getId() == null) && (selector.getId() == null);
            boolean areIdsNotNull = (newSelector.getId() != null) && (selector.getId() != null);

            if ((areJavaClassNamesNull || (areJavaClassNamesNotNull && newSelector.getJavaClassName().equals(selector.getJavaClassName()))) && (areStyleClassesNull || (areStyleClassesNotNull && newSelector.getStyleClass().equals(selector.getStyleClass()))) && (arePseudoClassesNull || (arePseudoClassesNotNull && newSelector.getPseudoClass().equals(selector.getPseudoClass()))) && (areIdsNull || (areIdsNotNull && newSelector.getId().equals(selector.getId())))) {
                return true;
            }
        }

        return false;
    }
}

