/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.nuiton.guix.tags;

//~--- non-JDK imports --------------------------------------------------------
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.guix.generator.JavaArgument;
import org.nuiton.guix.generator.JavaField;
import org.nuiton.guix.generator.JavaMethod;
import org.nuiton.guix.parser.JavaParser;
import org.nuiton.guix.parser.JavaParserTreeConstants;
import org.nuiton.guix.parser.SimpleNode;

/**
 * Handles the <code>script</code> tag and its contents.
 *
 * @author morin
 */
public class ScriptHandler {

    private static final Log log = LogFactory.getLog(ScriptHandler.class);

    private static final String SWING_POSTPROCESS_FIRST_TOKEN_PATTERN = "%1$s old%2$s = %2$s;\n%2$s";
    private static final String SWING_POSTPROCESS_LAST_TOKEN_PATTERN = "%1$s;\nfirePropertyChange(\"%2$s\", old%2$s, %2$s)";
    /**
     * Loads the content of a script file
     *
     * @param scriptFile    the file to load
     * @return              the content of the script file
     * @throws java.io.IOException if an error occurs while reading scriptFile
     */
    public String loadScriptFile(File scriptFile) throws IOException {
        StringWriter scriptBuffer = new StringWriter();
        FileReader in = new FileReader(scriptFile);
        char[] readBuffer = new char[2048];
        int c;

        while ((c = in.read(readBuffer)) > 0) {
            scriptBuffer.write(readBuffer, 0, c);
        }

        return scriptBuffer.toString();
    }

    /**
     * Parse the <code>script</code> tag.
     *
     * @param xpp       the parser referencing the style tag
     * @param scriptFile   the file containing the code to compile
     * @return          the content of the file specified by the <code>source</code> attribute
     *                  or the script between the script tags
     * @throws IOException
     * @throws XmlPullParserException
     */
    public String compileScript(XmlPullParser xpp, File scriptFile) throws IOException, XmlPullParserException {
        StringBuffer script = new StringBuffer();

        if ((scriptFile != null) && scriptFile.exists()) {
            script.append(loadScriptFile(scriptFile));
        }
        
        xpp.nextToken();

        script.append(xpp.getText());

        while ((xpp.getEventType() != XmlPullParser.END_TAG) || (!xpp.getName().equals("script"))) {
            xpp.nextToken();
        }

        return script.toString();
    }

    /**
     * The different parts of the script
     */
    public enum ScriptPart {

        IMPORTS, BODYCODE, INITIALIZERS, METHODS, FIELDS
    }

    /**
     * Decompose the script into the different parts
     * @param script the script to decompose
     * @return a map containing the imports, the bodycode, the initializers and the methods of the script
     */
    public static Map<ScriptPart, Object> decomposeScript(String script) {
        Map<ScriptPart, Object> map = new HashMap<ScriptPart, Object>();

        List<String> imports = new ArrayList<String>();
        StringBuffer bodyCode = new StringBuffer();
        List<JavaMethod> methods = new ArrayList<JavaMethod>();
        List<JavaField> fields = new ArrayList<JavaField>();
        StringBuffer initializers = new StringBuffer();

        JavaParser p = new JavaParser(new StringReader(script));
        //start parsing
        while (!p.Line()) {
            SimpleNode node = p.popNode();
            if (node != null) {
                //analyse the node
                Map<ScriptPart, Object> ssn = analyseScriptNode(node);
                //get the result
                imports.addAll((List<String>) ssn.get(ScriptPart.IMPORTS));
                bodyCode.append(ssn.get(ScriptPart.BODYCODE));
                initializers.append(ssn.get(ScriptPart.INITIALIZERS));
                methods.addAll((List<JavaMethod>)ssn.get(ScriptPart.METHODS));
                fields.addAll((List<JavaField>)ssn.get(ScriptPart.FIELDS));
            }
        }
        //add results
        map.put(ScriptPart.IMPORTS, imports);
        map.put(ScriptPart.BODYCODE, bodyCode.toString());
        map.put(ScriptPart.INITIALIZERS, initializers.toString());
        map.put(ScriptPart.METHODS, methods);
        map.put(ScriptPart.FIELDS, fields);
        return map;
    }

    public static String postProcessMethodBody(JavaMethod method, JavaField[] fields) {
        JavaParser p = new JavaParser(new StringReader(method.getBodyCode()));
        StringBuffer newMethodBody = new StringBuffer();
        //start parsing
        while (!p.Line()) {
            SimpleNode node = p.popNode();
            if (node != null) {
                postProcessContents(node, Modifier.isStatic(method.getModifiers()), fields);
                newMethodBody.append(node.getText());
            }
        }
        return newMethodBody.toString();
    }

    public static String postProcessInitializers(String initializers, JavaField[] fields) {
        JavaParser p = new JavaParser(new StringReader(initializers));
        StringBuffer newInitializers = new StringBuffer();
        //start parsing
        while (!p.Line()) {
            SimpleNode node = p.popNode();
            if (node != null) {
                postProcessContents(node, false, fields);
                newInitializers.append(node.getText());
            }
        }
        return newInitializers.toString();
    }

    private static void postProcessContents(SimpleNode node, boolean staticContext, JavaField[] fields) {
        // identify static methods and initializers -- we can't fire events statically
        if (node.getId() == JavaParserTreeConstants.JJTMETHODDECLARATION) {
            if (node.getParent().getChild(0).getText().indexOf("static") != -1) {
                staticContext = true;
            }
        }
        else if (node.getId() == JavaParserTreeConstants.JJTINITIALIZER) {
            if (node.getText().trim().startsWith("static")) {
                staticContext = true;
            }
        }

        int count = node.jjtGetNumChildren();
        for (int i = 0; i < count; i++) {
            postProcessContents(node.getChild(i), staticContext, fields);
        }

        int id = node.getId();
        if (!staticContext) {
            String lhs = null;
            if (id == JavaParserTreeConstants.JJTASSIGNMENTEXPRESSION || (id == JavaParserTreeConstants.JJTPOSTFIXEXPRESSION && node.jjtGetNumChildren() == 2)) {
                lhs = ((SimpleNode) node.jjtGetChild(0)).getText().trim();
            }
            else
            if (id == JavaParserTreeConstants.JJTPREINCREMENTEXPRESSION || id == JavaParserTreeConstants.JJTPREDECREMENTEXPRESSION) {
                lhs = ((SimpleNode) node.jjtGetChild(0)).getText().trim();
            }
            if (lhs != null) {
                for (JavaField field : fields) {
                    if (field.getName().equals(lhs)) {
                        //lhs.substring(lhs.lastIndexOf(".") + 1);
                        node.firstToken.image = String.format(SWING_POSTPROCESS_FIRST_TOKEN_PATTERN, field.getType(), node.firstToken.image);
                        node.lastToken.image = String.format(SWING_POSTPROCESS_LAST_TOKEN_PATTERN, node.lastToken.image, lhs);
                    }
                }
            }
        }
    }

    /**
     * Analyse a node of the parsed script
     * @param node the node to analyse
     * @return a map containing the imports, the bodycode, the initializers and the methods of the node
     */
    private static Map<ScriptPart, Object> analyseScriptNode(SimpleNode node) {
        List<String> imports = new ArrayList<String>();
        List<JavaMethod> methods = new ArrayList<JavaMethod>();
        List<JavaField> fields = new ArrayList<JavaField>();
        StringBuffer bodyCode = new StringBuffer();
        StringBuffer initializers = new StringBuffer();

        int nodeType = getLineType(node);

        //if the node declares imports
        if (nodeType == JavaParserTreeConstants.JJTIMPORTDECLARATION) {
            String text = node.getChild(0).getText().trim();
            if (text.startsWith("import")) {
                text = text.substring("import".length()).trim();
            }
            if (text.endsWith(";")) {
                text = text.substring(0, text.length() - 1);
            }
            //add the class to import to the imports
            imports.add(text);
        }
        //if the node declares an enum
        else if (nodeType == JavaParserTreeConstants.JJTENUMDECLARATION) {
            //add the enum to the bodycode
            bodyCode.append(node.getChild(0).getText().trim()).append("\n");
        }
        //if the node declares a method
        else if (nodeType == JavaParserTreeConstants.JJTMETHODDECLARATION) {
            //information needed to create the JavaMethod
            String returnType = null;
            String name = null;
            List<JavaArgument> parameterTypes = new ArrayList<JavaArgument>();
            List<String> exceptionTypes = new ArrayList<String>();
            String block = null;
            //get the javadoc of the method
            String javaDoc = getJavaDoc(node.getChild(0).getChild(0));
            //get the modifier of the method
            int modifier = getModifier(node.getChild(0).getChild(0));
            SimpleNode methodDeclaration = node.getChild(0).getChild(1);            
            assert methodDeclaration.getId() == JavaParserTreeConstants.JJTMETHODDECLARATION;
            //analyse the declaration
            for (int i = 0; i < methodDeclaration.jjtGetNumChildren(); i++) {
                SimpleNode child = methodDeclaration.getChild(i);
                int type = child.getId();
                //if the child is the returntype
                if (type == JavaParserTreeConstants.JJTRESULTTYPE) {                    
                    returnType = child.getText().trim();
                }
                //if the child is the declarator
                else if (type == JavaParserTreeConstants.JJTMETHODDECLARATOR) {
                    //the name is the first token of the declarator
                    name = child.firstToken.image.trim();
                    //the parameters are in the first child
                    SimpleNode formalParameters = child.getChild(0);
                    assert formalParameters.getId() == JavaParserTreeConstants.JJTFORMALPARAMETERS;
                    for (int j = 0; j < formalParameters.jjtGetNumChildren(); j++) {
                        SimpleNode parameter = formalParameters.getChild(j);
                        String parameterType = parameter.getChild(1).getText().trim().replaceAll("\\.\\.\\.", "[]");
                        String parameterName = parameter.getChild(2).getText().trim();
                        parameterTypes.add(new JavaArgument(parameterType, parameterName));
                    }                    
                }
                //if the child contains the exceptions
                else if (type == JavaParserTreeConstants.JJTNAMELIST) {
                    for (int j = 0; j < child.jjtGetNumChildren(); j++) {
                        SimpleNode exception = child.getChild(j);
                        String exceptionType = exception.getText().trim();
                        exceptionTypes.add(exceptionType);
                    }
                }
                //if the child contains the code of the method
                else if (type == JavaParserTreeConstants.JJTBLOCK) {
                    //removes the braces
                    Pattern p = Pattern.compile("[\\s]*\\{(.*)\\}[\\s]*", Pattern.DOTALL);
                    Matcher m = p.matcher(child.getText());
                    block = m.matches() ? m.group(1) : null;
                }
            }
            //creates the JavaMethod
            JavaMethod method = new JavaMethod(modifier, returnType, name, parameterTypes.isEmpty() ? null : parameterTypes.toArray(new JavaArgument[parameterTypes.size()]),
                    exceptionTypes.toArray(new String[exceptionTypes.size()]), block, javaDoc);
            methods.add(method);

        }
        //if the node declares an inner class or an inner interface
        else if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION ||
                nodeType == JavaParserTreeConstants.JJTINITIALIZER) {
            String str = node.getText().trim();
            if (!str.endsWith(";")) {
                str += ";";
            }
            //add the declaration to the bodycode
            bodyCode.append(str).append("\n");
        }
        //if the node declares a constructor
        else if (nodeType == JavaParserTreeConstants.JJTCONSTRUCTORDECLARATION) {
            bodyCode.append(processConstructor(node.getChild(0).getChild(0).getText(), node.getChild(0).getChild(1)));
        }
        //if the node declares a field or a local variable
        else if (nodeType == JavaParserTreeConstants.JJTLOCALVARIABLEDECLARATION || nodeType == JavaParserTreeConstants.JJTFIELDDECLARATION) {
            // the "local" variable declarations in this expression aren't actually local -- they are flagged local
            // just because there isn't an enclosing class scope visible to the parser.  "Real" local variable
            // declarations won't show up here, because they will be buried inside of methods.
            SimpleNode fieldDeclaration = node.getChild(0).getChild(0);
            String fieldClass = null, fieldName = null, fieldInit = null;
            int fieldModifier = 4; //4 is Modifier.PROTECTED

            for (int i = 0; i < fieldDeclaration.jjtGetNumChildren(); i++) {
                SimpleNode child = fieldDeclaration.getChild(i);
                int type = child.getId();                
                if(type == JavaParserTreeConstants.JJTMODIFIERS) {
                    fieldModifier = getModifier(child);
                }
                else if(type == JavaParserTreeConstants.JJTVARIABLEDECLARATOR) {
                    fieldName = child.getChild(0).getText().trim();
                    if(child.jjtGetNumChildren() > 1) {
                        fieldInit = child.getChild(1).getText().trim();
                    }
                }
                else if(type == JavaParserTreeConstants.JJTTYPE) {
                    fieldClass = child.getChild(0).getText().trim();
                }
            }
            JavaField field;
            if(fieldInit != null) {
                initializers.append(fieldName).append(" = ").append(fieldInit).append(";\n");
            }
            field = new JavaField(fieldModifier, fieldClass, fieldName, null, TagManager.getGuixClassHandler(fieldClass));
            
            fields.add(field);
        }
        //if the node initialize a field
        else {
            String text = node.getText().trim();
            if (text.length() > 0) {
                if (!text.endsWith(";")) {
                    text += ";";
                }
                //add the text to the initializers
                initializers.append(text);
            }
        }

        //creates the result
        Map<ScriptPart, Object> result = new HashMap<ScriptPart, Object>();
        result.put(ScriptPart.IMPORTS, imports);
        result.put(ScriptPart.BODYCODE, bodyCode.toString());
        result.put(ScriptPart.INITIALIZERS, initializers.toString());
        result.put(ScriptPart.METHODS, methods);
        result.put(ScriptPart.FIELDS, fields);
        return result;
    }

    /**
     * Examines a Line to determine its real type.  As all tokens returned by the parser are Lines, and
     * they are just a tiny wrapper around the real node, this method strips off the wrapper layers to identify
     * the real type of a node.
     *
     * @param line line to scan
     * @return the line type
     */
    private static int getLineType(SimpleNode line) {
        if (line.jjtGetNumChildren() == 1) {
            SimpleNode node = line.getChild(0);
            if (node.getId() == JavaParserTreeConstants.JJTBLOCKSTATEMENT) {
                if (node.jjtGetNumChildren() == 1) {
                    return node.getChild(0).getId();
                }
            }
            else if (node.getId() == JavaParserTreeConstants.JJTCLASSORINTERFACEBODYDECLARATION) {
                int id = node.getChild(0).getId();
                if (id == JavaParserTreeConstants.JJTMODIFIERS) {
                    return node.getChild(1).getId();
                }
                if (id == JavaParserTreeConstants.JJTINITIALIZER) {
                    return id;
                }
            }
            return node.getId();
        }
        return JavaParserTreeConstants.JJTLINE; // generic value implying that it's okay to put into the initializer block
    }

    /**
     * Analyses a node which declares a constructor
     *
     * @param modifiers
     * @param node
     * @return
     */
    private static String processConstructor(String modifiers, SimpleNode node) {
        //FIXME
        assert node.getId() == JavaParserTreeConstants.JJTCONSTRUCTORDECLARATION : "expected node to be ConstructorDeclaration, found " + JavaParserTreeConstants.jjtNodeName[node.getId()] + " instead";
        assert node.getChild(0).getId() == JavaParserTreeConstants.JJTFORMALPARAMETERS : "expected node 0 to be FormalParameters, found " + JavaParserTreeConstants.jjtNodeName[node.getChild(1).getId()] + " instead";
        String code = "";
        if (node.getChild(0).jjtGetNumChildren() == 0) {
            if (log.isErrorEnabled()) {
                log.error("The default no-argument constructor may not be redefined");
            }
        }
        else {
            SimpleNode explicitConstructorInvocation = findExplicitConstructorInvocation(node);
            if (explicitConstructorInvocation == null || explicitConstructorInvocation.getText().trim().startsWith("super(")) {
                code = "initialize();\n";
                if (explicitConstructorInvocation == null) {
                    node.getChild(1).firstToken.image = node.getChild(1).firstToken.image;
                }
                else {
                    explicitConstructorInvocation.lastToken.image += code;
                }
            }
        }

        return modifiers + " " + node.getText().substring(0, node.getText().length() - 1) + code + "}";
    }

    /**
     *
     */
    private static SimpleNode findExplicitConstructorInvocation(SimpleNode parent) {
        if (parent.getId() == JavaParserTreeConstants.JJTEXPLICITCONSTRUCTORINVOCATION) {
            return parent;
        }

        int count = parent.jjtGetNumChildren();
        for (int i = 0; i < count; i++) {
            SimpleNode result = findExplicitConstructorInvocation(parent.getChild(i));
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    /**
     * Extracts the javadoc of a SimpleNode
     *
     * @param node the node which contains the javaDoc
     * @return the javadoc
     */
    private static String getJavaDoc(SimpleNode node) {
        Pattern p = Pattern.compile(".*/\\*\\*(.*)\\*/.*", Pattern.DOTALL);
        Matcher m = p.matcher(node.getText());
        return m.matches() ? m.group(1).replace("*", "") : null;
    }

    /**
     * Extracts the java.lang.reflect.Modifier from a SimpleNode
     *
     * @param node the node which contains the modifier
     * @return the modifier of the node
     */
    private static int getModifier(SimpleNode node) {
        int result = 4;
        //remove the javadoc from the node text
        Pattern p = Pattern.compile(".*(/\\*\\*(.*)\\*/).*", Pattern.DOTALL);
        Matcher m = p.matcher(node.getText());
        String modifier = m.matches() ? node.getText().replace(m.group(1), "") : node.getText();
        
        if (modifier.contains("public")) {
            result = result == 4 ? Modifier.PUBLIC : result | Modifier.PUBLIC;
        }
        if (modifier.contains("private")) {
            result = result == 4 ? Modifier.PRIVATE : result | Modifier.PRIVATE;
        }
        if (modifier.contains("protected")) {
            result = result == 4 ? Modifier.PROTECTED : result | Modifier.PROTECTED;
        }
        if (modifier.contains("static")) {
            result = result == 4 ? Modifier.STATIC : result | Modifier.STATIC;
        }
        if (modifier.contains("final")) {
            result = result == 4 ? Modifier.FINAL : result | Modifier.FINAL;
        }
        return result;
    }
}

