/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.nuiton.guix.generator;

import org.nuiton.guix.generator.JavaMethod.MethodOrder;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A Java source file being generated for output.  Once the class is completely initialized, use the
 * {@link #toString} method to generate source code for it.
 *
 * @author kevin
 */
public class JavaFile {

    protected static final int CLASS = 0;
    protected static final int ABSTRACT_CLASS = 1;
    protected static final int INTERFACE = 2;

    protected static final String GETTER_PATTERN = "return %1$s;";
    protected static final String BOOLEAN_GETTER_PATTERN = "return %1$s !=null && %1$s;";
    protected static final String SETTER_PATTERN = "this.%2$s = newValue;";

    private int modifiers;
    private int classType;
    private String packageName;
    private String className;
    private List<String> imports = new ArrayList<String>();
    private List<JavaField> fields = new ArrayList<JavaField>();
    private List<JavaField> inheritedFields = new ArrayList<JavaField>();
    private List<JavaMethod> methods = new ArrayList<JavaMethod>();
    private List<JavaMethod> inheritedMethods = new ArrayList<JavaMethod>();
    private List<JavaFile> innerClasses = new ArrayList<JavaFile>();
    private String superClass;
    private List<String> interfaces;
    private StringBuffer rawBodyCode = new StringBuffer();
    private boolean superclassIsGuixObject = false;
    private String genericType;
    private String superGenericType;
    private String javaDoc;


    public JavaFile() {
    }


    public JavaFile(int modifiers, int classType, String packageName, String className, String superClass, String javaDoc) {
        this(modifiers, classType, packageName, className, superClass, null, javaDoc);
    }


    public JavaFile(int modifiers, int classType, String packageName, String className, String superClass, List<String> interfaces, String javaDoc) {
        this.classType = classType;
        this.packageName = packageName;
        this.modifiers = modifiers;
        this.className = className;
        this.superClass = superClass;
        this.interfaces = interfaces;
        this.javaDoc = javaDoc;
    }


    public void addImport(String importString) {
        imports.add(importString);
    }

    public void addImport(Class importString) {
        imports.add(importString.getName());
    }


    public String[] getImports() {
        return imports.toArray(new String[imports.size()]);
    }

    public int getModifiers() {
        return modifiers;
    }


    public void setModifiers(int modifiers) {
        this.modifiers = modifiers;
    }

    public String getPackageName() {
        return packageName;
    }

    public void setPackageName(String packageName) {
        this.packageName = packageName;
    }

    public String getClassName() {
        return className;
    }


    public void setClassName(String className) {
        this.className = className;
    }


    public String getSuperClass() {
        return superClass;
    }


    public void setSuperClass(String superClass) {
        this.superClass = superClass;
    }


    public List<String> getInterfaces() {
        if (interfaces == null) {
            interfaces = new ArrayList<String>();
        }
        return interfaces;
    }


    public void setInterfaces(List<String> interfaces) {
        this.interfaces = interfaces;
    }

    public void setGenericType(String genericType) {
        this.genericType = genericType;
    }

    public void addMethod(JavaMethod method) {
        methods.add(method);
    }


    public JavaMethod[] getMethods() {
        return methods.toArray(new JavaMethod[methods.size()]);
    }

    public JavaMethod getMethod(String methodName, JavaArgument[] args) {
        for(JavaMethod method : methods) {
            if(method.getName().equals(methodName)) {
                if(method.getArguments() == null && args == null) {
                    return method;
                }
                else if (method.getArguments() != null && args != null
                        && method.getArguments().length == args.length) {
                    boolean samesame = true;
                    for(int i = 0 ; i < args.length ; i++) {
                        if(!args[i].getName().equals(method.getArguments()[i].getName())
                                || !args[i].getType().equals(method.getArguments()[i].getType())) {
                            samesame = false;
                            break;
                        }
                    }
                    if(samesame) {
                        return method;
                    }
                }
            }
        }
        for(JavaMethod method : inheritedMethods) {
            if(method.getName().equals(methodName)) {
                if(method.getArguments() == null && args == null) {
                    return method;
                }
                else if (method.getArguments() != null && args != null
                        && method.getArguments().length == args.length) {
                    boolean samesame = true;
                    for(int i = 0 ; i < args.length ; i++) {
                        if(!args[i].getName().equals(method.getArguments()[i].getName())
                                || !args[i].getType().equals(method.getArguments()[i].getType())) {
                            samesame = false;
                            break;
                        }
                    }
                    if(samesame) {
                        return method;
                    }
                }
            }
        }
        return null;
    }

    public void addInheritedMethod(JavaMethod method) {
        inheritedMethods.add(method);
    }


    public JavaMethod[] getInheritedMethods() {
        return inheritedMethods.toArray(new JavaMethod[inheritedMethods.size()]);
    }

    public JavaMethod getInheritedMethod(String methodName, JavaArgument[] args) {
        for(JavaMethod method : inheritedMethods) {
            if(method.getName().equals(methodName)) {
                if(method.getArguments() == null && args == null) {
                    return method;
                }
                else if (method.getArguments() != null && args != null
                        && method.getArguments().length == args.length) {
                    boolean samesame = true;
                    for(int i = 0 ; i < args.length ; i++) {
                        if(!args[i].getName().equals(method.getArguments()[i].getName())
                                || !args[i].getType().equals(method.getArguments()[i].getType())) {
                            samesame = false;
                            break;
                        }
                    }
                    if(samesame) {
                        return method;
                    }
                }
            }
        }
        return null;
    }

    public JavaMethod[] getAllMethods() {
        List<JavaMethod> allMethods = new ArrayList<JavaMethod>();
        allMethods.addAll(methods);
        allMethods.addAll(inheritedMethods);
        return allMethods.toArray(new JavaMethod[allMethods.size()]);
    }

    public void addField(JavaField field) {
        addField(field, false);
    }

    public void addField(JavaField field, boolean javaBean) {
        addSimpleField(field);
        String id = field.getName();
        String capitalizedName = (id != null && id.length()>0)? Character.toUpperCase(id.charAt(0))+id.substring(1) : id;
        // add getter file
        String content = (classType == INTERFACE) ? null : String.format(GETTER_PATTERN, id);
        addMethod(new JavaMethod(Modifier.PUBLIC, field.getType(),
                "get" + capitalizedName, null, null, content, null));

        if (javaBean) {
            // add full javabean support
            if (Boolean.class.getName().equals(field.getType())) {
                content = (classType == INTERFACE) ? null : String.format(BOOLEAN_GETTER_PATTERN, id);
                addMethod(new JavaMethod(Modifier.PUBLIC, field.getType(), "is" + capitalizedName, null, null, content, null));
            }
            content = (classType == INTERFACE) ? null : String.format(SETTER_PATTERN, field.getType(), id);
            JavaArgument arg = new JavaArgument(field.getType(), "newValue");
            addMethod(new JavaMethod(Modifier.PUBLIC, "void", "set" + capitalizedName, new JavaArgument[]{arg}, null, content, null));
        }
    }

    public void addSimpleField(JavaField field) {
        fields.add(field);
    }

    public void addInheritedField(JavaField field) {
        inheritedFields.add(field);
    }


    public JavaField[] getFields() {
        return fields.toArray(new JavaField[fields.size()]);
    }

    public JavaField getField(String fieldName) {
        for(JavaField field : fields) {
            if(field.getName().equals(fieldName)) {
                return field;
            }
        }
        for(JavaField field : inheritedFields) {
            if(field.getName().equals(fieldName)) {
                return field;
            }
        }
        return null;
    }

    public JavaField[] getInheritedFields() {
        return inheritedFields.toArray(new JavaField[inheritedFields.size()]);
    }

    public JavaField getInheritedField(String fieldName) {
        for(JavaField field : inheritedFields) {
            if(field.getName().equals(fieldName)) {
                return field;
            }
        }
        return null;
    }

    public JavaField[] getAllFields() {
        List<JavaField> allFields = new ArrayList<JavaField>();
        allFields.addAll(fields);
        allFields.addAll(inheritedFields);
        return allFields.toArray(new JavaField[allFields.size()]);
    }

    public static String addIndentation(String source, String lineSeparator) {
        return indent(source, false, lineSeparator);
    }


    public static String setIndentation(String source, String lineSeparator) {
        return indent(source, true, lineSeparator);
    }


    public static String indent(String source, boolean trim, String lineSeparator) {
        if (trim) {
            source = source.trim();
        }
        char[] spaces = new char[1];
        Arrays.fill(spaces, '\t');
        StringBuffer result = new StringBuffer();
        String[] lines = source.split(System.getProperty("line.separator") + "|\n");
        for (int i = 0; i < lines.length; i++) {
            if (i > 0) {
                result.append(lineSeparator);
            }
            if(lines[i].equals(lines[i].trim()) && (lines[i].trim().startsWith("}") || lines[i].trim().endsWith("}"))) {
                spaces = new char[spaces.length - 1];
                Arrays.fill(spaces, '\t');
            }
            result.append(spaces);
            result.append(trim ? lines[i].trim() : lines[i]);
            if(lines[i].equals(lines[i].trim()) && lines[i].trim().endsWith("{")) {
                spaces = new char[spaces.length + 1];
                Arrays.fill(spaces, '\t');
            }
        }
        return result.toString();
    }


    public void addBodyCode(String bodyCode) {
        rawBodyCode.append(bodyCode);
    }


    public String getClassBody(String lineSeparator) {
        StringBuffer result = new StringBuffer("\t");
        if (classType != INTERFACE && fields.size() > 0) {
            java.util.Collections.sort(fields); // sort fields

            for (JavaField field : fields) {
                result.append(field.toString(lineSeparator));
            }

            result.append(lineSeparator);
        }

        if (rawBodyCode.length() > 0) {
            result.append("/* begin raw body code */\n");
            String s = rawBodyCode.toString();
            if (!s.startsWith(lineSeparator)) {
                result.append(lineSeparator);
            }
            result.append(s);
            result.append(lineSeparator);
            result.append("/* end raw body code */");
            result.append(lineSeparator).append(lineSeparator);
        }

        for (JavaFile innerClass : innerClasses) {
            result.append(innerClass.toString());
            result.append(lineSeparator).append(lineSeparator);
        }

        EnumMap<MethodOrder, List<JavaMethod>> map = JavaMethod.getSortedMethods(methods);
        for (Entry<MethodOrder, List<JavaMethod>> entry : map.entrySet()) {
            List<JavaMethod> list = entry.getValue();
            entry.getKey().generate(result, list, lineSeparator);
            list.clear();
        }
        map.clear();

        Pattern p = Pattern.compile("\\n");
        //System.out.println(result.toString());
        Matcher m = p.matcher(result.toString());
        return m.replaceAll("\n\t");

    }


    public String getClassDefinition(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        result.append(getModifiersText(modifiers))
                .append(classType == INTERFACE ? "interface " : "class ")
                .append(className.substring(className.lastIndexOf(".") + 1));
        if (genericType != null) {
            result.append('<').append(genericType).append('>');
        }
        if(superClass != null) {
            result.append(" extends ");
            result.append(superClass);
            if (superGenericType != null) {
                result.append('<').append(superGenericType).append('>');
            }
        }
        if (interfaces != null && !interfaces.isEmpty()) {
            result.append(" implements ").append(interfaces.get(0));
            for (int i = 1; i < interfaces.size(); i++) {
                result.append(", ").append(interfaces.get(i));
            }
        }
        result.append(" {");
        result.append(lineSeparator);
        result.append(getClassBody(lineSeparator));
        result.append("}");
        return result.toString();
    }


    public static String getModifiersText(int modifiers) {
        if (modifiers == 0) {
            return "";
        } else {
            return Modifier.toString(modifiers) + ' ';
        }
    }


    /**
     * Returns the Java source code for this class.
     *
     * @param lineSeparator line separator
     * @return a complete Java file for this class
     */
    public String toString(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        if (packageName != null && !packageName.equals("")) {
            result.append("package ").append(packageName).append(";");
            result.append(lineSeparator);
            result.append(lineSeparator);
        }

        if (imports.size() > 0) {
            for (String anImport : imports) {
                result.append("import ");
                result.append(anImport);
                result.append(';');
                result.append(lineSeparator);
            }
            result.append(lineSeparator);
        }
        
        if(javaDoc != null && !javaDoc.equals(""))
            result.append("/**\n * ")
                    .append(javaDoc.replace("\n", "\n * "))
                    .append("\n */")
                    .append(lineSeparator);

        result.append(getClassDefinition(lineSeparator));
        return result.toString();
    }

    public void addInterface(String canonicalName) {
        if (interfaces == null || !interfaces.contains(canonicalName)) {
            getInterfaces().add(canonicalName);
        }
    }

    public void addInterfaces(String[] canonicalNames) {
        if (canonicalNames == null) {
            return;
        }
        for (String canonicalName : canonicalNames) {
            if (interfaces == null || !interfaces.contains(canonicalName)) {
                getInterfaces().add(canonicalName);
            }
        }
    }

    public boolean isSuperclassIsGuixObject() {
        return superclassIsGuixObject;
    }

    public void setSuperclassIsGuixObject(boolean superclassIsGuixObject) {
        this.superclassIsGuixObject = superclassIsGuixObject;
    }

    public void setSuperGenericType(String superGenericType) {
        this.superGenericType = superGenericType;
    }
}