package org.nuiton.eugene.plugin;

import org.nuiton.util.PluginIOContext;
import java.io.File;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;

/**
 * La classe de base pour definir des mojos dans eugene.
 * 
 * @author chemit
 * @since 1.0.0-rc-8
 */
public abstract class EugeneAbstractMojo extends AbstractMojo {

    /**
     * Maven project.
     *
     * @description Dépendance du projet.
     * @parameter default-value="${project}"
     * @readonly
     * @since 1.0.0-rc-8
     */
    protected MavenProject project;
    /**
     * @description le settings (pour obtenir le mode offline)
     * @parameter default-value="${settings}"
     * @readonly
     * @since 1.0.0-rc-4
     */
    protected Settings settings;
    /**
     * Ecrase les fichiers générés.
     *
     * @parameter expression="${eugene.overwrite}" default-value="false"
     * @since 0.50
     */
    protected boolean overwrite;
    /**
     * Pour activer le mode verbeux.
     *
     * @parameter expression="${eugene.verbose}" default-value="${maven.verbose}"
     * @since 1.0.0-rc-8
     */
    protected boolean verbose;
    /**
     * Encoding to be used for generation of files.
     *
     * @parameter expression="${eugene.encoding}" default-value="${project.build.sourceEncoding}"
     * @since 0.60
     */
    protected String encoding;
    /**
     * A flag to mark the mojo to be used in a test phase. This will permits to add generated sources in test compile roots.
     *
     * @parameter expression="${eugene.testPhase}" default-value="false"
     * @since 0.64
     */
    protected boolean testPhase;

    /**
     *
     * Note : l'objet peut etre {@null} si la configuration ne definit
     * pas la propriété associées dans l'implentation.
     * <p/>
     *
     * Dans tous les cas, une fois la methode {@link #initResources()} appele,
     * l'objet devrait ne plus être null!
     * 
     * @return l'io du plugin ou {@code null} si non initialisé
     *
     */
    protected abstract PluginIOContext getResources();

    /**
     * Initialiser les io du plugins.
     *
     * @return l'io intialisé (avec les valeurs par défaut, là où rien n'a
     * été renseigné).
     */
    protected abstract PluginIOContext initResources();

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        PluginIOContext p = initResources();

        if ((p.getInputs() == null || p.getInputs().length == 0)) {
            throw new MojoExecutionException("no input defined");
        }
        if (p.getOutput() == null) {
            throw new MojoExecutionException("no output defined");
        }

        p.getOutput().mkdirs();
    }

    protected PluginIOContext initResources(File defaultIn, File defaultOut, File defaultTestIn, File defaultTestOut) {

        PluginIOContext resources = getResources();
        if (resources == null) {
            resources = new PluginIOContext();
        }

        // adding default inputs only if not consumed
        if (resources.getInputs() == null) {
            if (testPhase) {
                resources.setInput(defaultTestIn);
            } else {
                resources.setInput(defaultIn);
            }
            getLog().info(" using default in : " + Arrays.asList(resources.getInputs()));
        } else {
            getLog().info(" in               : " + Arrays.asList(resources.getInputs()));
        }

        if (resources.getOutput() == null) {
            if (testPhase) {
                resources.setOutput(defaultTestOut);
            } else {
                resources.setOutput(defaultOut);
            }
            getLog().info(" using default out : " + resources.getOutput());
        } else {
            getLog().info(" out               : " + resources.getOutput());
        }

        return resources;
    }

    /**
     * Recupere le fichier donnée à partir de son chemin relatif sur le basedir
     * du projet maven.
     * 
     * @param paths les paths pour atteindre le fichier ou le répertoire
     *
     * @return le fichier de la destination
     */
    protected File getFile(String... paths) {
        File result = project.getBasedir();
        for (String path : paths) {
            result = new File(result, path);
        }
        return result;
    }

    /**
     * Ajout dans la liste des urls et l'ensemble des urls sous forme de string, l'url
     * donné.
     *
     * Note : l'utilisation de l'ensemble des urls sous forme de String est utile
     * sinon la comparaison entre url requiere des appels reseaux et cela n'est
     * pas acceptable dans notre cas (mode offline ne marcherait plus...).
     *
     * @param url l'url a rajouter dans la liste
     * @param urls la liste des urls
     * @param urlsAsString l'ensemble des urls sous forme de String
     */
    protected void addUrl(URL url, List<URL> urls, Set<String> urlsAsString) {
        String toString = url.toString();
        if (!urlsAsString.contains(toString)) {
            urlsAsString.add(toString);
            urls.add(url);
        }
    }
}
