/*
 * #%L
 * EUGene :: Java templates
 * 
 * $Id: JavaBeanTransformer.java 1181 2012-10-24 19:30:08Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/eugene/tags/eugene-2.5.5/eugene-java-templates/src/main/java/org/nuiton/eugene/java/JavaBeanTransformer.java $
 * %%
 * Copyright (C) 2004 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.eugene.java;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.EugeneTagValues;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;




/**
 * JavaBeanTransformer generates simple bean with pcs support
 * (and nothing else) according to the JavaBeans 1.1 norm.
 *
 * Since version 2.2.1, it is possible to
 * <ul>
 * <li>generate a simple POJO (says with no PCS support) by using the tag value {@link EugeneTagValues#TAG_NO_PCS}.</li>
 * <li>generate i18n keys using the tag value {@link EugeneTagValues#TAG_I18N_PREFIX}.</li>
 * </ul>
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.JavaBeanTransformer"
 * @since 2.0.2
 */
public class JavaBeanTransformer extends ObjectModelTransformerToJava {

    private static final Log log = LogFactory.getLog(JavaBeanTransformer.class);

    public static final String DEFAULT_CONSTANT_PREFIX = "PROPERTY_";

    @Override
    public void transformFromClass(ObjectModelClass input) {

        if (!JavaGeneratorUtil.hasBeanStereotype(input)) {

            //  not a bean
            return;
        }

        if (canGenerateImpl(input)) {

            generateImpl(input);
        }
        
        if (!canGenerateAbstract(input)) {

            // nothing more to do
            return;
        }

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if(CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaGeneratorUtil.hasBeanStereotype(superclass)) {
                    superClassIsBean = true;
                    break;
                }
            }
        }

        ObjectModelClass output =
                createAbstractClass(input.getName(), input.getPackageName());

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        String i18nPrefix = JavaGeneratorUtil.getI18nPrefixTagValue(input, model);
        if (!StringUtils.isEmpty(i18nPrefix)) {
            generateI18nBlock(input, output, i18nPrefix);
        }

        String noPCSTagValue = JavaGeneratorUtil.getNoPCSTagValue(model, input);
        boolean usePCS = StringUtils.isEmpty(noPCSTagValue) ||
                         !"true".equals(noPCSTagValue.trim());

        String noGenerateBooleanGetMethods =
                JavaGeneratorUtil.getDoNotGenerateBooleanGetMethods(model, input);
        boolean generateBooleanGetMethods =
                StringUtils.isEmpty(noGenerateBooleanGetMethods) ||
                !"true".equals(noGenerateBooleanGetMethods.trim());

        String prefix = getConstantPrefix(input, DEFAULT_CONSTANT_PREFIX);

        setConstantPrefix(prefix);

        addSuperClass(input, output);

        boolean serializableFound = addInterfaces(input, output);

        if (superClassIsBean) {
            serializableFound = true;
        }

        addSerializable(input, output, serializableFound);

        Set<String> constantNames = addConstantsFromDependency(input, output);

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        // Add properties constant
        for (ObjectModelAttribute attr : properties) {

            createPropertyConstant(output, attr, prefix, constantNames);
        }

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output, attr, usePCS, generateBooleanGetMethods);
        }

        // Add operations
        createAbstractOperations(output, input.getOperations());

        if (!superClassIsBean) {

            if (usePCS) {

                // Add property change support
                createPropertyChangeSupport(output);
            }

            boolean hasAMultipleProperty = containsMutiple(properties);

            // Add helper operations
            if (hasAMultipleProperty) {
                createGetChildMethod(output);
            }
        }
    }

    protected boolean canGenerateAbstract(ObjectModelClass input) {
        boolean b = !isInClassPath(input);
        return b;
    }

    protected boolean canGenerateImpl(ObjectModelClass input) {
        String fqn = input.getQualifiedName() + "Impl";

        if (isInClassPath(fqn)) {

            // already in class-path
            return false;
        }
        
        boolean hasOperations = !input.getAllOtherOperations(true).isEmpty() ||
                                !input.getOperations().isEmpty();

        // generate only if no user operations found
        return !hasOperations;
    }
    
    protected void createPropertyConstant(ObjectModelClass output,
                                          ObjectModelAttribute attr,
                                          String prefix,
                                          Set<String> constantNames) {

        String attrName = getAttributeName(attr);

        String constantName = prefix + builder.getConstantName(attrName);

        if (constantNames.contains(constantName)) {

            // already generated
            return;
        }

        addConstant(output,
                    constantName,
                    String.class,
                    "\"" + attrName + "\"",
                    ObjectModelJavaModifier.PUBLIC
        );
    }

    protected String getAttributeName(ObjectModelAttribute attr) {
        String attrName = attr.getName();
        if (attr.hasAssociationClass()) {
            String assocAttrName = JavaGeneratorUtil.getAssocAttrName(attr);
            attrName = JavaGeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
        }
        return attrName;
    }

    protected String getAttributeType(ObjectModelAttribute attr) {
        String attrType = attr.getType();
        if (attr.hasAssociationClass()) {
            attrType = attr.getAssociationClass().getName();
        }
        return attrType;
    }

    protected boolean containsMutiple(List<ObjectModelAttribute> attributes) {

        boolean result = false;

        for (ObjectModelAttribute attr : attributes) {

            if (JavaGeneratorUtil.isNMultiplicity(attr)) {
                result = true;

                break;
            }

        }
        return result;
    }

    protected void createProperty(ObjectModelClass output,
                                  ObjectModelAttribute attr,
                                  boolean usePCS,
                                  boolean generateBooleanGetMethods) {

        String attrName = getAttributeName(attr);
        String attrType = getAttributeType(attr);

        boolean multiple = JavaGeneratorUtil.isNMultiplicity(attr);

        String constantName = getConstantName(attrName);
        String simpleType = JavaGeneratorUtil.getSimpleName(attrType);

        if (multiple) {

            createGetChildMethod(output,
                                 attrName,
                                 attrType,
                                 simpleType
            );

            createIsEmptyMethod(output,
                                attrName
            );

            createSizeMethod(output,
                             attrName
            );

            createAddChildMethod(output,
                                 attrName,
                                 attrType,
                                 constantName,
                                 usePCS
            );

            createAddAllChildrenMethod(output,
                                 attrName,
                                 attrType,
                                 constantName,
                                 usePCS
            );

            createRemoveChildMethod(output,
                                    attrName,
                                    attrType,
                                    constantName,
                                    usePCS
            );

            createRemoveAllChildrenMethod(output,
                                    attrName,
                                    attrType,
                                    constantName,
                                    usePCS
            );

            createContainsChildMethod(output,
                    attrName,
                    attrType,
                    constantName,
                    usePCS
            );

            createContainsAllChildrenMethod(output,
                    attrName,
                    attrType,
                    constantName,
                    usePCS
            );

            // Change type for Multiple attribute
            if (attr.isOrdered()) {
                attrType = List.class.getName() + "<" + attrType + ">";
            } else {
                attrType = Collection.class.getName() + "<" + attrType + ">";
            }

            simpleType = JavaGeneratorUtil.getSimpleName(attrType);
        }

        boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(attr);

        if (booleanProperty && !multiple) {

            // creates a isXXX method
            createGetMethod(output,
                            attrName,
                            attrType,
                            JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX
            );
        }

        if (multiple || !booleanProperty || generateBooleanGetMethods) {

            createGetMethod(output,
                            attrName,
                            attrType,
                            JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX
            );

        }
        createSetMethod(output,
                        attrName,
                        attrType,
                        simpleType,
                        constantName,
                        usePCS
        );

        // Add attribute to the class
        addAttribute(output,
                     attrName,
                     attrType,
                     "",
                    ObjectModelJavaModifier.PROTECTED
        );

    }

    protected void createAbstractOperations(ObjectModelClass ouput,
                                            Iterable<ObjectModelOperation> operations) {
        JavaGeneratorUtil.cloneOperations(
                this,
                operations,
                ouput,
                true,
                ObjectModelJavaModifier.ABSTRACT
        );
    }

    protected List<ObjectModelAttribute> getProperties(ObjectModelClass input) {
        List<ObjectModelAttribute> attributes =
                (List<ObjectModelAttribute>) input.getAttributes();

        List<ObjectModelAttribute> attrs =
                new ArrayList<ObjectModelAttribute>();
        for (ObjectModelAttribute attr : attributes) {
            if (attr.isNavigable()) {

                // only keep navigable attributes
                attrs.add(attr);
            }
        }
        return attrs;
    }

    protected void createGetMethod(ObjectModelClass output,
                                   String attrName,
                                   String attrType,
                                   String methodPrefix) {

        ObjectModelOperation getter = addOperation(
                output,
                getJavaBeanMethodName(methodPrefix , attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
                setOperationBody(getter, ""
    +"\n"
+"        return "+attrName+";\n"
+"    "
                );
    }

    protected void createGetChildMethod(ObjectModelClass output,
                                        String attrName,
                                        String attrType,
                                        String simpleType) {
        ObjectModelOperation getChild = addOperation(
                output,
                getJavaBeanMethodName("get", attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(getChild, "int", "index");
        setOperationBody(getChild, ""
    +"\n"
+"        "+simpleType+" o = getChild("+attrName+", index);\n"
+"        return o;\n"
+"    "
        );
    }

    protected void createIsEmptyMethod(ObjectModelClass output,
                                        String attrName) {
        ObjectModelOperation getChild = addOperation(
                output,
                getJavaBeanMethodName("is", attrName)+"Empty",
                boolean.class,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(getChild, ""
    +"\n"
+"        return "+attrName+" != null && !"+attrName+".isEmpty();\n"
+"    "
        );
    }

    protected void createSizeMethod(ObjectModelClass output,
                                       String attrName) {
        ObjectModelOperation getChild = addOperation(
                output,
                getJavaBeanMethodName("size", attrName),
                int.class,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(getChild, ""
     +"\n"
+"         return "+attrName+" == null ? 0 : "+attrName+".size();\n"
+"     "
        );
    }
    protected void createAddChildMethod(ObjectModelClass output,
                                        String attrName,
                                        String attrType,
                                        String constantName,
                                        boolean usePCS) {
        ObjectModelOperation addChild = addOperation(
                output,
                getJavaBeanMethodName("add", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(addChild, attrType, attrName);

        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder(""
    +"\n"
+"        "+methodName+"().add("+attrName+");\n"
+"    "
        );
        if (usePCS) {
            buffer.append(""
    +"\n"
+"        firePropertyChange("+constantName+", null, "+attrName+");\n"
+"    "
            );
        }
        setOperationBody(addChild, buffer.toString());
    }

    protected void createAddAllChildrenMethod(ObjectModelClass output,
                                        String attrName,
                                        String attrType,
                                        String constantName,
                                        boolean usePCS) {
        ObjectModelOperation addAllChild = addOperation(
                output,
                getJavaBeanMethodName("addAll", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(addAllChild, "java.util.Collection<" + attrType + ">", attrName);

        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder(""
    +"\n"
+"        "+methodName+"().addAll("+attrName+");\n"
+"    "
        );
        if (usePCS) {
            buffer.append(""
    +"\n"
+"        firePropertyChange("+constantName+", null, "+attrName+");\n"
+"    "
            );
        }
        setOperationBody(addAllChild, buffer.toString());
    }

    protected void createRemoveChildMethod(ObjectModelClass output,
                                           String attrName,
                                           String attrType,
                                           String constantName,
                                           boolean usePCS) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("remove", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);
        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder();
        buffer.append(""
    +"\n"
+"        boolean  removed = "+methodName+"().remove("+attrName+");\n"
+"    "
        );

        if (usePCS) {
            buffer.append(""
    +"\n"
+"        if (removed) {\n"
+"            firePropertyChange("+constantName+", "+attrName+", null);\n"
+"      }\n"
+"    "
            );
        }
        buffer.append(""
    +"\n"
+"        return removed;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createRemoveAllChildrenMethod(ObjectModelClass output,
                                           String attrName,
                                           String attrType,
                                           String constantName,
                                           boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("removeAll", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "java.util.Collection<" + attrType + ">", attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean  removed = "+methodName+"().removeAll("+attrName+");\n"
+"    "
        );

        if (usePCS) {
            buffer.append(""
    +"\n"
+"        if (removed) {\n"
+"            firePropertyChange("+constantName+", "+attrName+", null);\n"
+"      }\n"
+"    "
            );
        }
        buffer.append(""
    +"\n"
+"        return removed;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createContainsChildMethod(ObjectModelClass output,
                                             String attrName,
                                             String attrType,
                                             String constantName,
                                             boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("contains", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean contains = "+methodName+"().contains("+attrName+");\n"
+"    "
        );

        buffer.append(""
    +"\n"
+"        return contains;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createContainsAllChildrenMethod(ObjectModelClass output,
                                           String attrName,
                                           String attrType,
                                           String constantName,
                                           boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("containsAll", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "java.util.Collection<" + attrType + ">", attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean  contains = "+methodName+"().containsAll("+attrName+");\n"
+"    "
        );

        buffer.append(""
    +"\n"
+"        return contains;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createSetMethod(ObjectModelClass output,
                                   String attrName,
                                   String attrType,
                                   String simpleType,
                                   String constantName,
                                   boolean usePCS) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("set", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);

        if (usePCS) {
            String methodName = getJavaBeanMethodName("get", attrName);
            setOperationBody(operation, ""
    +"\n"
+"        "+simpleType+" oldValue = "+methodName+"();\n"
+"        this."+attrName+" = "+attrName+";\n"
+"        firePropertyChange("+constantName+", oldValue, "+attrName+");\n"
+"    "
            );
        } else {
            setOperationBody(operation, ""
    +"\n"
+"        this."+attrName+" = "+attrName+";\n"
+"    "
            );
        }
    }

    protected void createGetChildMethod(ObjectModelClass output) {
        ObjectModelOperation getChild = addOperation(
                output,
                "getChild", "<T> T",
                ObjectModelJavaModifier.PROTECTED
        );
        addParameter(getChild, "java.util.Collection<T>", "childs");
        addParameter(getChild, "int", "index");
        setOperationBody(getChild, ""
    +"\n"
+"        if (childs != null) {\n"
+"            int i = 0;\n"
+"            for (T o : childs) {\n"
+"                if (index == i) {\n"
+"                    return o;\n"
+"                }\n"
+"                i++;\n"
+"            }\n"
+"        }\n"
+"        return null;\n"
+"    "
        );
    }

    protected void addSerializable(ObjectModelClass input,
                                   ObjectModelClass output,
                                   boolean interfaceFound) {
        if (!interfaceFound) {
            addInterface(output, Serializable.class);
        }

        // Generate the serialVersionUID
        long serialVersionUID = JavaGeneratorUtil.generateSerialVersionUID(input);

        addConstant(output,
                    JavaGeneratorUtil.SERIAL_VERSION_UID,
                    "long",
                    serialVersionUID + "L",
                    ObjectModelJavaModifier.PRIVATE
        );
    }

    /**
     * Add all interfaces defines in input class and returns if
     * {@link Serializable} interface was found.
     * 
     * @param input the input model class to process
     * @param output the output generated class
     * @return {@code true} if {@link Serializable} was found from input,
     *         {@code false} otherwise
     */
    protected boolean addInterfaces(ObjectModelClass input,
                                 ObjectModelClass output) {
        boolean foundSerializable = false;
        for (ObjectModelInterface parentInterface : input.getInterfaces()) {
            String fqn = parentInterface.getQualifiedName();
            addInterface(output, fqn);
            if (Serializable.class.getName().equals(fqn))  {
                foundSerializable = true;
            }
        }
        return foundSerializable;
    }

    protected void addSuperClass(ObjectModelClass input,
                                 ObjectModelClass output) {
        // Set superclass
        Iterator<ObjectModelClass> j = input.getSuperclasses().iterator();
        if (j.hasNext()) {
            ObjectModelClass p = j.next();
            // We want to set the inheritance to the implementation class of the father
            // Ex for model : A -> B (a inherits B) we want : A -> BImpl -> B
            String qualifiedName = p.getQualifiedName() + "Impl";
            setSuperClass(output, qualifiedName);
        }
    }

    protected ObjectModelClass generateImpl(ObjectModelClass input) {

        ObjectModelClass resultClassImpl = createClass(
                input.getName() + "Impl",
                input.getPackageName()
        );

        // set the abstract resulClass as the resultClassImpl super class
        setSuperClass(resultClassImpl, input.getQualifiedName());

        // add a fix serialVersionUID, since the class has no field nor method
        addConstant(resultClassImpl,
                    JavaGeneratorUtil.SERIAL_VERSION_UID,
                    "long",
                    "1L",
                    ObjectModelJavaModifier.PRIVATE
        );
        return resultClassImpl;
    }

    protected void createPropertyChangeSupport(ObjectModelClass output) {

        addAttribute(output,
                     "pcs",
                     PropertyChangeSupport.class,
                     "new PropertyChangeSupport(this)",
                    ObjectModelJavaModifier.PROTECTED,
                    ObjectModelJavaModifier.FINAL
        );

        // Add PropertyListener

        ObjectModelOperation operation;

        operation = addOperation(output,
                                 "addPropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.addPropertyChangeListener(listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "addPropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.addPropertyChangeListener(propertyName, listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "removePropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.removePropertyChangeListener(listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "removePropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.removePropertyChangeListener(propertyName, listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "firePropertyChange",
                                 "void",
                                 ObjectModelJavaModifier.PROTECTED
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, Object.class, "oldValue");
        addParameter(operation, Object.class, "newValue");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.firePropertyChange(propertyName, oldValue, newValue);\n"
+"    "
        );

        operation = addOperation(output,
                                 "firePropertyChange",
                                 "void",
                                 ObjectModelJavaModifier.PROTECTED
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, Object.class, "newValue");
        setOperationBody(operation, ""
    +"\n"
+"        firePropertyChange(propertyName, null, newValue);\n"
+"    "
        );
    }

}
