/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.reflect.plugins.javassist;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMethod;
import javassist.CtPrimitiveType;
import javassist.NotFoundException;

import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.Classes;
import org.jboss.util.JBossStringBuilder;

/**
 * A JavassistUtil.
 * 
 * @author <a href="mailto:stale.pedersen@jboss.org">Stale W. Pedersen</a>
 * @author <a href="mailto:kabir.khan@jboss.org">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class JavassistUtil
{
   public static Class<?> ctClassToClass(CtClass ct)
   {
      if(ct.isModified() && !ct.isFrozen())
      {
         try
         {
            return ct.toClass();
         }
         catch (CannotCompileException e)
         {
            throw new org.jboss.reflect.spi.CannotCompileException(e.toString());
         }
      }
      
      try
      {
         if (ct.isArray())
         {
            int dim = 0;

            while (ct.getComponentType() != null)
            {
               dim++;
               ct = ct.getComponentType();
            }

            if (ct.isPrimitive())
            {
               StringBuilder sb = new StringBuilder();
               for (int i = 0 ; i < dim ; i++)
               {
                  sb.append("[");
               }

               sb.append(((CtPrimitiveType)ct).getDescriptor());
               try
               {
                  return getClassLoader(ct).loadClass(sb.toString());
               }
               catch(ClassNotFoundException cnfe)
               {
                  return Class.forName(sb.toString(), false, getClassLoader(ct));
               }
            }
            else
            {
               return Array.newInstance(ctClassToClass(ct), new int[dim]).getClass();
            }
         }
         else
         {
            return getClassLoader(ct).loadClass(ct.getName());
         }
      }
      catch (NotFoundException e)
      {
         throw new org.jboss.reflect.spi.NotFoundException(e.toString());
      }
      catch (ClassNotFoundException e)
      {
         try
         {
            return ct.toClass();
         }
         catch (CannotCompileException e1)
         {
            throw new org.jboss.reflect.spi.CannotCompileException(e1.toString());
         }
      }
   }
   
   public static CtClass toCtClass(ClassPool pool, String name)
   {
      try
      {
         return pool.get(name);
      }
      catch (NotFoundException e)
      {
        throw new org.jboss.reflect.spi.NotFoundException(e.toString());
      }
   }
   
   public static CtClass[] toCtClass(ClassPool pool, String[] names)
   {
      if(names == null)
         return new CtClass[0];
      CtClass[] classes = new CtClass[names.length];
      for(int i=0; i < names.length; i++)
      {
         classes[i] = toCtClass(pool, names[i]);
      }
      return classes;
   }
   
   public static CtClass toCtClass(ClassInfo clazz)
   {
      if(clazz instanceof JavassistTypeInfo)
         return ((JavassistTypeInfo) clazz).getCtClass();
      else
      {
         try
         {
            ClassLoader classLoader = clazz.getClassLoader();
            ClassPool classPool = JavassistTypeInfoFactoryImpl.getPoolFactory().getPoolForLoader(classLoader);
            return classPool.get(clazz.getName());
         }
         catch (NotFoundException e)
         {
            throw new org.jboss.reflect.spi.NotFoundException(e.toString());
         }
      }
   }

   public static TypeInfo toTypeInfo(CtClass ctClass)
   {
      JavassistTypeInfoFactoryImpl impl = JavassistTypeInfoFactory.delegate;
      return impl.get(ctClass);
   }

   public static CtClass[] toCtClass(ClassInfo[] classes)
   {
      CtClass[] clazzes = new CtClass[classes.length];
      for(int i=0; i < classes.length; i++)
      {
         clazzes[i] = toCtClass(classes[i]);
      }
      return clazzes;
   }

   public static ClassLoader getClassLoader(CtClass ctClass)
   {
      ClassPool pool = ctClass.getClassPool();
      ClassLoader loader = null;
      if (pool != null)
         loader = pool.getClassLoader();
      
      if (loader == null)
         loader = SecurityActions.getContextClassLoader();
      
      return loader;
   }

   public static String getSignature(CtConstructor ctor)
   {
      return ctor.getDeclaringClass().getName() + ctor.getSignature();
   }

   public static String getSignature(CtMethod method)
   {
      return method.getDeclaringClass().getName() + "." + method.getName() + method.getSignature();
   }

   public static String getSignature(CtField field)
   {
      return field.getDeclaringClass().getName() + "." + field.getName();
   }
   
   public static Field ctFieldToField(CtField field)
   {
      ClassLoader cl = null;
      try
      {
         CtClass declaring = field.getDeclaringClass(); 
         cl = getClassLoader(declaring);
         Class<?> realClass = cl.loadClass(declaring.getName());
         //TODO might need to check if the class is being generated in which case we should call ctClassToClass?
         
         
         return SecurityActions.getDeclaredField(realClass, field.getName());
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(field.getDeclaringClass().getName() + " not found in " + cl);
      }
      catch (NoSuchFieldException e)
      {
         throw new RuntimeException(field.getName() + " not found in " + field.getDeclaringClass().getName() );
      }
   }

   private static Class<?> loadClass(ClassLoader cl, CtClass ctClass)
   {
      if (ctClass.isPrimitive())
      {
         return getPrimitiveClass(ctClass);
      }
      else if (ctClass.isArray())
      {
         JBossStringBuilder buffer = new JBossStringBuilder();
         try
         {
            CtClass temp = ctClass;
            int dims = 0;
            while (temp.isArray())
            {
               buffer.append('[');
               temp = temp.getComponentType();
            }
            if (temp.isPrimitive())
            {
               CtPrimitiveType primitive = (CtPrimitiveType) temp;
               buffer.append(Character.toString(primitive.getDescriptor()));
            }
            else
            {
               buffer.append('L');
               buffer.append(temp.getName());
               buffer.append(';');
            }
            return Classes.loadClass(buffer.toString(), cl);
         }
         catch (NotFoundException e)
         {
            throw new RuntimeException(ctClass.getName(), e);
         }
         catch (ClassNotFoundException e)
         {
            throw new RuntimeException(buffer.toString() + " not found in " + cl);
         }
      }
      
      try
      {
         return cl.loadClass(ctClass.getName());
         //TODO might need to check if the class is being generated in which case we should call ctClassToClass?
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(ctClass.getName() + " not found in " + cl);
      }
   }
   
   private static Class<?> getPrimitiveClass(CtClass ctClass)
   {
      if (!ctClass.isPrimitive())
         throw new IllegalArgumentException("Not a primitive ");

      if (ctClass == CtClass.booleanType)
         return Boolean.TYPE;
      else if (ctClass == CtClass.byteType)
         return Byte.TYPE;
      else if (ctClass == CtClass.charType)
         return Character.TYPE;
      else if (ctClass == CtClass.doubleType)
         return Double.TYPE;
      else if (ctClass == CtClass.floatType)
         return Float.TYPE;
      else if (ctClass == CtClass.intType)
         return Integer.TYPE;
      else if (ctClass == CtClass.longType)
         return Long.TYPE;
      else if (ctClass == CtClass.shortType)
         return Short.TYPE;
      
      return null;
   }
   
   private static Class<?>[] loadParameters(ClassLoader cl, CtBehavior behavior)
   {
      CtClass[] params;
      try
      {
         params = behavior.getParameterTypes();
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException("Could not determine the parameters for " + behavior + " in " + behavior.getDeclaringClass().getClassPool(), e);
      }
      Class<?>[] real = new Class<?>[params.length];
      
      for (int i = 0 ; i < params.length ; i++)
         real[i] = loadClass(cl, params[i]);
      
      return real;
   }
   
   public static Method ctMethodToMethod(CtMethod method)
   {
      ClassLoader cl = null;
      try
      {
         CtClass declaring = method.getDeclaringClass(); 
         cl = getClassLoader(declaring);
         Class<?> realClass = loadClass(cl, declaring);
         Class<?>[] params = loadParameters(cl, method);
         
         return SecurityActions.getDeclaredMethod(realClass, method.getName(), params);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(method.getName() + method.getSignature() + " not found in " + method.getDeclaringClass().getName() );
      }
   }


   public static Constructor<?> ctConstructorToConstructor(CtConstructor constructor)
   {
      ClassLoader cl = null;
      try
      {
         CtClass declaring = constructor.getDeclaringClass(); 
         cl = getClassLoader(declaring);
         Class<?> realClass = cl.loadClass(declaring.getName());
         Class<?>[] params = loadParameters(cl, constructor);
         
         return SecurityActions.getDeclaredConstructor(realClass, params);
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(constructor.getDeclaringClass().getName() + " not found in " + cl);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(constructor.getName() + constructor.getSignature() + " not found in " + constructor.getDeclaringClass().getName() );
      }
   }
}
