/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator.framework.recipes.queue;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.queue.QueueAllocator;
import org.apache.curator.framework.recipes.queue.QueueBase;
import org.apache.curator.framework.recipes.queue.QueueSharderPolicies;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueueSharder<U, T extends QueueBase<U>>
implements Closeable {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final CuratorFramework client;
    private final QueueAllocator<U, T> queueAllocator;
    private final String queuePath;
    private final QueueSharderPolicies policies;
    private final ConcurrentMap<String, T> queues = Maps.newConcurrentMap();
    private final Set<String> preferredQueues = Sets.newSetFromMap((Map)Maps.newConcurrentMap());
    private final AtomicReference<State> state = new AtomicReference<State>(State.LATENT);
    private final LeaderLatch leaderLatch;
    private final Random random = new Random();
    private final ExecutorService service;
    private static final String QUEUE_PREFIX = "queue-";

    public QueueSharder(CuratorFramework client, QueueAllocator<U, T> queueAllocator, String queuePath, String leaderPath, QueueSharderPolicies policies) {
        this.client = client;
        this.queueAllocator = queueAllocator;
        this.queuePath = queuePath;
        this.policies = policies;
        this.leaderLatch = new LeaderLatch(client, leaderPath);
        this.service = Executors.newSingleThreadExecutor(policies.getThreadFactory());
    }

    public void start() throws Exception {
        Preconditions.checkState((boolean)this.state.compareAndSet(State.LATENT, State.STARTED), (Object)"Cannot be started more than once");
        this.client.newNamespaceAwareEnsurePath(this.queuePath).ensure(this.client.getZookeeperClient());
        this.getInitialQueues();
        this.leaderLatch.start();
        this.service.submit(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                try {
                    while (!Thread.currentThread().isInterrupted() && QueueSharder.this.state.get() == State.STARTED) {
                        Thread.sleep(QueueSharder.this.policies.getThresholdCheckMs());
                        QueueSharder.this.checkThreshold();
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                return null;
            }
        });
    }

    @Override
    public void close() {
        if (this.state.compareAndSet(State.STARTED, State.CLOSED)) {
            this.service.shutdownNow();
            CloseableUtils.closeQuietly((Closeable)this.leaderLatch);
            for (QueueBase queue : this.queues.values()) {
                try {
                    queue.close();
                }
                catch (IOException e) {
                    this.log.error("Closing a queue", (Throwable)e);
                }
            }
        }
    }

    public T getQueue() {
        Preconditions.checkState((this.state.get() == State.STARTED ? 1 : 0) != 0, (Object)"Not started");
        ArrayList localPreferredQueues = Lists.newArrayList(this.preferredQueues);
        if (localPreferredQueues.size() > 0) {
            String key = (String)localPreferredQueues.get(this.random.nextInt(localPreferredQueues.size()));
            return (T)((QueueBase)this.queues.get(key));
        }
        ArrayList keys = Lists.newArrayList(this.queues.keySet());
        String key = (String)keys.get(this.random.nextInt(keys.size()));
        return (T)((QueueBase)this.queues.get(key));
    }

    public int getShardQty() {
        return this.queues.size();
    }

    public Collection<String> getQueuePaths() {
        return ImmutableSet.copyOf(this.queues.keySet());
    }

    private void getInitialQueues() throws Exception {
        List children = (List)this.client.getChildren().forPath(this.queuePath);
        for (String child : children) {
            String queuePath = ZKPaths.makePath((String)this.queuePath, (String)child);
            this.addNewQueueIfNeeded(queuePath);
        }
        if (children.size() == 0) {
            this.addNewQueueIfNeeded(null);
        }
    }

    private void addNewQueueIfNeeded(String newQueuePath) throws Exception {
        T queue;
        if (newQueuePath == null) {
            newQueuePath = ZKPaths.makePath((String)this.queuePath, (String)(QUEUE_PREFIX + UUID.randomUUID().toString()));
        }
        if (!this.queues.containsKey(newQueuePath) && this.queues.putIfAbsent(newQueuePath, queue = this.queueAllocator.allocateQueue(this.client, newQueuePath)) == null) {
            queue.start();
            this.preferredQueues.add(newQueuePath);
        }
    }

    private void checkThreshold() {
        try {
            boolean addAQueue = false;
            int size = 0;
            List children = (List)this.client.getChildren().forPath(this.queuePath);
            for (String child : children) {
                String queuePath = ZKPaths.makePath((String)this.queuePath, (String)child);
                this.addNewQueueIfNeeded(queuePath);
                Stat stat = (Stat)this.client.checkExists().forPath(queuePath);
                if (stat.getNumChildren() >= this.policies.getNewQueueThreshold()) {
                    if (!this.preferredQueues.contains(queuePath)) continue;
                    size = stat.getNumChildren();
                    addAQueue = true;
                    this.preferredQueues.remove(queuePath);
                    continue;
                }
                if (stat.getNumChildren() > this.policies.getNewQueueThreshold() / 2) continue;
                this.preferredQueues.add(queuePath);
            }
            if (addAQueue && this.leaderLatch.hasLeadership()) {
                if (this.queues.size() < this.policies.getMaxQueues()) {
                    this.log.info(String.format("Adding a queue due to exceeded threshold. Queue Size: %d - Threshold: %d", size, this.policies.getNewQueueThreshold()));
                    this.addNewQueueIfNeeded(null);
                } else {
                    this.log.warn(String.format("Max number of queues (%d) reached. Consider increasing the max.", this.policies.getMaxQueues()));
                }
            }
        }
        catch (Exception e) {
            this.log.error("Checking queue counts against threshold", (Throwable)e);
        }
    }

    private static enum State {
        LATENT,
        STARTED,
        CLOSED;

    }
}

