/*
 * Decompiled with CFR 0.152.
 */
package net.timewalker.ffmq4.storage.data.impl;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import net.timewalker.ffmq4.management.destination.AbstractDestinationDescriptor;
import net.timewalker.ffmq4.storage.data.DataStoreException;
import net.timewalker.ffmq4.storage.data.impl.AbstractDataStore;
import net.timewalker.ffmq4.utils.ArrayTools;
import net.timewalker.ffmq4.utils.FastBitSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class AbstractBlockBasedDataStore
extends AbstractDataStore {
    private static final Log log = LogFactory.getLog(AbstractBlockBasedDataStore.class);
    public static final String DATA_FILE_SUFFIX = ".store";
    public static final String ALLOCATION_TABLE_SUFFIX = ".index";
    private static final byte FLAG_START_BLOCK = 1;
    private static final byte FLAG_END_BLOCK = 2;
    public static final int AT_HEADER_SIZE = 12;
    public static final int AT_BLOCK_SIZE = 13;
    public static final int AT_HEADER_BLOCKCOUNT_OFFSET = 0;
    protected static final int AT_HEADER_FIRSTBLOCK_OFFSET = 8;
    protected static final int AB_FLAGS_OFFSET = 0;
    protected static final int AB_ALLOCSIZE_OFFSET = 1;
    protected static final int AB_PREVBLOCK_OFFSET = 5;
    protected static final int AB_NEXTBLOCK_OFFSET = 9;
    protected AbstractDestinationDescriptor descriptor;
    protected int blockSize;
    protected int blockCount;
    private int maxBlockCount;
    private int autoExtendAmount;
    protected byte[] flags;
    protected int[] allocatedSize;
    protected int[] nextBlock;
    protected int[] previousBlock;
    protected int firstBlock;
    protected File allocationTableFile;
    protected File dataFile;
    protected RandomAccessFile allocationTableRandomAccessFile;
    protected RandomAccessFile dataRandomAccessFile;
    private int lastEmpty;
    private int size;
    private int blocksInUse;

    public AbstractBlockBasedDataStore(AbstractDestinationDescriptor descriptor) {
        this.descriptor = descriptor;
        this.maxBlockCount = descriptor.getMaxBlockCount();
        this.autoExtendAmount = descriptor.getAutoExtendAmount();
    }

    @Override
    public final void init() throws DataStoreException {
        this.initFilesystem();
        this.loadAllocationTable();
    }

    protected void initFilesystem() throws DataStoreException {
        String baseName = this.descriptor.getName();
        File dataFolder = this.descriptor.getDataFolder();
        log.debug((Object)("[" + baseName + "] Initializing store '" + baseName + "' filesystem"));
        try {
            this.allocationTableFile = new File(dataFolder, baseName + ALLOCATION_TABLE_SUFFIX);
            if (!this.allocationTableFile.canRead()) {
                throw new DataStoreException("Cannot access store allocation table : " + this.allocationTableFile.getAbsolutePath());
            }
            this.allocationTableRandomAccessFile = new RandomAccessFile(this.allocationTableFile, "rw");
            this.dataFile = new File(dataFolder, baseName + DATA_FILE_SUFFIX);
            if (!this.dataFile.canRead()) {
                throw new DataStoreException("Cannot access store data file : " + this.dataFile.getAbsolutePath());
            }
            this.dataRandomAccessFile = new RandomAccessFile(this.dataFile, "rw");
        }
        catch (FileNotFoundException e) {
            throw new DataStoreException("Cannot access file : " + e.getMessage());
        }
    }

    private final void loadAllocationTable() throws DataStoreException {
        log.debug((Object)("[" + this.descriptor.getName() + "] Loading allocation table " + this.allocationTableFile.getAbsolutePath()));
        FilterInputStream in = null;
        try {
            in = new DataInputStream(new BufferedInputStream(new FileInputStream(this.allocationTableFile), 16384));
            this.blockCount = ((DataInputStream)in).readInt();
            this.blockSize = ((DataInputStream)in).readInt();
            this.firstBlock = ((DataInputStream)in).readInt();
            this.flags = new byte[this.blockCount];
            this.allocatedSize = new int[this.blockCount];
            this.previousBlock = new int[this.blockCount];
            this.nextBlock = new int[this.blockCount];
            this.blocksInUse = 0;
            int msgCount = 0;
            for (int n = 0; n < this.blockCount; ++n) {
                this.flags[n] = ((DataInputStream)in).readByte();
                this.allocatedSize[n] = ((DataInputStream)in).readInt();
                this.previousBlock[n] = ((DataInputStream)in).readInt();
                this.nextBlock[n] = ((DataInputStream)in).readInt();
                if (this.allocatedSize[n] == -1) continue;
                ++this.blocksInUse;
                if ((this.flags[n] & 1) <= 0) continue;
                ++msgCount;
            }
            this.locks = new FastBitSet(this.blockCount);
            this.size = msgCount;
            log.debug((Object)("[" + this.descriptor.getName() + "] " + msgCount + " entries found"));
        }
        catch (EOFException e) {
            throw new DataStoreException("Allocation table is truncated : " + this.allocationTableFile.getAbsolutePath(), e);
        }
        catch (IOException e) {
            throw new DataStoreException("Cannot initialize allocation table : " + this.allocationTableFile.getAbsolutePath(), e);
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException e) {
                    log.error((Object)("[" + this.descriptor.getName() + "] Could not close file input stream"), (Throwable)e);
                }
            }
        }
    }

    private int findEmpty() throws DataStoreException {
        int pos = this.lastEmpty;
        for (int n = 0; n < this.blockCount; ++n) {
            if (pos >= this.blockCount) {
                pos = 0;
            }
            if (this.allocatedSize[pos] == -1) {
                this.lastEmpty = pos + 1;
                return pos;
            }
            ++pos;
        }
        throw new DataStoreException("Allocation table is full (" + this.blockCount + " blocks)");
    }

    @Override
    protected void checkHandle(int handle) throws DataStoreException {
        if (handle < 0 || handle >= this.blockCount || this.allocatedSize[handle] == -1 || (this.flags[handle] & 1) == 0) {
            throw new DataStoreException("Invalid handle : " + handle);
        }
    }

    @Override
    public final int first() throws DataStoreException {
        return this.firstBlock;
    }

    @Override
    public final int next(int handle) throws DataStoreException {
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        int current = this.nextBlock[handle];
        while (current != -1 && (this.flags[current] & 1) == 0) {
            current = this.nextBlock[current];
        }
        return current;
    }

    @Override
    public final int previous(int handle) throws DataStoreException {
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        int current = this.previousBlock[handle];
        while (current != -1 && (this.flags[current] & 1) == 0) {
            current = this.previousBlock[current];
        }
        return current;
    }

    private int computeSize(int handle) throws DataStoreException {
        int totalSize = 0;
        int current = handle;
        while (current != -1) {
            totalSize += this.allocatedSize[current];
            if ((this.flags[current] & 2) > 0) break;
            current = this.nextBlock[current];
        }
        if (current == -1) {
            throw new DataStoreException("Can't find end block for " + handle);
        }
        return totalSize;
    }

    private int computeBlocks(int handle) throws DataStoreException {
        int totalBlocks = 0;
        int current = handle;
        while (current != -1) {
            ++totalBlocks;
            if ((this.flags[current] & 2) > 0) break;
            current = this.nextBlock[current];
        }
        if (current == -1) {
            throw new DataStoreException("Can't find end block for " + handle);
        }
        return totalBlocks;
    }

    public final byte[] retrieveHeader(int handle, int headerSize) throws DataStoreException {
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        byte[] data = new byte[headerSize];
        this.readDataBlock(data, 0, headerSize, handle);
        return data;
    }

    @Override
    public final Object retrieve(int handle) throws DataStoreException {
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        int totalSize = this.computeSize(handle);
        byte[] data = new byte[totalSize];
        int offset = 0;
        int current = handle;
        while (current != -1) {
            int blockLen = this.allocatedSize[current];
            this.readDataBlock(data, offset, blockLen, current);
            offset += blockLen;
            if ((this.flags[current] & 2) > 0) break;
            current = this.nextBlock[current];
        }
        if (current == -1) {
            throw new DataStoreException("Can't find end block for " + handle);
        }
        return data;
    }

    @Override
    public final int size() {
        return this.size;
    }

    private byte[] asByteArray(Object obj) throws DataStoreException {
        if (obj instanceof byte[]) {
            return (byte[])obj;
        }
        throw new DataStoreException("Unsupported object type : " + obj.getClass().getName());
    }

    @Override
    public final int delete(int handle) throws DataStoreException {
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        int previousHandle = this.previousBlock[handle];
        int current = handle;
        int nextHandle = -1;
        while (current != -1) {
            nextHandle = this.nextBlock[current];
            boolean isEndBlock = (this.flags[current] & 2) > 0;
            this.flags[current] = 0;
            this.allocatedSize[current] = -1;
            this.previousBlock[current] = -1;
            this.nextBlock[current] = -1;
            this.locks.clear(current);
            --this.blocksInUse;
            this.writeAllocationBlock(current);
            if (isEndBlock) break;
            current = nextHandle;
        }
        if (nextHandle != -1) {
            this.previousBlock[nextHandle] = previousHandle;
            this.writeAllocationBlock(nextHandle);
        }
        if (previousHandle != -1) {
            this.nextBlock[previousHandle] = nextHandle;
            this.writeAllocationBlock(previousHandle);
        }
        if (this.firstBlock == handle) {
            this.firstBlock = nextHandle;
            this.writeFirstBlock();
        }
        --this.size;
        this.flush();
        while (previousHandle != -1 && (this.flags[previousHandle] & 1) == 0) {
            previousHandle = this.previousBlock[previousHandle];
        }
        return previousHandle;
    }

    @Override
    public final int replace(int handle, Object obj) throws DataStoreException {
        int requiredFreeBlocks;
        int originalBlocks;
        if (SAFE_MODE) {
            this.checkHandle(handle);
        }
        byte[] data = this.asByteArray(obj);
        int previousHandle = this.previousBlock[handle];
        int targetBlocks = data.length / this.blockSize + (data.length % this.blockSize > 0 ? 1 : 0);
        if (targetBlocks > (originalBlocks = this.computeBlocks(handle)) && (requiredFreeBlocks = targetBlocks - originalBlocks) + this.blocksInUse > this.blockCount) {
            log.error((Object)("[" + this.descriptor.getName() + "] Not enough free blocks to update message (" + requiredFreeBlocks + " needed, " + (this.blockCount - this.blocksInUse) + " left), queue is full."));
            return -1;
        }
        boolean wasLocked = this.isLocked(handle);
        this.delete(handle);
        int newHandle = this.store(obj, previousHandle);
        if (wasLocked) {
            this.lock(newHandle);
        }
        this.flush();
        return newHandle;
    }

    @Override
    public final int store(Object obj, int previousHandle) throws DataStoreException {
        int firstBlockOfNextEntry;
        int lastBlockOfPreviousEntry;
        byte[] data = this.asByteArray(obj);
        if (previousHandle != -1) {
            lastBlockOfPreviousEntry = previousHandle;
            while (lastBlockOfPreviousEntry != -1 && (this.flags[lastBlockOfPreviousEntry] & 2) == 0) {
                lastBlockOfPreviousEntry = this.nextBlock[lastBlockOfPreviousEntry];
            }
            if (lastBlockOfPreviousEntry == -1) {
                throw new DataStoreException("Can't find end block for " + previousHandle);
            }
            firstBlockOfNextEntry = this.nextBlock[lastBlockOfPreviousEntry];
        } else {
            lastBlockOfPreviousEntry = previousHandle;
            firstBlockOfNextEntry = this.firstBlock;
        }
        int newHandle = this.storeData(data, lastBlockOfPreviousEntry, firstBlockOfNextEntry);
        if (newHandle == -1) {
            return -1;
        }
        if (previousHandle == -1) {
            this.firstBlock = newHandle;
            this.writeFirstBlock();
        }
        ++this.size;
        this.flush();
        return newHandle;
    }

    private boolean autoExtendStore() throws DataStoreException {
        if (this.autoExtendAmount == 0) {
            return false;
        }
        int oldBlockCount = this.blockCount;
        int newBlockCount = Math.min(this.blockCount + this.autoExtendAmount, this.maxBlockCount);
        if (newBlockCount <= oldBlockCount) {
            return false;
        }
        log.debug((Object)("[" + this.descriptor.getName() + "] Auto-extending store to " + newBlockCount + " blocks"));
        this.flags = ArrayTools.extend(this.flags, newBlockCount);
        this.allocatedSize = ArrayTools.extend(this.allocatedSize, newBlockCount);
        this.nextBlock = ArrayTools.extend(this.nextBlock, newBlockCount);
        this.previousBlock = ArrayTools.extend(this.previousBlock, newBlockCount);
        for (int n = this.blockCount; n < newBlockCount; ++n) {
            this.allocatedSize[n] = -1;
            this.previousBlock[n] = -1;
            this.nextBlock[n] = -1;
        }
        this.locks.ensureCapacity(newBlockCount);
        this.blockCount = newBlockCount;
        this.extendStoreFiles(oldBlockCount, newBlockCount);
        return true;
    }

    private int storeData(byte[] data, int previousBlockHandle, int nextBlockHandle) throws DataStoreException {
        int fullBlocks = data.length / this.blockSize;
        int remaining = data.length % this.blockSize;
        int requiredFreeBlocks = fullBlocks + (remaining > 0 ? 1 : 0);
        if (this.blocksInUse + requiredFreeBlocks > this.blockCount) {
            if (this.blocksInUse + requiredFreeBlocks > this.maxBlockCount) {
                log.debug((Object)("[" + this.descriptor.getName() + "] Not enough free blocks to store message (" + requiredFreeBlocks + " needed, " + (this.blockCount - this.blocksInUse) + " left), queue is full."));
                return -1;
            }
            if (!this.autoExtendStore()) {
                return -1;
            }
        }
        int newHandle = -1;
        int lastBlock = previousBlockHandle;
        int offset = 0;
        for (int i = 0; i < fullBlocks; ++i) {
            boolean isStartBlock = i == 0;
            boolean isEndBlock = remaining == 0 && i == fullBlocks - 1;
            lastBlock = this.storeDataBlock(data, offset, this.blockSize, lastBlock, isStartBlock, isEndBlock);
            offset += this.blockSize;
            if (!isStartBlock) continue;
            newHandle = lastBlock;
        }
        if (remaining > 0) {
            lastBlock = this.storeDataBlock(data, offset, remaining, lastBlock, fullBlocks == 0, true);
            if (newHandle == -1) {
                newHandle = lastBlock;
            }
        }
        if (nextBlockHandle != -1) {
            this.nextBlock[lastBlock] = nextBlockHandle;
            this.previousBlock[nextBlockHandle] = lastBlock;
            this.writeAllocationBlock(nextBlockHandle);
        }
        if (previousBlockHandle != -1) {
            this.nextBlock[previousBlockHandle] = newHandle;
            this.writeAllocationBlock(previousBlockHandle);
        }
        int current = newHandle;
        for (int i = 0; i < requiredFreeBlocks; ++i) {
            this.writeAllocationBlock(current);
            current = this.nextBlock[current];
        }
        this.blocksInUse += requiredFreeBlocks;
        return newHandle;
    }

    private int storeDataBlock(byte[] data, int offset, int len, int previousHandle, boolean startBlock, boolean endBlock) throws DataStoreException {
        int nextEmpty = this.findEmpty();
        byte flag = 0;
        if (startBlock) {
            flag = (byte)(flag | 1);
        }
        if (endBlock) {
            flag = (byte)(flag | 2);
        }
        this.flags[nextEmpty] = flag;
        this.nextBlock[nextEmpty] = -1;
        this.previousBlock[nextEmpty] = previousHandle;
        this.allocatedSize[nextEmpty] = len;
        if (previousHandle != -1) {
            this.nextBlock[previousHandle] = nextEmpty;
        }
        this.writeDataBlock(data, offset, len, nextEmpty);
        return nextEmpty;
    }

    @Override
    public void close() {
        try {
            if (this.allocationTableRandomAccessFile != null) {
                this.allocationTableRandomAccessFile.close();
            }
        }
        catch (IOException e) {
            log.error((Object)("[" + this.descriptor.getName() + "] Could not close allocation table file : " + e.toString()));
        }
        try {
            if (this.dataRandomAccessFile != null) {
                this.dataRandomAccessFile.close();
            }
        }
        catch (IOException e) {
            log.error((Object)("[" + this.descriptor.getName() + "] Could not close map file : " + e.toString()));
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Allocation Table (size=" + this.size + ")\n");
        sb.append("------------------------------------\n");
        sb.append("first block index : ");
        sb.append(this.firstBlock);
        sb.append("\n");
        for (int n = 0; n < this.blockCount; ++n) {
            sb.append(n);
            sb.append(": ");
            if (this.allocatedSize[n] == -1) {
                sb.append("(free)\n");
                continue;
            }
            sb.append(this.previousBlock[n]);
            sb.append("\t");
            sb.append(this.nextBlock[n]);
            sb.append("\t");
            sb.append(this.allocatedSize[n]);
            if ((this.flags[n] & 1) > 0) {
                sb.append("\t");
                sb.append("START");
            }
            if ((this.flags[n] & 2) > 0) {
                sb.append("\t");
                sb.append("END");
            }
            sb.append("\n");
        }
        sb.append("------------------------------------\n");
        return sb.toString();
    }

    protected void integrityCheck() throws DataStoreException {
        try {
            long expectedDataFileSize;
            long dataFileSize;
            long atFileSize = this.allocationTableRandomAccessFile.length();
            if (atFileSize < 25L) {
                throw new DataStoreException("Allocation table is truncated : " + this.allocationTableFile.getAbsolutePath());
            }
            FileInputStream inFile = new FileInputStream(this.allocationTableFile);
            DataInputStream in = new DataInputStream(new BufferedInputStream(inFile, 16384));
            int blockCount = in.readInt();
            int blockSize = in.readInt();
            int firstBlock = in.readInt();
            long expectedATFileSize = 12L + 13L * (long)blockCount;
            if (atFileSize != expectedATFileSize) {
                log.error((Object)("[" + this.descriptor.getName() + "] Allocation table has an invalid size (actual:" + atFileSize + ",expected:" + expectedATFileSize + "), fixing."));
                this.allocationTableRandomAccessFile.setLength(expectedATFileSize);
            }
            if ((dataFileSize = this.dataRandomAccessFile.length()) != (expectedDataFileSize = (long)blockSize * (long)blockCount)) {
                log.error((Object)("[" + this.descriptor.getName() + "] Data file has an invalid size (actual:" + dataFileSize + ",expected:" + expectedDataFileSize + "), fixing."));
                this.dataRandomAccessFile.setLength(expectedDataFileSize);
            }
            byte[] flags = new byte[blockCount];
            int[] allocatedSize = new int[blockCount];
            int[] previousBlock = new int[blockCount];
            int[] nextBlock = new int[blockCount];
            int blocksInUse = 0;
            int msgCount = 0;
            for (int n = 0; n < blockCount; ++n) {
                flags[n] = in.readByte();
                allocatedSize[n] = in.readInt();
                previousBlock[n] = in.readInt();
                nextBlock[n] = in.readInt();
                if (allocatedSize[n] == -1) continue;
                ++blocksInUse;
                if ((flags[n] & 1) <= 0) continue;
                ++msgCount;
            }
            in.close();
            log.debug((Object)("[" + this.descriptor.getName() + "] Blocks in use before fix : " + blocksInUse));
            log.debug((Object)("[" + this.descriptor.getName() + "] Messages count before fix : " + msgCount));
            boolean changed = false;
            if (firstBlock < -1 || firstBlock >= blockCount) {
                log.error((Object)("[" + this.descriptor.getName() + "] Invalid allocation table first block index (" + firstBlock + "), guessing new one ..."));
                firstBlock = this.guessFirstBlockIndex(blockCount, allocatedSize, nextBlock);
                log.debug((Object)("[" + this.descriptor.getName() + "] Guessed first block index : " + firstBlock));
                changed = true;
            }
            if (msgCount == 0) {
                if (firstBlock == -1) {
                    changed = changed || this.cleanupEmptyBlocks(blockCount, flags, allocatedSize, previousBlock, nextBlock);
                } else {
                    log.error((Object)("[" + this.descriptor.getName() + "] First block index should be -1, clearing ..."));
                    firstBlock = -1;
                    changed = true;
                }
            } else {
                if (firstBlock == -1) {
                    log.error((Object)("[" + this.descriptor.getName() + "] Invalid first block index, guessing value ..."));
                    firstBlock = this.guessFirstBlockIndex(blockCount, allocatedSize, nextBlock);
                    log.debug((Object)("[" + this.descriptor.getName() + "] Guessed first block index : " + firstBlock));
                    changed = true;
                }
                changed = changed || this.fixBlocks(blockCount, blockSize, firstBlock, flags, allocatedSize, previousBlock, nextBlock);
                boolean bl = changed = changed || this.cleanupEmptyBlocks(blockCount, flags, allocatedSize, previousBlock, nextBlock);
            }
            if (changed) {
                int n;
                msgCount = 0;
                blocksInUse = 0;
                for (n = 0; n < blockCount; ++n) {
                    if (allocatedSize[n] == -1) continue;
                    ++blocksInUse;
                    if ((flags[n] & 1) <= 0) continue;
                    ++msgCount;
                }
                log.debug((Object)("[" + this.descriptor.getName() + "] Blocks in use after fix : " + blocksInUse));
                log.debug((Object)("[" + this.descriptor.getName() + "] Messages count after fix : " + msgCount));
                log.debug((Object)("[" + this.descriptor.getName() + "] Allocation table was altered, saving ..."));
                this.allocationTableRandomAccessFile.seek(8L);
                this.allocationTableRandomAccessFile.writeInt(firstBlock);
                for (n = 0; n < blockCount; ++n) {
                    byte[] allocationBlock = new byte[]{flags[n], (byte)(allocatedSize[n] >>> 24 & 0xFF), (byte)(allocatedSize[n] >>> 16 & 0xFF), (byte)(allocatedSize[n] >>> 8 & 0xFF), (byte)(allocatedSize[n] >>> 0 & 0xFF), (byte)(previousBlock[n] >>> 24 & 0xFF), (byte)(previousBlock[n] >>> 16 & 0xFF), (byte)(previousBlock[n] >>> 8 & 0xFF), (byte)(previousBlock[n] >>> 0 & 0xFF), (byte)(nextBlock[n] >>> 24 & 0xFF), (byte)(nextBlock[n] >>> 16 & 0xFF), (byte)(nextBlock[n] >>> 8 & 0xFF), (byte)(nextBlock[n] >>> 0 & 0xFF)};
                    this.allocationTableRandomAccessFile.seek(12 + n * 13);
                    this.allocationTableRandomAccessFile.write(allocationBlock);
                }
                this.allocationTableRandomAccessFile.getFD().sync();
            } else {
                log.debug((Object)("[" + this.descriptor.getName() + "] Allocation table was not altered"));
            }
        }
        catch (IOException e) {
            throw new DataStoreException("Cannot check/fix store integrity : " + e);
        }
    }

    private int guessFirstBlockIndex(int blockCount, int[] allocatedSize, int[] nextBlock) {
        int n;
        FastBitSet referenced = new FastBitSet(blockCount);
        for (n = 0; n < blockCount; ++n) {
            if (allocatedSize[n] == -1 || nextBlock[n] == -1) continue;
            referenced.set(nextBlock[n]);
        }
        for (n = 0; n < blockCount; ++n) {
            if (allocatedSize[n] == -1 || nextBlock[n] == -1 || referenced.get(n)) continue;
            return n;
        }
        return -1;
    }

    private boolean fixBlocks(int blockCount, int blockSize, int firstBlock, byte[] flags, int[] allocatedSize, int[] previousBlock, int[] nextBlock) {
        boolean changed = false;
        FastBitSet fixedBlocks = new FastBitSet(blockCount);
        int previous = -1;
        int current = firstBlock;
        while (current != -1) {
            if (previousBlock[current] != previous) {
                log.debug((Object)("[" + this.descriptor.getName() + "] Fixing previous reference " + previousBlock[current] + " -> " + previous));
                previousBlock[current] = previous;
                changed = true;
            }
            fixedBlocks.set(current);
            previous = current;
            current = nextBlock[current];
        }
        for (int n = 0; n < blockCount; ++n) {
            if (allocatedSize[n] != -1 && !fixedBlocks.get(n)) {
                log.warn((Object)("[" + this.descriptor.getName() + "] Lost block found : " + n));
                allocatedSize[n] = -1;
                changed = true;
            }
            if (!fixedBlocks.get(n) || allocatedSize[n] > 0 && allocatedSize[n] <= blockSize) continue;
            log.warn((Object)("[" + this.descriptor.getName() + "] Block has an invalid size (" + allocatedSize[n] + "), replacing by " + blockSize));
            allocatedSize[n] = blockSize;
            changed = true;
        }
        boolean startExpected = true;
        previous = -1;
        current = firstBlock;
        while (current != -1) {
            if (startExpected) {
                if ((flags[current] & 1) == 0) {
                    log.warn((Object)("[" + this.descriptor.getName() + "] Missing start block flag, adding to block " + current));
                    int n = current;
                    flags[n] = (byte)(flags[n] | 1);
                    changed = true;
                }
                if ((flags[current] & 2) == 0) {
                    startExpected = false;
                }
            } else {
                if ((flags[current] & 1) > 0) {
                    log.warn((Object)("[" + this.descriptor.getName() + "] Missing end block flag, adding to previous block : " + previous));
                    int n = previous;
                    flags[n] = (byte)(flags[n] | 2);
                    changed = true;
                }
                if ((flags[current] & 2) > 0) {
                    startExpected = true;
                }
            }
            previous = current;
            current = nextBlock[current];
        }
        if (!startExpected) {
            log.warn((Object)("[" + this.descriptor.getName() + "] Missing end block flag, adding to last block : " + previous));
            int n = previous;
            flags[n] = (byte)(flags[n] | 2);
            changed = true;
        }
        return changed;
    }

    private boolean cleanupEmptyBlocks(int blockCount, byte[] flags, int[] allocatedSize, int[] previousBlock, int[] nextBlock) {
        boolean changed = false;
        for (int n = 0; n < blockCount; ++n) {
            if (allocatedSize[n] != -1 || flags[n] == 0 && nextBlock[n] == -1 && previousBlock[n] == -1) continue;
            flags[n] = 0;
            nextBlock[n] = -1;
            previousBlock[n] = -1;
            changed = true;
        }
        return changed;
    }

    public final int getBlockSize() {
        return this.blockSize;
    }

    protected abstract void writeFirstBlock() throws DataStoreException;

    protected abstract void writeAllocationBlock(int var1) throws DataStoreException;

    protected abstract void writeDataBlock(byte[] var1, int var2, int var3, int var4) throws DataStoreException;

    protected abstract void readDataBlock(byte[] var1, int var2, int var3, int var4) throws DataStoreException;

    protected abstract void extendStoreFiles(int var1, int var2) throws DataStoreException;

    protected abstract void flush() throws DataStoreException;

    protected final byte[] serializeAllocationBlock(int blockIndex) {
        byte[] allocationBlock = new byte[]{this.flags[blockIndex], (byte)(this.allocatedSize[blockIndex] >>> 24 & 0xFF), (byte)(this.allocatedSize[blockIndex] >>> 16 & 0xFF), (byte)(this.allocatedSize[blockIndex] >>> 8 & 0xFF), (byte)(this.allocatedSize[blockIndex] >>> 0 & 0xFF), (byte)(this.previousBlock[blockIndex] >>> 24 & 0xFF), (byte)(this.previousBlock[blockIndex] >>> 16 & 0xFF), (byte)(this.previousBlock[blockIndex] >>> 8 & 0xFF), (byte)(this.previousBlock[blockIndex] >>> 0 & 0xFF), (byte)(this.nextBlock[blockIndex] >>> 24 & 0xFF), (byte)(this.nextBlock[blockIndex] >>> 16 & 0xFF), (byte)(this.nextBlock[blockIndex] >>> 8 & 0xFF), (byte)(this.nextBlock[blockIndex] >>> 0 & 0xFF)};
        return allocationBlock;
    }

    @Override
    public final int getStoreUsage() {
        long ratio = this.blockCount > 0 ? (long)this.blocksInUse * 100L / (long)this.blockCount : 0L;
        return (int)ratio;
    }

    @Override
    public int getAbsoluteStoreUsage() {
        long ratio = this.maxBlockCount > 0 ? (long)this.blocksInUse * 100L / (long)this.maxBlockCount : 0L;
        return (int)ratio;
    }
}

