/*
 * Copyright 2007 Robert Hanson <iamroberthanson AT gmail.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gwtwidgets.client.ui.canvas.impl;

import org.gwtwidgets.client.ui.canvas.Canvas;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;

/**
 * <p>
 * Canvas implementation for Firefox, Safari, Opera and all browsers that
 * support the <code>&lt;canvas&gt;</code> element. This implementation wraps
 * around a <code>&lt;canvas&gt;</code> element and maintains a reference on a
 * 2D graphics context.
 * </p>
 * <p>
 * It was observed that complex scenes cause flickering in Opera and Safari,
 * thus this implementation relies on double buffering and as such creates two
 * canvas elements: a visible element and an offscreen element. All drawing
 * operations are applied on the buffer. Invocations to {@link #flush()} will
 * copy the buffer to the visible canvas.
 * </p>
 * 
 * @author George Georgovassilis g.georgovassilis[at]gmail.com
 * 
 */
public class FFCanvasImpl extends Canvas {

	private double offsetLeft;
	private double offsetTop;
	private JavaScriptObject canvas;
	private double strokeAlpha = 1.0;
	private org.gwtwidgets.client.ui.canvas.Font font;
	private Element buffer;

	private native int getWidth()/*-{
		return this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::buffer.width;
		}-*/;

	private native int getHeight()/*-{
		return this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::buffer.height;
		}-*/;

	private native JavaScriptObject createCanvasContext()/*-{
		var element = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::buffer;
		var canvas = element.getContext("2d");
		return canvas; 
		}-*/;

	private native void _drawImage(Element image, double sx, double sy, double swidth, double sheight, double dx, double dy, double dwidth,
			double dheight)/*-{
		var alpha = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::strokeAlpha;
		var canvas = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas; 
		if (alpha != 1){
			var oldAlpha = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.globalAlpha;
			canvas.globalAlpha = alpha;
			canvas.drawImage(image, sx, sy, swidth, sheight, dx, dy, dwidth, dheight);
			canvas.globalAlpha = oldAlpha;
			}
		else 
			canvas.drawImage(image, sx, sy, swidth, sheight, dx, dy, dwidth, dheight);
		}-*/;

	private native void fillRectangle(double left, double top, double width, double height)/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.fillRect(left,top,width,height); 
		}-*/;

	private native void clearRectangle(double left, double top, double width, double height)/*-{
	this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.clearRect(left,top,width,height); 
	}-*/;

	private native void beginPath()/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.beginPath();
	}-*/;

	private native void fillPath()/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.fill(); 
	}-*/;

	private native void strokePath()/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.stroke(); 
	}-*/;

	private native void lineTo(double x, double y)/*-{
			this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.lineTo(x,y); 
	}-*/;

	private native void moveTo(double x, double y)/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.moveTo(x,y); 
	}-*/;

	static native void paintCopy(Element canvas, Element image, double offsetX, double offsetY, double width, double height)/*-{
		var context = canvas.getContext("2d");
		context.drawImage(image, offsetX, offsetY, width, height, 0, 0, width, height);
	}-*/;

	static native void paintCopy(Element canvas, Element image)/*-{
		var context = canvas.getContext("2d");
		context.drawImage(image, 0, 0);
	}-*/;

	/**
	 * Creates a native canvas element and draws an image on it.
	 * 
	 * @param image Image element.
	 * @return
	 */
	static Element toNativeCanvas(Element image) {
		Element canvas = DOM.createElement("canvas");
		int width = Utils.getWidth(image);
		int height = Utils.getHeight(image);
		Utils.applySize(canvas, width, height);
		paintCopy(canvas, image);
		return canvas;
	};

	FFCanvasImpl(int width, int height) {
		Element e = DOM.createElement("canvas");
		setElement(e);
		setSize(width + "px", height + "px");
		Utils.applySize(e, width, height);

		buffer = DOM.createElement("canvas");
		Utils.applySize(buffer, width, height);
		canvas = createCanvasContext();
	}

	public void clear() {
		setOffset(0, 0);
		setRotation(0);
		setStroke(0, 0, 0, 0);
		setFill(255, 255, 255, 1);
		setStrokeWeight(1);
		clearRectangle(0, 0, getWidth(), getHeight());
		setStroke(0, 0, 0, 1);
	}
// TODO: there is currently an issue with Opera when fromAngle and twoAngle project to the same angle.
// Curiously enough, by traversing the arc counterclockwise it is painted correctly.
	public native void drawArc(double centerLeft, double centerTop, double radiusX, double radiusY, double fromAngle, double toAngle)/*-{
		var canvas = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas;
		var startAngle = toAngle-Math.PI/2;
		var endAngle = fromAngle-Math.PI/2;
		var r = radiusY/radiusX;
		canvas.save();
		canvas.beginPath();
		canvas.arc(centerLeft, centerTop, radiusX, startAngle, endAngle, 1); 
		canvas.fill();
		canvas.stroke();
		canvas.restore();
		}-*/;

	public native void drawLine(double fromLeft, double fromTop, double toLeft, double toTop)/*-{
		var canvas = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas;
		canvas.beginPath();
		canvas.moveTo(fromLeft, fromTop); 
		canvas.lineTo(toLeft, toTop); 
		canvas.stroke(); 
		}-*/;

	public void drawPolyLine(double[] x, double[] y) {
		beginPath();
		moveTo(x[0], y[0]);
		int length = x.length - 1;
		for (int i = 1; i < length; i++) {
			lineTo(x[i], y[i]);
		}
		strokePath();
	}

	public native void drawRectangle(double left, double top, double width, double height)/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.fillRect(left,top,width,height); 
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.strokeRect(left,top,width,height); 
		}-*/;

	public native void setOffset(double left, double top)/*-{
		var dl = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::offsetLeft;
		var dt = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::offsetTop;
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.translate(left-dl,top-dt);
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::offsetLeft = left;
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::offsetTop = top;
		}-*/;

	public native void setStrokeWeight(double weight)/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.lineWidth = weight;
		}-*/;

	public native void flush()/*-{
			var screenElement = this.@com.google.gwt.user.client.ui.UIObject::getElement()();
			var screenCanvas = screenElement.getContext("2d");
			var bufferCanvas = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::buffer;
			var width = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::getWidth()();
			var height = this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::getHeight()();
		
			screenCanvas.clearRect(0,0, width, height);
			screenCanvas.drawImage(bufferCanvas, 0, 0); 
		}-*/;

	public native void setFill(int red, int green, int blue, double alpha)/*-{
		var fillStyle = "rgba("+red+","+green+","+blue+","+alpha+")";
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.fillStyle = fillStyle;
		}-*/;

	public native void setStroke(int red, int green, int blue, double alpha)/*-{
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.strokeStyle = "rgba("+red+","+green+","+blue+","+alpha+")";
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::strokeAlpha = alpha;
		}-*/;

	public void drawPolygon(double[] x, double[] y) {
		beginPath();
		moveTo(x[0], y[0]);
		
		int mod = x.length;
		int length = mod + 1;
		
		for (int i = 1; i < length; i++) {
			lineTo(x[i%mod], y[i%mod]);
		}
		fillPath();
		strokePath();
	}

	private native void _setRotation(double angle)/*-{
		var oldAngle = this.@org.gwtwidgets.client.ui.canvas.Canvas::getRotation()();
		this.@org.gwtwidgets.client.ui.canvas.impl.FFCanvasImpl::canvas.rotate(angle - oldAngle);
	}-*/;
	
	public void setRotation(double angle) {
		_setRotation(angle);
		super.setRotation(angle);
	}

	public void drawImage(Element image, double sx, double sy, double swidth, double sheight, double dx, double dy, double dwidth,
			double dheight) {
		double oldOffsetLeft = offsetLeft;
		double oldOffsetTop = offsetTop;
		double oldRotation = rotation;

		dx += offsetLeft;
		dy += offsetTop;

		double offsx = dx + dwidth / 2;
		double offsy = dy + dheight / 2;

		setRotation(0);
		setOffset(offsx, offsy);

		setRotation(oldRotation);
		setOffset(dx, dy);
		_drawImage(image, sx+1, sy+1, swidth-1, sheight-1, 0, 0, dwidth, dheight);
		setOffset(offsx, offsy);
		setRotation(0);
		setOffset(oldOffsetLeft, oldOffsetTop);
		setRotation(oldRotation);
	}

	public void drawText(String text, double x, double y) {
		font.drawText(text, this, x, y);
	}

	public void setFont(org.gwtwidgets.client.ui.canvas.Font font) {
		this.font = font;
	}

	public Element prepareImage(Element image) {
		return toNativeCanvas(image);
	}

}