/*
 * Cantharella, Pharmacochemical database of natural substances - http://sourceforge.net/p/cantharella/ 
 * 
 * Copyright (C) 2009-2012 IRD (Institut de Recherche pour le Developpement) and by respective authors (see below)
 *
 * Cantharella is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * Cantharella is distributed in the hope that it will be useful,but WITHOUT ANY WARRANTY; without even the implied 
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more 
 * details.
 *
 * You should have received a copy of the GNU General Public License along with Cantharella.  If not, see 
 * <http://www.gnu.org/licenses/>.
 */
package nc.ird.cantharella.data.dao.impl;

import java.io.Serializable;
import java.util.List;

import nc.ird.cantharella.data.dao.GenericDao;
import nc.ird.cantharella.data.model.utils.AbstractModel;
import nc.ird.module.utils.AssertTools;
import nc.ird.module.utils.GenericsTools;

import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.ObjectRetrievalFailureException;
import org.springframework.orm.hibernate3.HibernateCallback;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;
import org.springframework.stereotype.Repository;

/**
 * Generic DAO implementation for Hibernate
 * @author Mickael Tricot
 * @author Adrien Cheype
 */
@Repository
public class HibernateTemplateDao extends HibernateDaoSupport implements GenericDao {

	/** Logger */
	// private static final Log LOG = LogTools.getLog();

	public HibernateTemplateDao() {
		super();
	}

	/**
	 * Criteria : model from a property value
	 * @param modelClass Model class
	 * @param propertyName Property name
	 * @param value Value
	 * @return Criteria
	 */
	private static DetachedCriteria criteriaByProperty(Class<? extends AbstractModel> modelClass, String propertyName,
			Serializable value) {
		return DetachedCriteria.forClass(modelClass).add(Restrictions.eq(propertyName, value));
	}

	/**
	 * Constructor
	 * @param sessionFactory Session factory
	 */
	@Autowired
	public HibernateTemplateDao(SessionFactory sessionFactory) {
		setSessionFactory(sessionFactory);
		getHibernateTemplate().setAllowCreate(false);
		getHibernateTemplate().setCacheQueries(true);
		getHibernateTemplate().afterPropertiesSet();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long count(DetachedCriteria criteria) {
		AssertTools.assertNotNull(criteria);
		return (Long) list(criteria.setProjection(Projections.rowCount())).get(0);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> long count(Class<M> modelClass) {
		AssertTools.assertNotNull(modelClass);
		return (Long) list(DetachedCriteria.forClass(modelClass).setProjection(Projections.rowCount())).get(0);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long count(String hqlQuery, Object... parameters) {
		AssertTools.assertNotEmpty(hqlQuery);
		return (Long) getHibernateTemplate().iterate(hqlQuery, parameters).next();
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void create(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().save(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void createOrUpdate(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().saveOrUpdate(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void delete(Class<M> modelClass, Serializable id) {
		AssertTools.assertNotNull(modelClass);
		AssertTools.assertNotNull(id);
		getHibernateTemplate().delete(read(modelClass, id));
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void delete(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().delete(model);
	}

	/** {@inheritDoc} */
	@Override
	public List<?> list(DetachedCriteria criteria) {
		AssertTools.assertNotNull(criteria);
		return getHibernateTemplate().findByCriteria(criteria);
	}

	/** {@inheritDoc} */
	@Override
	public List<?> list(String hqlQuery, Object... parameters) {
		AssertTools.assertNotEmpty(hqlQuery);
		return getHibernateTemplate().find(hqlQuery, parameters);
	}

	/** {@inheritDoc} */
	@Override
	public int execute(final String sqlQuery, final Object... parameters) {
		AssertTools.assertNotEmpty(sqlQuery);
		int nbLines = getHibernateTemplate().executeWithNativeSession(new HibernateCallback<Integer>() {
			@Override
			public Integer doInHibernate(Session session) {
				Query query = session.createSQLQuery(sqlQuery);
				if (parameters != null) {
					for (int i = 0; i < parameters.length; ++i) {
						query.setParameter(i, parameters[i]);
					}
				}
				return Integer.valueOf(query.executeUpdate());
			}
		});
		// This way to execute requests shortcuts the Hibernate process, so we need to refresh the data cache
		getHibernateTemplate().flush();
		getHibernateTemplate().clear();
		return nbLines;
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> boolean exists(Class<M> modelClass, Serializable id) {
		return count(criteriaByProperty(modelClass, AbstractModel.getIdField(modelClass).getName(), id).setProjection(
				Projections.rowCount())) > 0;
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> boolean exists(Class<M> modelClass, String property, Serializable value) {
		return count(criteriaByProperty(modelClass, property, value).setProjection(Projections.rowCount())) > 0;
	}

	/** {@inheritDoc} */
	@Override
	public boolean exists(DetachedCriteria criteria) {
		return count(criteria) > 0;
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> List<M> readList(Class<M> modelClass) {
		AssertTools.assertNotNull(modelClass);
		return GenericsTools.cast(getHibernateTemplate().findByCriteria(
				DetachedCriteria.forClass(modelClass).addOrder(
						Order.asc(AbstractModel.getIdField(modelClass).getName()))));
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> List<M> readList(Class<M> modelClass, String... sortColumns) {
		AssertTools.assertNotNull(modelClass);
		AssertTools.assertNotEmpty(sortColumns);
		DetachedCriteria criteria = DetachedCriteria.forClass(modelClass);
		for (String sortCol : sortColumns) {
			criteria.addOrder(Order.asc(sortCol));
		}
		return GenericsTools.cast(getHibernateTemplate().findByCriteria(criteria));
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> List<M> readList(Class<M> modelClass, int firstResult, int maxResults) {
		AssertTools.assertNotNull(modelClass);
		AssertTools.assertPositive(firstResult);
		AssertTools.assertPositive(maxResults);
		return GenericsTools.cast(getHibernateTemplate().findByCriteria(
				DetachedCriteria.forClass(modelClass).addOrder(
						Order.asc(AbstractModel.getIdField(modelClass).getName())), firstResult, maxResults));
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> List<M> readList(Class<M> modelClass, int firstResult, int maxResults,
			String... sortColumns) {
		AssertTools.assertNotNull(modelClass);
		AssertTools.assertNotEmpty(sortColumns);
		AssertTools.assertPositive(firstResult);
		AssertTools.assertPositive(maxResults);
		DetachedCriteria criteria = DetachedCriteria.forClass(modelClass);
		for (String sortCol : sortColumns) {
			criteria.addOrder(Order.asc(sortCol));
		}
		return GenericsTools.cast(getHibernateTemplate().findByCriteria(criteria, firstResult, maxResults));
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> M read(Class<M> modelClass, Serializable id) {
		AssertTools.assertNotNull(modelClass);
		AssertTools.assertNotNull(id);

		M m = modelClass.cast(getHibernateTemplate().get(modelClass, id));
		if (m == null) {
			throw new ObjectRetrievalFailureException(modelClass, id);
		}
		return m;
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	public <M extends AbstractModel> M read(Class<M> modelClass, String uniqueProperty, Serializable value) {
		List<M> list = (List<M>) list(criteriaByProperty(modelClass, uniqueProperty, value));
		if (list.size() != 1) {
			throw new ObjectRetrievalFailureException(modelClass, value);
		}
		return list.get(0);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void refresh(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().refresh(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void update(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().update(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void evict(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().evict(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> boolean contains(M model) {
		AssertTools.assertNotNull(model);
		return getHibernateTemplate().contains(model);
	}

	/** {@inheritDoc} */
	@Override
	public <M extends AbstractModel> void merge(M model) {
		AssertTools.assertNotNull(model);
		getHibernateTemplate().merge(model);
	}
}