/*
 * #%L
 * IsisFish
 * 
 * $Id: LogLevelUtil.java 4156 2014-12-09 11:27:18Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/logging/LogLevelUtil.java $
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging;

import java.util.ArrayList;
import java.util.List;

/**
 * This classes contains some usefull methods to perform actions on
 * set of LogLevel.
 *
 * To make performant filtering operations on a lots of LogRecord,
 * we use as representation for set of LogLevel an integer
 * as the sum of the {@link LogLevel#mask()} values.
 * 
 * The <code>0</code> value correspond to all levels and is a violating the rule
 * (but how to make a search with no level ?)
 * 
 * to test if a set of LogLevel contains a given LogLevel use
 * {@link #contains(int, int)}
 * 
 * to add a level in a set of LogLevel use {@link #addToSet(int, int)}
 * 
 * to remove a level in a set of LogLevel use {@link #removeFromSet(int, int)}
 * 
 * to obtain all levels contained in a set of LogLevel use
 * {@link #getLogLevels(int)}
 *
 * @author chemit
 * @see LogLevel
 */
public class LogLevelUtil {

    /**
     * Using a performant algorithm to know if a LogLevel is contained in a
     * list of levels using his severity.
     *
     * @param ordinal the ordinal of the level
     * @param levels  the sums of levels
     * @return <code>true</code> if this level is in levels given.
     */
    public static boolean contains(int ordinal, int levels) {
        return (levels >> ordinal) % 2 == 1;
    }

    public static LogLevel[] getLogLevels(int levels) {
        List<LogLevel> result = new ArrayList<LogLevel>();
        for (LogLevel logLevel : LogLevel.values()) {            
            if (contains(logLevel.ordinal(),levels)) {
                // it means present
                result.add(logLevel);
            }
        }
        return result.toArray(new LogLevel[result.size()]);
    }

    /**
     * Compute the sum of levels toInit() value for
     * all given levels (given with their ordinal value).
     * <p>
     * This value will be used for a performant filter process.
     *
     * @param firstOrdinal the first level of levels to use
     * @param ordinals     the level of levels to use
     * @return the sum of the mask() value of each level found, 0 if
     */
    public static int createSet(int firstOrdinal, int... ordinals) {
        List<LogLevel> levelsUsed = new ArrayList<LogLevel>();
        int result = 0;
        LogLevel levelFound;
        levelFound = LogLevel.getLogLevel(firstOrdinal);
        if (levelFound != null) {
            levelsUsed.add(levelFound);
        }
        for (int ordinal : ordinals) {
            levelFound = LogLevel.getLogLevel(ordinal);
            if (levelFound != null && !levelsUsed.contains(levelFound)) {
                levelsUsed.add(levelFound);
            }
        }
        for (LogLevel logLevel : levelsUsed) {
            result += logLevel.mask();
        }
        return result;
    }

    public static int createSet(LogLevel firstLevel, LogLevel... levels) {
        int[] evels = new int[levels.length];
        if (levels.length>0) {
            for (int i = 0; i < levels.length; i++) {                
                evels[i]=levels[i].ordinal();
            }
        }
        return createSet(firstLevel.ordinal(),evels);
    }

    /**
     * add a LogLevel to a set
     *
     * @param ordinal     the ordinal of the level to add
     * @param setOflevels the sum of mask() of some LogLevel
     * @return the new sum (if LogLevel was already in the sum the
     *         <code>levels</code> levels)
     */
    public static int addToSet(int ordinal, int setOflevels) {
        int result = setOflevels;
        if (!contains(ordinal, setOflevels)) {
            result = setOflevels + LogLevel.getLogLevel(ordinal).mask();
        }
        return result;
    }

    /**
     * remove a LogLevel from a set
     *
     * @param ordinal     the ordinal of the level to remove
     * @param setOfLevels the sum of mask() of some LogLevel
     * @return the new sum (if LogLevel was not in the sum the
     *         <code>levels</code> levels)
     */
    public static int removeFromSet(int ordinal, int setOfLevels) {
        int result = setOfLevels;
        if (contains(ordinal, setOfLevels)) {
            result = setOfLevels - LogLevel.getLogLevel(ordinal).mask();
        }
        return result;
    }


}
