/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.email;

import static com.jurismarches.vradi.VradiContext.EMAIL_UI_ENTRY_DEF;
import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import java.awt.Cursor;
import java.awt.event.MouseListener;
import java.io.File;
import java.lang.reflect.Constructor;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.tree.TreePath;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.ui.email.helpers.EmailBeanConstant;
import com.jurismarches.vradi.ui.email.helpers.EmailNavigationTreeHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.offer.OfferListColumnFactory;
import com.jurismarches.vradi.ui.offer.OfferListHandler;
import com.jurismarches.vradi.ui.offer.OfferTable;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.task.AbstractProgressBarUI;
import com.jurismarches.vradi.ui.task.VradiTask;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import com.sun.star.lang.IllegalArgumentException;

/**
 * Handler de l'UI de gestion des email.
 * 
 * @author morin
 */
public class EmailHandler {

    static private final Log log = LogFactory.getLog(EmailHandler.class);

    public EmailPopupUI init(JAXXContext rootContext) {
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(rootContext);

        if (ui == null) {
            JAXXInitialContext context = new JAXXInitialContext()
                    .add(rootContext).add(this);

            ui = new EmailPopupUI(context);

            EMAIL_UI_ENTRY_DEF.setContextValue(rootContext, ui);
        }
        return ui;
    }

    public EmailPopupUI getUI(JAXXContext context) {
        if (context instanceof EmailPopupUI) {
            return (EmailPopupUI) context;
        }
        EmailPopupUI ui = EMAIL_UI_ENTRY_DEF
                .getContextValue(context);
        return ui;
    }

    public SessionsListUI getSessionsListUI(JAXXContext context) {
        if (context instanceof SessionsListUI) {
            return (SessionsListUI) context;
        }
        return context.getContextValue(SessionsListUI.class);
    }

    public VradiStorageService getStorageService() {
        return VradiService.getVradiStorageService();
    }

    public WikittyProxy getProxy() {
        return VradiService.getWikittyProxy();
    }

    protected void close(JAXXContext context) {
        EmailPopupUI ui = getUI(context);
        ui.setVisible(false);
        ui.dispose();
    }

    public List<Session> getSessions(Date date) {
        if (log.isDebugEnabled()){
            log.debug("getSessions for date : " + date);
        }
        List<Session> sessions = null;
        try {
            sessions = getStorageService().getSessions(date);
        } catch (VradiException eee) {
            log.error(eee);
            ErrorDialogUI.showError(eee);
        }

        return sessions;
    }

    /**
     * Creer une nouvelle session.
     */
    public void createNewSession(EmailPopupUI ui) {

        try {
            Session session = getStorageService().createNewSession();

            // FIXME EC20100909 do not update all tree, just current session
            ui.updateTree();
        }
        catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't create new session", ex);
            }
            JOptionPane.showMessageDialog(ui, _("Can't create new session for reason : %s", ex.getMessage()), _("Can't create new session"), JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Lien les formulaires non lié sur la session selectionnée.
     * 
     * @param ui ui
     */
    public void bindEmailsWithForms(final EmailPopupUI ui) {

        VradiTask<Void> task = new VradiTask<Void>(ui) {
            @Override
            public Void doAction() throws Exception {

                try {
                    Session selectedSession = ui.getHelper().getSelectedActiveSession();
                    if (selectedSession == null) {
                        throw new IllegalArgumentException("No active session selected");
                    }

                    Session currentSession = getStorageService().bindForms(selectedSession);

                    // FIXME SLE 01092010 : Save its again from client to fire properties changed
                    VradiService.getWikittyProxy().store(currentSession);

                } catch (WikittyException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("An exception occured while binding form", eee);
                    }
                    JOptionPane.showMessageDialog(ui,
                            _("binding.forms.concurrence.message"));
                }
                return null;
            }

            @Override
            public void doWhenDone() throws Exception {
                // FIXME EC20100909 do not update all tree, just current session
                ui.updateTree();
            }
        };

        task.execute();
    }

    /**
     * Recupere la liste des form pou un sending.
     * 
     * @param sending
     * @return
     */
    protected List<Form> getSendingForms(Sending sending) {
        Set<String> formIds = sending.getForm();
        if (formIds == null) {
            return null;
        }
        List<Form> forms = getProxy().restore(Form.class, new ArrayList<String>(formIds));

        if (log.isDebugEnabled()) {
            log.debug("display " + forms.size() + " forms");
        }
        return forms;
    }

    /**
     * Open UI depending on node type.
     *
     * @param context
     * @param dataProvider
     * @param contentLayout
     * @param content
     * @param node
     */
    protected void openUI(JAXXContext context, VradiDataProvider dataProvider, CardLayout2 contentLayout, JPanel content, VradiTreeNode node) {

        // Get node type
        Class<?> editType = node.getInternalClass();
        Class<? extends EmailContentUI> uiClass = EmailBeanConstant.getUIClass(editType);

        // Verify if instance is existing
        EmailContentUI ui = UIHelper.getContentIfExist(contentLayout, content, uiClass);

        // Get layout identifier
        String contentName = editType.getName();

        if (ui == null) {
            try {
                // Get constructor
                Constructor<? extends EmailContentUI> constructor =
                        uiClass.getConstructor(JAXXContext.class);

                // Invoke instance creation
                ui = constructor.newInstance(context);

            } catch (Exception eee) {
                log.error(eee);
                ErrorDialogUI.showError(eee);
            }

            // Add to content panel
            content.add(ui, contentName);
        }

        // Open UI
        ui.openUI(node);

        // show ui
        contentLayout.show(content, contentName);
    }

    public void displaySessions(JAXXContext context, EmailNavigationTreeHelper helper, VradiTreeNode node) {
        SessionsListUI ui = getSessionsListUI(context);
        
        // Get datas
        List<Sending> sendings = helper.getSendingToDisplay(node);
        if (log.isDebugEnabled()){
            log.debug(sendings.size() + " sendings to display");
        }

        // Remove all sessions ui
        ui.getSessionContent().removeAll();
        for (Sending sending : sendings){

            if (log.isDebugEnabled()){
                log.debug("Sending displayed : " + sending);
            }

            // Get all forms
            List<Form> forms = getSendingForms(sending);

            // Init listTable
            OfferListTableModel offerListTableModel =
                    new OfferListTableModel(true);
            OfferListColumnFactory offerListColumnFactory =
                    new OfferListColumnFactory();

            // Init session ui
            JAXXInitialContext initContext = new JAXXInitialContext()
                    .add(context).add(this).add(sending)
                    .add(offerListTableModel)
                    .add(offerListColumnFactory);

            SessionViewUI viewUI = new SessionViewUI(initContext);
            ui.getSessionContent().add(viewUI);

            // listTable cellRenderer
            final OfferListTableModel.OfferListTableCellRenderer renderer =
                    new OfferListTableModel.OfferListTableCellRenderer();

            OfferTable listTable = viewUI.getListTable();

            OfferListTableModel listTableModel = (OfferListTableModel) listTable.getModel();

            FormPagedResult formPageDTO = new FormPagedResult();
            formPageDTO.setFormsToShow(forms);
            
            listTableModel.setFormPagedResult(formPageDTO);
            listTable.setModel(listTableModel);

            listTable.setDefaultRenderer(Boolean.class, renderer);
            listTable.setDefaultRenderer(Date.class, renderer);
            listTable.setDefaultRenderer(Double.class, renderer);
            listTable.setDefaultRenderer(String.class, renderer);

            // listTable highlighter
            final Highlighter highlighter =
                    HighlighterFactory.createAlternateStriping(    
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);

            listTable.addHighlighter(highlighter);
         }
    }

    public void displayForm(JAXXContext context, EmailNavigationTreeHelper helper, VradiTreeNode node) {

        Session session = helper.getParentSession(node);
        context.setContextValue(session);

        String id = node.getId();

        Form form = getProxy().restore(Form.class, id);

        context.setContextValue(form);
    }

    public <E extends QueryMaker> void addQueryMaker(
            JPanel ui,
            Session session,
            List<Form> forms,
            E selected) {

        for (Form form : forms) {
            addQueryMaker(ui, session, form, selected);
        }
    }

    public <E extends QueryMaker> void addQueryMaker(
            JPanel ui,
            Session session,
            Form form,
            E selected) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmAdd", form.getObjet()),
                _("vradi.email.confirmAddTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION){
            // Prepare creation
            List<Form> formsToBind = new ArrayList<Form>();
            formsToBind.add(form);

            // Get bean in cache
            QueryMaker bean = getRealQueryMaker(selected);
            
            try {
                // Create session if null
                /*if (session == null) {
                    session = VradiService.getVradiStorageService().getLastOpenSession(true);
                }*/

                // Create sendings
                List<Sending> sending = getStorageService().createSending(session, bean, formsToBind);

                // FIXME SLE 01092010 : Save its again from client to fire properties changed
                VradiService.getWikittyProxy().store(sending);

            } catch (VradiException eee) {
                ErrorDialogUI.showError(eee);
                if (log.isErrorEnabled()) {
                    log.error("Cant add sending : " + eee);
                }
            }

//            reloadTree(queryMakerViewUI.getParentContainer(EmailPopupUI.class));
        }
    }

    public <E extends QueryMaker> void removeQueryMaker(
            JPanel ui,
            Session session,
            List<Form> forms,
            Class<E> clazz,
            Object[] selecteds) {

        for (Form form : forms) {
            removeQueryMaker(ui, session, form, clazz, selecteds);
        }
    }
    @SuppressWarnings({"unchecked"})
    public <E extends QueryMaker> void removeQueryMaker(
            JPanel ui,
            Session session,
            Form form,
            Class<E> clazz,
            Object[] selecteds) {

        int yes = JOptionPane.showConfirmDialog(
                ui,
                _("vradi.email.confirmDelete", form.getObjet()),
                _("vradi.email.confirmDeleteTitle"),
                JOptionPane.YES_NO_OPTION);

        if (yes == JOptionPane.YES_OPTION){

            for (Object selected : selecteds) {

                try{
                    List<Sending> sending = getStorageService().removeAllSending(session, form, getRealQueryMaker((E) selected));

                    // FIXME SLE 01092010 : Save its again from client to fire properties changed
                    VradiService.getWikittyProxy().store(sending);

                } catch (VradiException eee){
                    ErrorDialogUI.showError(eee);
                    log.error("Cant add sending : " + eee);
                }

            }
//            reloadTree(queryMakerViewUI.getParentContainer(EmailPopupUI.class));
        }
    }

    protected void reloadTree(EmailPopupUI ui) {

        // Refresh
        EmailNavigationTreeHelper helper = ui.getHelper();

        // Get selected path
        TreePath[] selectionPaths = helper.getSelectionModel().getSelectionPaths();

        // Update datas
        ui.updateTree();

        // Select
        if (selectionPaths != null){
            helper.getSelectionModel().setSelectionPaths(selectionPaths);
        }
    }

    protected <E extends QueryMaker> QueryMaker getRealQueryMaker(E toConvert){
        String wikittyId = toConvert.getWikittyId();
        return getProxy().restore(toConvert.getClass(), wikittyId);
    }

//    public Map<VradiClientDTO, Set<String>> getAllClientsAndUsers() {
//        SortedMap<VradiClientDTO, Set<String>> clientsAndUsers = new TreeMap<VradiClientDTO, Set<String>>(VradiComparators.CLIENT_COMPARATOR);
//
//        // map client id = user ids
//        Map<String, Set<String>> clientsIdAndUsers = new TreeMap<String, Set<String>>();
//
//        List<VradiClientDTO> clients = getAllClients();
//        List<User> users = VradiContext.getUsersInEntryDef();
//
//        for (VradiClientDTO client : clients) {
//            Set<String> ids = new HashSet<String>();
//            clientsAndUsers.put(client, ids);
//            clientsIdAndUsers.put(client.getWikittyId(), ids);
//        }
//
//        for (User user : users) {
//            clientsIdAndUsers.get(user.getClient()).add(user.getWikittyId());
//        }
//
//        return clientsAndUsers;
//    }

    /**
     * Called to send only one simple sending.
     * 
     * @param ui parent ui
     * @param sending sending to send
     */
    public void sendSending(SessionViewUI ui, final Sending sending){
        Object[] options = {_("vradi.email.confirmSendSending.optionYes"),
                            _("vradi.email.confirmSendSending.optionNo"),
                            _("vradi.email.confirmSendSending.optionOtherMail")};
        
        int i = JOptionPane.showOptionDialog(ui,
                _("vradi.email.confirmSendSending", getQueryMakerName(sending)),
                _("vradi.email.confirmSendSendingTitle"),
                JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
                null,
                options,
                options[2]);

        String email = null;

        if (i == JOptionPane.CANCEL_OPTION) {

            // Ask email to send
            email = JOptionPane.showInputDialog(
                ui,
                _("vradi.email.confirmSendSending.askMail"),
                _("vradi.email.confirmSendSending.askMailTitle"),
                JOptionPane.PLAIN_MESSAGE);

            i = JOptionPane.OK_OPTION;
        }

        if (i == JOptionPane.OK_OPTION) {

            // Get session paragraph
            Session session = getSessionBySending(sending.getWikittyId());
            final String sessionParagraph = session.getParagraph();

            final String finalEmail = email;
            final EmailPopupUI finalUI = ui.getParentContainer(EmailPopupUI.class);
            VradiTask<Void> sendSendingTask = new VradiTask<Void>(ui, "vradi.email.sendSending"){
                
                @Override
                public Void doAction() throws VradiException {
                    Sending sendingResult = null;
                    try {
                        sendingResult = getStorageService().sendMessage(sending.getWikittyId(), sessionParagraph, finalEmail);
                    } catch (VradiException eee) {
                        JOptionPane.showMessageDialog(finalUI, _("vradi.email.sendSending.sendError",
                                eee.getMessage()), _("vradi.email.sendSession.error"),
                                JOptionPane.ERROR_MESSAGE);

                        // Set session status to error
                        updateSendingStatus(sendingResult, VradiConstants.SendingStatus.ERROR);
                    }
                    return null;
                }

                @Override
                public void doWhenError(Exception eee){
                    updateSendingStatus(sending, VradiConstants.SendingStatus.ERROR);
                }
            };
            sendSendingTask.execute();
        }
    }

    /**
     * Called to send a full session.
     * 
     * Don't call without VradiSessionDTO selection !
     * 
     * @param ui parent ui
     * @return {@code true} if task has been 
     */
    public boolean send(final EmailPopupUI ui) {
        final Session session = ui.getHelper().getSelectedActiveSession();

        // ask user for confirmation
        int confirm = JOptionPane.showConfirmDialog(ui,
                _("vradi.email.confirmSendSession"),
                _("vradi.email.confirmSendSessionTitle"),
                JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
        if (confirm != JOptionPane.YES_OPTION) {
            return false;
        }

        // Check if emails and templates exists
        if (!checkAllSending(ui, session)){
            return false;
        }

        // Display progess bar with cancel button
        final AbstractProgressBarUI pb = new AbstractProgressBarUI(ui) {
            /** serialVersionUID. */
            private static final long serialVersionUID = 5926558779822425487L;

            @Override
            public void cancel() {
                try {
                    VradiService.getVradiStorageService().stopSentMail(session);
                    this.dispose();
                } catch (VradiException eee) {
                    log.info("Cant stop sending emails : ", eee);
                    ErrorDialogUI.showError(eee);
                }
                ui.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            }
        };

        if (log.isInfoEnabled()) {
            SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
            log.info("Sending session : " + session.getNum() + " - " +
                format.format(session.getSessionDate()) + " status : " +
                VradiConstants.SessionStatus.getStatus(session.getStatus()).getDescription());
        }
        

        // premiere tache de génération des PDF
        VradiTask<Boolean> pdfGenerationAndSendingTask = new VradiTask<Boolean>(
                ui, pb, n_("vradi.email.generation")) {

            @Override
            public Boolean doAction() throws Exception {
                try {
                    
                    VradiService.getVradiStorageService().generatePDFForSession(
                            session.getWikittyId());

                } catch (Exception eee){
                    if (log.isErrorEnabled()) {
                        log.error(getErrorMsg(), eee);
                    }
                    doWhenError(eee);
                    ErrorDialogUI.showError(eee);
                    return false;
                }
                return true;
            }

            @Override
            public void doWhenDone() throws Exception {

                // If pdf generation is succefuly done
                Boolean result = get();
                if (result != null && result) {

                    if (log.isInfoEnabled()) {
                        log.info("Start sending mails");
                    }

                    VradiTask<Void> sentTask = new VradiTask<Void>(ui, pb, n_("vradi.email.sendSession")){

                        @Override
                        public Void doAction() throws Exception {
                            try {
                                VradiService.getVradiStorageService().sendMessages(session.getWikittyId());
                            } catch (VradiException eee) {
                                JOptionPane.showMessageDialog(ui, _("vradi.email.sendSending.sendError",
                                        eee.getMessage()), _("vradi.email.sendSession.error"),
                                        JOptionPane.ERROR_MESSAGE);

                                // Set session status to error
                                updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
                            }
                            return null;
                        }

                        @Override
                        public void doWhenError(Exception eee){
                            updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
                        }
                    };

                    sentTask.execute();
                }
            }

            @Override
            public void doWhenError(Exception eee){
                updateSessionStatus(session, VradiConstants.SessionStatus.ERROR);
            }
        };
        
        // Executes task
        pdfGenerationAndSendingTask.execute();
        
        return true;
    }

    protected boolean checkAllSending(EmailPopupUI ui, Session session){

        List<String> alreadyAskFormType = new ArrayList<String>();

        for (Sending sending : getSendingList(session)){

            // Check users email
            String userId = sending.getUser();
            User user = getProxy().restore(User.class, userId);
            if (user != null){
                String emailUser = user.getEmail();
                if (emailUser == null || emailUser.isEmpty()){
                    Client client = VradiService.getWikittyProxy().restore(Client.class, user.getClient());
                    boolean abord = askAbordSending(ui, _("vradi.email.askUserEmailEmpty", user.getName(), client.getName()));
                    if (abord){

                        // Set to error status
                        updateSendingStatus(sending, VradiConstants.SendingStatus.ERROR);

                        return false;
                    }
                }
            }

            // Check PDF generation
            List<String> formIds = new ArrayList<String>(sending.getForm());
            List<Form> forms = getProxy().restore(Form.class, formIds);
            for (Form form : forms) {

                String extensionName = null;
                for(String ext : form.getExtensionNames()) {
                    if(!ext.equals(Infogene.EXT_INFOGENE)
                            && !ext.equals(Form.EXT_FORM)) {
                        extensionName = ext;
                        break;
                    }
                }
                VradiStorageService storageService = VradiService.getVradiStorageService();
                WikittyExtension extension = null;
                try {
                    extension = storageService.getFormType(extensionName);
                } catch (VradiException eee) {
                    log.error("Cant get formType for form " + form.getObjet() +
                            " with extention : " + extensionName, eee);
                    ErrorDialogUI.showError(eee);
                }

                String template = VradiHelper.getFormTypeTemplate(extension);

                // FIXME EC20100614 correct this code to not download file
                // just verify file existence
                // File file = storageService.getTemplate(extensionName, template);
                boolean fileExists = false;
                try {
                    File file = VradiService.getFileService().downloadTemplate(extensionName, template);
                    if (file != null) {
                        fileExists = file.isFile();
                    }
                }
                catch (VradiException eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't download template", eee);
                        ErrorDialogUI.showError(eee);
                    }
                }

                if (!fileExists && !alreadyAskFormType.contains(extension.getName())){
                    boolean abord = askAbordSending(ui, _("vradi.email.askModelEmpty", extension.getName()));
                    if (abord){
                        return false;
                    }
                    alreadyAskFormType.add(extension.getName());
                }
            }
        }
        return true;
    }

    protected boolean askAbordSending(EmailPopupUI ui, String msg) {
        int i = JOptionPane.showConfirmDialog(ui,
                msg,
                _("vradi.email.askAbordSendingTitle"),
                JOptionPane.YES_NO_OPTION);

        return i != JOptionPane.YES_OPTION;
    }

    public Sending updateSendingStatus(Sending sending, VradiConstants.SendingStatus status) {
        if (sending == null) {
            return null;
        }

        Sending sendingRestored = getProxy().restore(Sending.class, sending.getWikittyId());

        sendingRestored.setStatus(status.getValue());

        log.info("Saving sending " + sendingRestored.getWikittyId() + " with status " + status.getDescription());
        return getProxy().store(sendingRestored);

    }

    public Session updateSessionStatus(Session session, VradiConstants.SessionStatus status) {
        if (session == null) {
            return null;
        }

        Session sessionRestored = getProxy().restore(Session.class, session.getWikittyId());

        sessionRestored.setStatus(status.getValue());
        sessionRestored.setSessionDate(new Date());

        log.info("Saving session " + sessionRestored.getWikittyId() + " with status " + status.getDescription());
        return getProxy().store(sessionRestored);

    }

    protected String getQueryMakerName(Sending sending){
        String userId = sending.getUser();
        String groupId = sending.getGroup();
        if (userId != null){
            return getProxy().restore(User.class, userId).getName();
        } else if (groupId != null){
            return getProxy().restore(Group.class, groupId).getName();
        }
        return "";
    }

    /**
     * Close selected session in tree.
     *
     * Don't call it if no session is selected
     *
     * @param ui parent ui
     */
    public void cancelSession(EmailPopupUI ui) {
        Session session = ui.getHelper().getSelectedActiveSession();
        if (session != null) {

            int i = JOptionPane.showConfirmDialog(
                    ui, _("vradi.email.confimCloseSession"),
                    _("vradi.email.confimCloseSession.titles"),
                    JOptionPane.YES_NO_OPTION);

            if (i == JOptionPane.YES_OPTION){
                updateSessionStatus(session, VradiConstants.SessionStatus.CANCELED);
            }
        }
    }

    // Update all sending reception proof recursivly (for clients and groups)
    public void updateReceptionProof(Session session, Sending sending, boolean isSelected){
        sending.setReceptionProof(isSelected);
//        String clientId = sending.getClient();
//        List<User> users = getUserOfClient(clientId);
//        for (User user : users) {
//            Sending userSending = getSendingForUser(session, user.getWikittyId());
//
//            updateReceptionProof(session, userSending, isSelected);
//        }
        String groupId = sending.getGroup();
        Group group = getProxy().restore(Group.class, groupId);
        if (group != null) {
            List<String> userIds = new ArrayList<String>(group.getUser());
            List<User> users = getProxy().restore(User.class, userIds);
            for (User user : users) {
                Sending userSending = getSendingForUser(session, user.getWikittyId());

                updateReceptionProof(session, userSending, isSelected);
            }
//            List<String> clientIds = new ArrayList<String>(group.getClient());
//            List<Client> clients = getProxy().restore(Client.class, clientIds);
//            for (Client groupClient : clients) {
//                Sending clientSending = getSendingForClient(session, groupClient.getWikittyId());
//
//                updateReceptionProof(session, clientSending, isSelected);
//            }
        }
        getProxy().store(sending);
    }

    // Update all sending paragraphs recursivly (for clients and groups)
    public void updateParagraph(Session session, Sending sending, String text){
        sending.setParagraph(text);
//        String clientId = sending.getClient();
//        Client client = getProxy().restore(Client.class, clientId);
//        if (client != null) {
//            List<User> users = getUserOfClient(clientId);
//            for (User user : users) {
//                Sending userSending = getSendingForUser(session, user.getWikittyId());
//                String userParagraph = userSending.getParagraph();
//
//                // Update only if paragraph is empty
//                if (userParagraph == null || userParagraph.isEmpty()){
//                    updateParagraph(session, userSending, text);
//                }
//            }
//        }
        String groupId = sending.getGroup();
        Group group = getProxy().restore(Group.class, groupId);
        if (group != null) {
            List<String> userIds = new ArrayList<String>(group.getUser());
            List<User> users = getProxy().restore(User.class, userIds);
            for (User user : users) {
                Sending userSending = getSendingForUser(session, user.getWikittyId());
                String userParagraph = userSending.getParagraph();

                // Update only if paragraph is empty
                if (userParagraph == null || userParagraph.isEmpty()){
                    updateParagraph(session, userSending, text);
                }
            }
        }
        getProxy().store(sending);
    }

    protected List<Sending> getSendingList(Session session) {
        if (session == null || session.getSending() == null) {
            return new ArrayList<Sending>();
        }
        List<String> sendingIds = new ArrayList<String>(session.getSending());
        return getProxy().restore(Sending.class, sendingIds);
    }

    protected List<User> getUserOfClient(String clientId) {
        // Get all sending with form
        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Client.EXT_CLIENT);

        query.contains(User.FQ_FIELD_USER_CLIENT, getStringAsList(clientId));

        Criteria criteria = query.criteria();
        PagedResult<User> nodes = getProxy()
                .findAllByCriteria(User.class, criteria);
        return nodes.getAll();
    }

    protected Sending getSendingForUser(Session session, String userId) {
        List<Sending> sendings = getSendingList(session);
        for (Sending sending : sendings) {
            if (sending.getUser().equals(userId)) {
                return sending;
            }
        }
        return null;
    }

//    protected Sending getSendingForClient(Session session, String clientId) {
//        List<Sending> sendings = getSendingList(session);
//        for (Sending sending : sendings) {
//            if (sending.getClient().equals(clientId)) {
//                return sending;
//            }
//        }
//        return null;
//    }

    protected Session getSessionBySending(String sendingId) {
        // Get all sending with form
        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        query.contains(Session.FQ_FIELD_SESSION_SENDING, getStringAsList(sendingId));

        Criteria criteria = query.criteria();
        PagedResult<Session> nodes = getProxy()
                .findAllByCriteria(Session.class, criteria);
        return nodes.getFirst();
    }

    protected List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }
}

