/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.admin;

import static org.nuiton.i18n.I18n._;

import java.awt.*;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.*;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileFilter;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.VradiDataService;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.ui.admin.helpers.AdminBeanConstant;
import com.jurismarches.vradi.ui.admin.helpers.AdminNavigationTreeHelper;
import com.jurismarches.vradi.ui.admin.loadors.ClientsAndUsersNodeLoadors;
import com.jurismarches.vradi.ui.admin.loadors.FormTypeNodeLoadors;
import com.jurismarches.vradi.ui.admin.loadors.GroupsNodeLoadors;
import com.jurismarches.vradi.ui.admin.loadors.StreamNodeLoadors;
import com.jurismarches.vradi.ui.admin.models.FieldSelectModel;
import com.jurismarches.vradi.ui.admin.models.FieldTypeModel;
import com.jurismarches.vradi.ui.admin.models.TemplateFieldTableModel;
import com.jurismarches.vradi.ui.admin.models.XmlStreamFieldTableModel;
import com.jurismarches.vradi.ui.admin.popups.AdminRequestUI;
import com.jurismarches.vradi.ui.admin.renderers.JListCellRenderer;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.VradiComparators;
import com.jurismarches.vradi.ui.offer.OfferListUI;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.HBox;
import jaxx.runtime.swing.VBox;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdom.Element;
import org.nuiton.util.FileUtil;
import org.nuiton.wikitty.*;
import org.nuiton.wikitty.search.Search;
import org.webharvest.gui.Ide;

import com.jurismarches.vradi.VradiConstants.XmlStreamConfig;
import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.ui.offer.OfferListColumnFactory;
import com.jurismarches.vradi.ui.offer.OfferListHandler;
import com.jurismarches.vradi.ui.VradiMainUI;
import com.jurismarches.vradi.ui.task.VradiTask;
import com.jurismarches.vradi.ui.admin.content.AdminClientUI;
import com.jurismarches.vradi.ui.admin.content.AdminFormTypeUI;
import com.jurismarches.vradi.ui.admin.content.AdminGroupUI;
import com.jurismarches.vradi.ui.admin.content.AdminStatusUI;
import com.jurismarches.vradi.ui.admin.content.AdminXmlStreamUI;
import com.jurismarches.vradi.ui.admin.content.FieldSelectUI;
import com.jurismarches.vradi.ui.admin.content.RequestSelectUI;
import com.jurismarches.vradi.ui.admin.popups.TemplateFieldBindingUI;
import com.jurismarches.vradi.ui.admin.popups.XmlStreamConfigUI;
import com.jurismarches.vradi.ui.offer.editors.JListCellEditor;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.helpers.XmlStreamHelper;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.search.SearchHandler;
import com.jurismarches.vradi.ui.search.SearchUI;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;

/**
 * @author letellier
 */
public class AdminHandler {

    static private final Log log = LogFactory.getLog(AdminHandler.class);

    /** Max element in xml stream sample tooltip. */
    static public final int TOOLTIP_ELEMENT_MAX_CHAR_NB = 500;

    public AdminPopupUI init(JAXXContext rootContext) {
        AdminPopupUI adminPopupUI = VradiContext.ADMIN_UI_ENTRY_DEF
                .getContextValue(rootContext);

        if (adminPopupUI == null) {
            JAXXInitialContext context = new JAXXInitialContext()
                    .add(rootContext).add(this);

            VradiMainUI mainUI = VradiContext.MAIN_UI_ENTRY_DEF
                    .getContextValue(context);

            adminPopupUI = new AdminPopupUI(context, mainUI);

            VradiContext.ADMIN_UI_ENTRY_DEF.setContextValue(rootContext,
                    adminPopupUI);
        }
        
        return adminPopupUI;
    }

    protected AdminPopupUI getUI(JAXXContext context) {
        if (context instanceof AdminPopupUI) {
            return (AdminPopupUI) context;
        }
        AdminPopupUI ui = VradiContext.ADMIN_UI_ENTRY_DEF
                .getContextValue(context);
        return ui;
    }

    /**
     * Close ui (release resources).
     * 
     * @param context context
     */
    protected void close(JAXXContext context) {
        AdminPopupUI ui = getUI(context);
        ui.setVisible(false);
        ui.dispose();
    }

    /**
     * Open UI depending on node type.
     * 
     * @param context
     * @param dataProvider
     * @param contentLayout
     * @param content
     * @param node
     */
    protected void openUI(JAXXContext context, VradiDataProvider dataProvider,
                          CardLayout2 contentLayout, JPanel content,
                          VradiTreeNode node) {

        // Get node type
        Class<?> editType = node.getInternalClass();
        String id = node.getId();

        // If it's category node
        if (editType.equals(String.class)) {
            Class uiClass = AdminBeanConstant.getUIClass(id);
            showUI(context, contentLayout, content, null, uiClass);
        }
        else {
            // If it's entity node
            Class uiClass = AdminBeanConstant.getUIClass(editType);
            Object bean = AdminBeanConstant.getConstantByBeanClass(editType).getBean(dataProvider, id);

            // FIXME SLE 16/07/10 : Hack, find a better way
            if (User.class.equals(editType)) {
                User user = (User)bean;
                if (user.getClient() == null) {
                    String clientId = node.getParent().getParent().getId();
                    user.setClient(clientId);
                }
            }

            showUI(context, contentLayout, content, bean, uiClass);
        }
    }

    /**
     * Create by introspection content ui.
     * 
     * @param <B>
     * @param context
     * @param contentLayout
     * @param content
     * @param bean
     * @param uiClass
     */
    protected <B> void showUI(JAXXContext context, CardLayout2 contentLayout,
                              JPanel content, B bean,
                              Class<? extends AdminContentUI> uiClass) {

        // Verify if instance is existing
        AdminContentUI ui = UIHelper.getContentIfExist(contentLayout, content,
                uiClass);

        // Get layout identifier
        String contentName = uiClass.getName();

        if (ui == null) {
            try {
                // Get constructor
                Constructor<? extends AdminContentUI> constructor =
                        uiClass.getConstructor(JAXXContext.class);

                // Invoke instance creation
                ui = constructor.newInstance(context);
            } catch (Exception eee) {
                log.error(eee);
                ErrorDialogUI.showError(eee);
            }

            // Add to content panel
            content.add(ui, contentName);
        }

        // Attach bean
        //FIXME JC 27/08/2010 set bean to null to fire property change. If not, save button does not enable.
        ui.setBean(null);
        ui.setBean(bean);

        // show ui
        contentLayout.show(content, contentName);
    }

    /**
     * Save group in wikitty.
     * 
     * @param group group to save
     * @return saved group
     */
    public Group saveGroup(Group group) {

        try {
            VradiService.getVradiStorageService().archiveQueries(group);
        } catch (VradiException eee) {
            log.error("Cant archive query : ", eee);
            ErrorDialogUI.showError(eee);
        }

        if (log.isDebugEnabled()) {
            log.debug("Save group '" + group.getName() + "'");
            if (group.getQueries() != null) {
                for (String query : group.getQueries()) {
                    log.debug("Save group '" + group.getName()
                            + "' with query : " + query);
                }
            }
            Set<String> users = group.getUser();
            if (users != null) {
                for (String user : users) {
                    log.debug("Saving group " + group.getName()
                            + " with user : " + user);
                }
            }
        }

        // save group
        Group newGroup = VradiService.getWikittyProxy().store(group);

        return newGroup;
    }

    /**
     * Save groups in wikitty.
     * 
     * @param groups groups
     * @return groups
     */
    protected Collection<Group> saveGroups(List<Group> groups) {

        List<Group> newGroups = new ArrayList<Group>();

        for (Group group : groups) {
            newGroups.add(saveGroup(group));
        }

        return newGroups;
    }

    /**
     * Save client in wikitty.
     * 
     * @param client client
     * @return client
     */
    protected Client saveClient(Client client) {
        // save client
        Client newClient = VradiService.getWikittyProxy().store(client);

        return newClient;
    }

    /**
     * Save user in wikitty.
     * 
     * @param user user
     * @return user
     */
    protected User saveUser(User user) {
        // save client
        User newUser = VradiService.getWikittyProxy().store(user);
        
        return newUser;
    }

    /**
     * Save stream in wikitty.
     * 
     * @param <E>
     * @param xmlStream
     * @param bindings
     * @return
     */
    protected <E extends XmlStream> E saveStream(E xmlStream,
            List<XmlFieldBinding> bindings) {
        
        // save xmlStream
        List<XmlFieldBinding> newXmlFieldBinding =
                VradiService.getWikittyProxy().store(bindings);
        
        xmlStream.clearXmlFieldBinding();
        for (XmlFieldBinding binding : newXmlFieldBinding) {
            xmlStream.addXmlFieldBinding(binding.getWikittyId());
        }
        
        E newXmlStream = VradiService.getWikittyProxy().store(xmlStream);

        return newXmlStream;
    }
    
    public Client saveClient(
            Client client, Collection<Group> groups) {

        try {
            // save client
            client = saveClient(client);
            
            // save associated groups
            if (groups != null && !groups.isEmpty()) {
                saveGroups(new ArrayList<Group>(groups));
            }

        } catch (Exception e) {
            log.error("Cant save client : ", e);
        }

        return client;
    }
    
    public User saveUser(User user, Client client, Collection<Group> groups) {
        try {

            // save user
            user = saveUser(user);
            
            // save associated client
            if (client != null) {
                saveClient(client);
            }
            
            // save associated groups
            if (groups != null && !groups.isEmpty()) {
                if (log.isDebugEnabled()){
                    log.debug("Save groups");
                }
                saveGroups(new ArrayList<Group>(groups));
            }

        } catch (Exception e) {
            log.error("Cant save user : ", e);
        }

        return user;
    }
    
    /**
     * Export all clients database (with groups, users, clients) as CSV.
     * 
     * Used in {@link com.jurismarches.vradi.ui.admin.nav.AdminClientNavUI}.
     * 
     * @param adminNavUI parent ui
     */
    public void exportClientUsersDBCSV(AdminNavUI adminNavUI) {

        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminClient.exportGroupTitle"));
            chooser.setApproveButtonText(_("vradi.adminClient.exportButtonText"));
            int returnVal = chooser.showOpenDialog(adminNavUI);

            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();

                if (log.isDebugEnabled()) {
                    log.debug("Exporting groups to : " + selectedFile.getAbsolutePath());
                }

                // export
                VradiStorageService vradiStorageService = VradiService.getVradiStorageService();
                String vscContent = vradiStorageService.exportClientDB();
                
                // write content into file
                FileUtils.writeStringToFile(selectedFile, vscContent);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export groups", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export groups", eee);
            }
        }
    }

    /**
     * Import CSV file.
     * 
     * Can be used to import clients, groups or thesaurus.
     * 
     * @param adminNavUI parent ui
     */
    public void importClientUsersDBCSV(AdminNavUI adminNavUI) {
        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminClient.importTitle"));
            chooser.setApproveButtonText(_("vradi.adminClient.importButtonText"));
            int returnVal = chooser.showOpenDialog(adminNavUI);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();
    
                if (log.isDebugEnabled()) {
                    log.debug("Importing file : " + selectedFile.getAbsolutePath());
                }

                //upload file on server
                String uri = VradiService.getFileService().uploadTempFile(selectedFile);
                // import from file on server
                VradiService.getVradiStorageService().importAsCSV(uri);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
        }
    }

    public boolean answerToSave(AdminContentUI<?> content) {
        if (content == null) {
            return true;
        }
        VradiTreeNode selected = null;
        if (content.getHelper() != null || content.getHelper().getSelectedNode() != null) {
            selected = content.getHelper().getSelectedNode();
        }
        return answerToSave(content, selected);
    }
    
    public boolean answerToSave(AdminContentUI<?> content, VradiTreeNode oldNode) {
        if (content == null) {
            return true;
        }
        if (content.isModified()) {
            int n = JOptionPane.showConfirmDialog(content,
                    _("vradi.admin.saveAnswer")
                    + VradiHelper.getEntityName(content.getBean())
                    + " ?", _("vradi.admin.saveTitle"),
                    JOptionPane.YES_NO_CANCEL_OPTION);

            switch (n) {
            case JOptionPane.NO_OPTION:
                // annule les modifications
                cancel(content, oldNode, false);
                break;
            case JOptionPane.OK_OPTION:
                // sauvegarde les modifications
                content.setModified(false);
                content.save();

                if (oldNode != null) {
                    content.getHelper().refresh(oldNode.getParent());
                }
                break;
            case JOptionPane.CANCEL_OPTION:
                // annule le changement d'onglet
                return false;
            }
        }
        return true;
    }

    public boolean confirmDeletion(AdminContentUI<?> content) {
        int confirm = JOptionPane.showConfirmDialog(content,
                _("vradi.admin.confirmDelete", VradiHelper.getEntityName(content.getBean())),
                _("vradi.admin.deleteTitle"), JOptionPane.YES_NO_OPTION);

        return confirm != JOptionPane.NO_OPTION;
    }
    
    public <B> void delete(AdminContentUI<B> context, AdminNavigationTreeHelper helper,
            B bean) {
        if (!confirmDeletion(context)) {
            return;
        }
        
        try {

            VradiTreeNode selectedNode = helper.getSelectedNode();

            // select parent node
            helper.selectNode(selectedNode.getParent());

            //Clear the bean that was editing as it does not exist anymore
            context.setModified(false);
            context.setBean(null);

            // deleteEntity is too generic to apply business rules on service layer
            BusinessEntity entity = (BusinessEntity) bean;
            String id = entity.getWikittyId();

            //When deleting client, need to also delete users before
            if (bean instanceof Client) {
                deleteUsersForClient(id);
            }

            //When deleting user, need to remove it from groups first
            if (bean instanceof User) {
                deleteUserFromGroups(id);
            }
            
            // check if the entity is to be delete or if it is a newly created one
            // which have never been saved. modified is null if newly created
            Object modified = VradiService.getWikittyProxy().restore(id);
            if (modified != null) {
                //Delete in database, the event will reload tree
                VradiService.getWikittyProxy().delete(id);
            } else {
                //Remove from tree by hand when not in database
                helper.removeNode(selectedNode);
            }

        } catch (Exception eee) {
            log.error(eee.getMessage(), eee);
            ErrorDialogUI.showError(eee);
        }
    }

    //FIXME JC 25/08/2010 Put that in service
    /**
     * Delete all the users of a client
     * @param id the client's wikittyId
     */
    protected void deleteUsersForClient(String id) {
        List<User> users = VradiService.getVradiDataService().findClientUsers(id);
        for(User user:users){
            deleteUserFromGroups(user.getWikittyId());
            VradiService.getWikittyProxy().delete(user.getWikittyId());
        }
    }

    //FIXME JC 25/08/2010 Put that in service
    /**
     * Remove the user from all the groups he is in.
     * @param wikittyId the user's wikittyId
     */
    protected void deleteUserFromGroups(String wikittyId) {
        ClientManager manager = new ClientManager();
        try {
            List<Group> groups = manager.getGroupsForUser(wikittyId);
            for (Group group:groups) {
                group.removeUser(wikittyId);
            }
        } catch (VradiException eee) {
            log.error("An error occured", eee);
        }
    }

    protected <B> void cancel(AdminContentUI<B> content,
                              VradiTreeNode oldNode,
                              boolean trigerredByCancelButton) {
        int confirm = -1;
        if(trigerredByCancelButton) {
            confirm = JOptionPane.showConfirmDialog(content,
                    _("vradi.admin.cancel.message"),
                    _("vradi.admin.cancel.title"),
                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }
        
        if(!trigerredByCancelButton || confirm == JOptionPane.YES_OPTION) {
            try {

                // Retrieve bean concerned
                B bean = content.getBean();

                // Get id
                String id = null;

                // Prepare data for restoring bean
                VradiDataProvider vradiDataProvider = content.getHelper().getDataProvider();
                Class<? extends AdminContentUI> uiClass = content.getClass();
                AdminBeanConstant beanClass = AdminBeanConstant.getConstantByUIClass(uiClass);

                if (bean instanceof BusinessEntity) {
                    cancelIfCreation(content, oldNode, (BusinessEntity) bean);
                } else if (bean instanceof WikittyExtension) {
                    id = ((WikittyExtension) bean).getId();
                }

                content.setModified(false);
                if (id != null) {
                    Object beanCanceled = beanClass.getBean(vradiDataProvider, id);

                    // Push it in UI
                    //FIXME JC 25/08/2010 bean is the same (because of cache ?) so no property change fired. Set bean to null before to force fire.
                    content.setBean(null);
                    content.setBean((B) beanCanceled);
                }

                if (oldNode != null) {
                    content.getHelper().refresh(oldNode.getParent());
                }

            } catch (Exception eee) {
                log.error("Cant cancel editing", eee);
                ErrorDialogUI.showError(eee);
            }
        }
    }

    protected <B> void cancelIfCreation(AdminContentUI<B> content, VradiTreeNode oldNode, BusinessEntity bean){

        VradiDataProvider vradiDataProvider = content.getHelper().getDataProvider();
        Class<? extends AdminContentUI> uiClass = content.getClass();
        AdminBeanConstant beanClass = AdminBeanConstant.getConstantByUIClass(uiClass);

        String id = bean.getWikittyId();

        // Creation if beanCanceled is null (not yet in base)
        Object beanCanceled = beanClass.getBean(vradiDataProvider, id);
        if (beanCanceled == null) {

            // Get helper
            AdminNavigationTreeHelper helper = content.getHelper();

            if (oldNode != null) {
                // Select parent
                helper.refresh(oldNode.getParent());
            }

            // Remove creation node
//            helper.removeNode(oldNode);
            content.setModified(false);
        }
    }

    public ComboBoxModel getTypesModel() {
        ComboBoxModel model = new FieldTypeModel();
        return model;
    }

    // REQUEST
                     
    // From show imply
    public void openRequest(final JAXXContext rootContext, Thesaurus thesaurus) {

        // Creating request bean
        String request = ThesaurusDataHelper.getRequestName(thesaurus);
        QueryBean requestBean = null;
        try {
            requestBean = new QueryBean(request);
        } catch (IOException eee) {
            log.error("Cant parse query : " + request, eee);
        }

        // Open request ui
        openRequest(rootContext, requestBean, thesaurus, true);
    }

    public void openRequest(final JAXXContext rootContext, QueryBean request) {
        openRequest(rootContext, request, null, false);
    }
    
    public void openRequest(final JAXXContext rootContext, QueryBean request, Thesaurus thesaurus, boolean isShowImply) {
        final AdminRequestUI requestUI = createAdminRequestUI(rootContext, request, thesaurus);

        SearchHandler searchHandler = requestUI.getContextValue(SearchHandler.class);
        SearchUI searchUI = requestUI.getSearchUI();
        OfferListTableModel offerListTableModel = requestUI.getContextValue(OfferListTableModel.class);
        String query = requestUI.getRequest();
        
        // execute search
        final FormPagedResult formPageDTO = new FormPagedResult();
        searchHandler.executeQuery(searchUI, searchUI.getCriterias(), formPageDTO, null, query);
        offerListTableModel.setFormPagedResult(formPageDTO);

        requestUI.addPropertyChangeListener(AdminRequestUI.PROPERTY_REQUEST,
            new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    String newValue = (String) evt.getNewValue();
                    RequestSelectUI ui = (RequestSelectUI) rootContext;
                    QueryBean bean = ui.getBean();
                    ui.setBean(bean.setQuery(newValue));
                }
            });

        if (isShowImply) {
            requestUI.setModal(true);
            requestUI.getSearchUI().getCriterias().setVisible(false);
            requestUI.getSearchPanel().setVisible(false);
            requestUI.getSearchUI().executeQuery();
        }

        requestUI.setVisible(true);
    }

    public AdminRequestUI createAdminRequestUI(final JAXXContext rootContext, QueryBean request, Thesaurus thesaurus) {
        final JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        final SearchHandler searchHandler = new SearchHandler();
        final OfferListTableModel offerListTableModel = new OfferListTableModel();
        final OfferListColumnFactory offerListColumnFactory = new OfferListColumnFactory();

        context.add(searchHandler);
        context.add(offerListTableModel);
        context.add(offerListColumnFactory);

        final AdminRequestUI requestUI = new AdminRequestUI(context);

        // add listeners
        final JXTable listTable = requestUI.getResultTable();

        // Add listener to open AO
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(context);
        final MouseListener offerListTableMouseListener =
            UIHelper.getHandler(context, OfferListHandler.class).getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);

        final TableModelListener offerListTableModelListener =
            getOfferListTableModelListener(listTable);
        offerListTableModel.addTableModelListener(offerListTableModelListener);

        // listTable cellRenderer
        final OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();
        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);

        // listTable highlighter
        final Highlighter highlighter = HighlighterFactory.createAlternateStriping(
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        listTable.addHighlighter(highlighter);

        // search ui
        SearchUI searchUI = searchHandler.initCloneUI(context, offerListTableModel, thesaurus);
        String query = null;

        if (request != null) {
            query = request.getQuery();
            searchUI.setRequest(query);

        } else {
            // If it's for visualisation
            if (thesaurus != null) {
                requestUI.setCanSave(false);
                searchUI.setHidable(false);
            }
        }
        requestUI.getSearchPanel().add(searchUI, BorderLayout.CENTER);
        return requestUI;
    }

//    private MouseListener getOfferListTableMouseListener(
//            final AdminRequestUI requestUI,
//            final JTable resultTable,
//            final OfferListTableModel offerListTableModel) {
//
//        // double click on resultTable shows selected offer
//        return new MouseAdapter() {
//            @Override
//            public void mouseClicked(MouseEvent e) {
//                if (e.getClickCount() == 2) {
//                    requestUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
//
//                    int row = resultTable.getSelectedRow();
//                    Form form = offerListTableModel.getFormPageDTO().getFormsToShow().get(row);
//                    UIHelper.openFormPopup(requestUI, form);
//
//                    requestUI.setCursor(null);
//                }
//            }
//        };
//    }

    private TableModelListener getOfferListTableModelListener(final JXTable listTable) {
        return new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                listTable.packAll();
            }
        };
    }

    // STATUS
    public static List<Status> getAllStatuses() {
        try {

            Search search = Search.query().eq(org.nuiton.wikitty.search.Element.ELT_EXTENSION,
                    Status.EXT_STATUS);
            Criteria criteria = search.criteria();

            PagedResult<Status> statuses = VradiService.getWikittyProxy().findAllByCriteria(Status.class,
                    criteria);

            List<Status> all = new ArrayList<Status>(statuses.getAll());

            // If no found, creating all
            if (all.isEmpty()) {
                log.info("No status found, creating default one");
                all = new ArrayList<Status>(VradiService.getVradiDataService().createDefaultStatuses());
            }

            // Sort by value
            Collections.sort(all, VradiComparators.STATUSES_COMPARATOR);

            return all;
        } catch (Exception eee) {
            log.error("Cant get all status : ", eee);
            ErrorDialogUI.showError(eee);
        }
        return new ArrayList<Status>();
    }

    public void saveStatuses(AdminStatusUI adminStatusUI) {
        List<Status> statuses = adminStatusUI.getStatuses();
        List<Status> toSave = new ArrayList<Status>();
        List<Status> deleted = adminStatusUI.getDeleted();

        //FIXME JC 2010/09/06 From here, method should go in service with unit test

        int i = 0;
        for (Status status : statuses) {
            i++;
            status.setValue(i);
            toSave.add(status);
        }

        // extract ids
        List<String> idsToDelete = new ArrayList<String>();
        for (Status toDelete : deleted) {
            log.info("Delete status " + toDelete.getName());
            idsToDelete.add(toDelete.getWikittyId());
        }
        try {
            //FIXME JC 2010/09/06 check that status is not used. If it is used, cannot delete it with nice message for user ;)
            VradiService.getWikittyProxy().delete(idsToDelete);
            VradiService.getWikittyProxy().store(toSave);
        } catch (WikittyException eee) {
            log.error("Cant save status : ", eee);
            
        } finally {
            adminStatusUI.createStatuses(false);
        }
    }

    public void createClient(AdminClientUI context,
                             AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving client
        getUI(context).getClientAdminNavUI().resetFilter();

        // Create new node and select
        helper.createNode(ClientsAndUsersNodeLoadors.class,
                AdminBeanConstant.CLIENT.getCategoryName());
    }

    // USER
    public void createUser(JAXXContext context,
                           AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving client
        getUI(context).getClientAdminNavUI().resetFilter();

        // find client to create user into
        VradiTreeNode clientNode = helper.getParentSelectedNode(Client.class);

        // Create new node and select
        helper.createUserNode(clientNode);
    }

    /**
     * Get all clients.
     * 
     * @return all clients
     */
    public List<Client> getAllClients() {
        List<Client> values = VradiService.getVradiDataService().findAllClients();
        Collections.sort(values, VradiComparators.CLIENT_COMPARATOR);
        return values;
    }

    /**
     * Get all groups.
     * 
     * @return all groups
     */
    public List<Group> getAllGroups() {
        List<Group> values = VradiService.getVradiDataService().findAllGroups();
        Collections.sort(values, VradiComparators.GROUP_COMPARATOR);
        return values;
    }

    /**
     * Get all users.
     * 
     * @return all users
     */
    public List<User> getAllUsers() {
        List<User> values = VradiService.getVradiDataService().findAllUsers();
        Collections.sort(values, VradiComparators.USER_COMPARATOR);
        return values;
    }

    public void createGroup(AdminGroupUI context,
                            AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving group
        getUI(context).getGroupAdminNavUI().resetFilter();
        
        helper.createNode(GroupsNodeLoadors.class,
                AdminBeanConstant.GROUP.getCategoryName());
    }

    // FORM
    public void createFormType(AdminFormTypeUI context,
                               AdminNavigationTreeHelper helper) {
        String name = JOptionPane
                .showInputDialog(getUI(context), _("vradi.adminForm.askName"));

        if (name != null && !name.isEmpty()) {
            WikittyExtension formType = new WikittyExtension(name,
                    WikittyUtil.DEFAULT_VERSION,
                    Form.EXT_FORM, new LinkedHashMap<String, FieldType>());

            // reset tree filter
            // cause NPE when saving form
            getUI(context).getFormAdminNavUI().resetFilter();

            // FIXME use only proxy here, not service
            WikittyProxy proxy = VradiService.getWikittyProxy();
            proxy.getWikittyService().storeExtension(proxy.getSecurityToken(), formType);

            // FIXME SLE 16/07/20 : Remove it, helper must listen modification
            helper.createNode(FormTypeNodeLoadors.class,
                    AdminBeanConstant.FORM_TYPE.getCategoryName(), formType.getId());
        }
    }

    public void createForms(JAXXContext context) {
        final AdminXmlStreamUI xmlStreamUI = getXmlStreamUI(context);
        
        final XmlStream xmlStream = xmlStreamUI.getBean();
        if (xmlStream == null || xmlStream.getFormTypeName() == null) {
            JOptionPane.showMessageDialog(xmlStreamUI, _("vradi.admin.formsCreated.missingFormType"),
                _("vradi.admin.formsCreated.title"), JOptionPane.WARNING_MESSAGE);
            return;
        }
        
        //final VradiUser vradiUser = xmlStreamUI.getContextValue(VradiUser.class);
        
        VradiTask<Object> task = new VradiTask<Object>(context){

            @Override
            public Object doAction() throws Exception {
                return createForms(xmlStream);
            }

            @Override
            public void doWhenDone() throws Exception {
                xmlStreamUI.getCreateForms().setEnabled(true);
                    Object object = get();
                    showFormCreationResults(xmlStreamUI, object);
                }
        };
        
        xmlStreamUI.getCreateForms().setEnabled(false);

        task.execute();
    }
    
    public Object createForms(XmlStream xmlStream) {
        try {
            // Cast as real type
            xmlStream = EntityHelper.castAsRealStream(xmlStream);

            VradiStorageService service = VradiService.getVradiStorageService();
            if (xmlStream instanceof WebHarvestStream){
                return service.getFormsFromWebHarvestStream((WebHarvestStream)xmlStream);
            }
            return service.getFormsFromXmlStream(xmlStream);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return e;
        }
    }

    protected void showFormCreationResults(AdminXmlStreamUI xmlStreamUI, Object object) {
        if (object == null) {
            return;
            
        } else if (object instanceof Exception) {
            ErrorDialogUI.showError((Exception)object);
            return;
            
        } else if (object.getClass().isArray()) {
            int[] formCreationResults = (int[]) object;
            
            int nbFormCreated = formCreationResults[0];
            int alreadyExisting = formCreationResults[1];
            int dateErrorParsing = formCreationResults[2];
            int numberErrorParsing = formCreationResults[3];
            
            /*String optionMessage;
            if (nbFormCreated > 0 && dateErrorParsing > 0 && numberErrorParsing > 0) {
                optionMessage = _("vradi.admin.formsCreated.message.allWarning",
                    nbFormCreated, dateErrorParsing, numberErrorParsing);
                
            } else if (nbFormCreated > 0 && numberErrorParsing > 0) {
                optionMessage = _("vradi.admin.formsCreated.message.numberWarning",
                    nbFormCreated, numberErrorParsing);
                
            } else if (nbFormCreated > 0 && dateErrorParsing > 0) {
               optionMessage = _("vradi.admin.formsCreated.message.dateWarning",
                    nbFormCreated, dateErrorParsing);
               
            } else {
               optionMessage = _("vradi.admin.formsCreated.message.noWarning",
                   nbFormCreated);
            }*/
            
            String optionMessage = _("vradi.admin.formsCreated.message.allWarning",
                    nbFormCreated, alreadyExisting, dateErrorParsing, numberErrorParsing);
            int messageType = JOptionPane.INFORMATION_MESSAGE;
            if (dateErrorParsing > 0 || numberErrorParsing > 0) {
                messageType = JOptionPane.WARNING_MESSAGE;
            }

            JOptionPane.showMessageDialog(xmlStreamUI,
                _("vradi.admin.formsCreated.message") + optionMessage,
                _("vradi.admin.formsCreated.title"), messageType);
        }
    }
    
    public void saveFormType(AdminFormTypeUI context) {
        try {
            List<FieldSelectModel> fields = new ArrayList<FieldSelectModel>();

            VBox contentPanel = context.getContent();
            for(Component component : contentPanel.getComponents()) {
                fields.add(((FieldSelectUI)component).getBean());
            }
            
            WikittyExtension uiExtension = context.getBean();
            WikittyExtension newFormType = new WikittyExtension(uiExtension.getName(),
                    uiExtension.getVersion(), uiExtension.getRequires(), null);
    
            for (int i = 0; i < fields.size(); i++) {
                FieldSelectModel model = fields.get(i);
                String name = model.getNameValue();
                
                if (name != null && model.getType() != null) {
                    FieldType fieldType = model.getType().getFieldType();
                
                    if (model.getDescValue() != null && !model.getDescValue().isEmpty()) {
                        fieldType.addTagValue(VradiHelper.TYPE_DESCRIPTION, model.getDescValue());
                    }
                    
                    fieldType.addTagValue(VradiHelper.TYPE_RANK, String.valueOf(i));
                    newFormType.addField(name, fieldType);
                }
            }
            
            Date now = new Date();
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_DATE,
                    String.format("%1$td/%1$tm/%1$ty", now));
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_TIME,
                    String.format("%1$tk:%1$tM:%1$tS", now));
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_BY,
                    context.getContextValue(VradiUser.class).getLogin());
    
            String templateName = (String)context.getTemplateChoice().getSelectedItem();
            if (templateName != null) {
                newFormType.addTagValue(VradiHelper.TYPE_TEMPLATE, templateName);
            }
            
            // save extension
            newFormType = VradiService.getVradiDataService().updateFormType(newFormType);

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

    }
    
    public WikittyExtension getExtension(String extensionName) {
        try {
            return VradiService.getVradiStorageService().getFormType(extensionName);
        } catch (Exception e) {
            return null;
        }
    }

    // RSS
    /**
     * Cache of the already loaded urls : keeps only the first Element parsed
     */
    public static final Map<String, Element> XML_STREAM_FIELDS_CACHE
            = new HashMap<String, Element>();

    public void createWebHarvestStream(AdminXmlStreamUI context,
                                AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving XmlStream
        getUI(context).getXmlStreamAdminNavUI().resetFilter();

        helper.createNode(StreamNodeLoadors.class, AdminBeanConstant.XML_STREAM.getCategoryName());
    }

    public void createXmlStream(AdminXmlStreamUI context,
                                AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving XmlStream
        getUI(context).getXmlStreamAdminNavUI().resetFilter();

        helper.createNode(StreamNodeLoadors.class, AdminBeanConstant.XML_STREAM.getCategoryName());
    }

    public void openWebHarvestEditor(JAXXContext context) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                new Ide().createAndShowGUI();
            }
        });
    }

    public void addWebHarvestScript(JAXXContext context, final WebHarvestStream stream) {
        JFileChooser chooser = new JFileChooser();

        int returnVal = chooser.showOpenDialog(getXmlStreamUI(context));

        if(returnVal == JFileChooser.APPROVE_OPTION) {
            String url = null;
            try {
                url = VradiService.getFileService().uploadWebHarvestScript(chooser.getSelectedFile());

            } catch (VradiException eee) {
                log.error("Cant upload script : ", eee);
            }
            log.info("Script uploaded to " + url);
            
            stream.setScriptUrl(url);

        }
    }

    public void initScriptComboBox(JComboBox combo, WebHarvestStream bean) {

        VradiStorageService service = VradiService.getVradiStorageService();

        List<String> webHarvestScripts = service.getAllWebHarvestScripts();

        combo.setModel(new DefaultComboBoxModel(webHarvestScripts.toArray()));
        combo.setRenderer(new ListCellRenderer(){

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                if (value == null) {
                    return new JLabel("");
                }
                String fileName = (String) value;
                return new JLabel(fileName);
            }
        });

        if (bean != null) {
            String fileToSelect = bean.getScriptUrl();
            combo.setSelectedItem(fileToSelect);
        }

    }

    public <E extends XmlStream> E saveStream(AdminXmlStreamUI context,
            AdminNavigationTreeHelper helper, E xmlStream,
            List<XmlFieldBinding> bindings) {
        try {
            // save xmlStream
            xmlStream = saveStream(xmlStream, bindings);

        } catch (Exception eee) {
            log.error(eee.getMessage(), eee);
        }

        return xmlStream;
    }

    /**
     * Find all XmlFieldBinding for XmlStream.
     * 
     * @param xmlStream XmlStream to find XmlFieldBinding
     * @return found XmlFieldBinding
     */
    public List<XmlFieldBinding> getXmlStreamFields(XmlStream xmlStream) {
        try {
            List<String> xmlFieldBindingIds = new ArrayList<String>();
            if (xmlStream.getXmlFieldBinding() != null) {
                xmlFieldBindingIds.addAll(xmlStream.getXmlFieldBinding());
            }
            List<XmlFieldBinding> result = VradiService.getWikittyProxy().restore(XmlFieldBinding.class, xmlFieldBindingIds);
            return result;
        } catch (WikittyException e) {
            log.warn(e.getMessage(), e);
            return new ArrayList<XmlFieldBinding>();
        }
    }

    private <E extends XmlStream> WikittyExtension getFormType(E xmlStream) {
        if(xmlStream == null) {
            return null;
        }
        
        String formTypeName = xmlStream.getFormTypeName();
        if(formTypeName == null) {
            return null;
        }
        
        WikittyExtension formType = getExtension(formTypeName);
        
        return formType;
    }

    public void openAdminXmlStreamUI(final AdminXmlStreamUI xmlStreamUI) {
        if (log.isDebugEnabled()) {
            log.debug("openAdminXmlStreamUI(xmlStreamUI)");
        }

        VradiTask<Void> task = new VradiTask<Void>(xmlStreamUI){

            @Override
            public Void doAction() throws Exception {
                updateXmlStreamFieldModel(xmlStreamUI);
                return null;
            }
    
            @Override
            public void doWhenDone() throws Exception {
                HBox lastModified = xmlStreamUI.getLastModifiedPanel();
                
                XmlStream bean = xmlStreamUI.getBean();
                
                xmlStreamUI.setModified(false);
                xmlStreamUI.setCursor(null);
                xmlStreamUI.validate();
            }
        };
        
        task.execute();
    }
    
    public void updateXmlStreamFieldModel(JAXXContext context) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlStreamFieldModel(context)");
        }

        AdminXmlStreamUI xmlStreamUI = (AdminXmlStreamUI) context;
        XmlStreamFieldTableModel fieldsModel = xmlStreamUI.getFieldsModel();
        fieldsModel.setValueIsAdjusting(true);
        fieldsModel.clear();
            
        try {
            XmlStream bean = xmlStreamUI.getBean();
            
            if (bean == null) {
                return;
            }

            bean = EntityHelper.castAsRealStream(bean);

            String url = bean.getUrl();
            if (bean instanceof WebHarvestStream) {
                final WebHarvestStream webHarvest = (WebHarvestStream) bean;
                if (webHarvest.getScriptUrl() == null) {
                    // Pas de script
                    return;
                }
                VradiTask<String> createPreview = new VradiTask<String>(context){
                    @Override
                    public String doAction() throws Exception {
                        return VradiService.getVradiStorageService().getWebHarvestPreviewURL(webHarvest);
                    }

                };
                createPreview.execute();

                url = createPreview.get();
            }

            if (url == null || url.trim().isEmpty()) {
                // pas d'url
                return;
            }
            
            WikittyExtension formType = getFormType(bean);
            if (formType == null) {
                // pas de type de formulaire
                return;
            }
            
            // liste des champs xml disponibles
            Set<String> xmlFieldNames = getXmlFieldNames(url);

            // liste des bindings existants
            List<XmlFieldBinding> bindings = getXmlStreamFields(bean);
            
            // map des bindings formfield/xmlfields indexée par formfield
            LinkedHashMap<String, XmlFieldBinding> fieldNames =
                    getAllBindingFields(formType);
            
            if (bindings != null && !bindings.isEmpty()) {
                // si la liste des champs xml est vide (pas de resau ou flux invalide),
                // on la recupère depuis la liste des champs déjà enregistrées.
                boolean empty = xmlFieldNames.isEmpty();
                
                for (XmlFieldBinding binding : bindings) {
                    String formField = binding.getFormField();
                    
                    if (!fieldNames.containsKey(formField)) {
                        // ce champ était présent lors du precedent enregistrement des bindings
                        // mais il n'existe plus dans l'extension formType
                        continue;
                    }
                    
                    fieldNames.put(formField, binding);
                    
                    if (empty) {
                        Set<String> xmlFields = binding.getXmlField();
                        if (xmlFields != null) {
                            xmlFieldNames.addAll(xmlFields);
                        }
                    }
                }
            }
            
            // mise a jour de l'univers des valeurs selectionnables
            ArrayList<String> xmlFieldList = new ArrayList<String>(xmlFieldNames);
            xmlStreamUI.getContextValue(JListCellEditor.class).updateUniverse(xmlFieldList);
            xmlStreamUI.getContextValue(JListCellRenderer.class).updateUniverse(xmlFieldList);

            // ajout des bindings au model
            Collection<XmlFieldBinding> bindingCollection = fieldNames.values();
            fieldsModel.setBindings(bindingCollection);

            JXTable fieldsTable = xmlStreamUI.getFieldsTable();
            if (!xmlFieldNames.isEmpty()) {
                fieldsTable.setRowHeight(19 * xmlFieldNames.size());
            } else {
                fieldsTable.setRowHeight(19);
            }
        
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            
        } finally {
            fieldsModel.fireTableDataChanged();
            fieldsModel.setValueIsAdjusting(false);
        }
    }

    /**
     * Crée une map de tous les bindings formfield/xmlfields indexée par nom de formfield.
     * 
     * @param formType  liste de fields d'un type de formulaire.
     * @return          une map de bindings ordonnée.
     */
    private LinkedHashMap<String, XmlFieldBinding> getAllBindingFields(WikittyExtension formType) {
        LinkedHashMap<String, XmlFieldBinding> fieldMap
            = new LinkedHashMap<String, XmlFieldBinding>();

        // form fields
        fieldMap.put(Form.FQ_FIELD_INFOGENE_OBJET, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_DESCRIPTION, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_FORM_DATEPUB, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_FORM_DATEPEREMPTION, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_ENTITY, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_SOURCETEXT, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_SOURCEURL, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_COUNTRY, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_INFOGENE_DEPARTMENT, new XmlFieldBindingImpl());
        
        // formtype's bindings fields
        Collection<String> fieldNames = formType.getFieldNames();
        for (String fieldName : fieldNames) {
            fieldMap.put(formType.getName() + "." + fieldName, new XmlFieldBindingImpl());
        }
        
        for (Map.Entry<String, XmlFieldBinding> entry : fieldMap.entrySet()) {
            XmlFieldBinding value = entry.getValue();
            value.setFormField(entry.getKey());
        }
        
        return fieldMap;
    }
    
    /**
     * Recherche les noms des elements fils d'un item rss.
     * 
     * @param url   url du flux rss
     * @return      liste de noms d'elements (les noms sont uniques)
     * @throws VradiException
     */
    private Set<String> getXmlFieldNames(String url) throws VradiException {
        Set<String> xmlFieldNames = new LinkedHashSet<String>();
        Map<String, Element> xmlStreamFields = XML_STREAM_FIELDS_CACHE;

        Element element = xmlStreamFields.get(url);
        if (element == null) {
            try {
                Element firstElement = XmlStreamHelper.getFirstElement(url);
                xmlFieldNames = XmlStreamHelper.getRSSFields(firstElement);
                xmlStreamFields.put(url, firstElement);

            } catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        } else {
            xmlFieldNames = XmlStreamHelper.getRSSFields(element);
        }
        
        return xmlFieldNames;
    }

    public void updateTemplateFieldTableModel(TemplateFieldBindingUI context,
            WikittyExtension extension, String templateName) {
            
        context.setTemplate(templateName);
        context.setExtension(extension.getName());
        TemplateFieldTableModel model = context.getFieldsModel();

        model.setValueIsAdjusting(true);
        model.clear();

        // mise a jour de l'univers des valeurs selectionnables
        List<String> fieldNames = new LinkedList<String>();
        fieldNames.add(Form.FQ_FIELD_INFOGENE_ID);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_OBJET);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_DESCRIPTION);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_CREATIONDATE);
        fieldNames.add(Form.FQ_FIELD_FORM_DATEPUB);
        fieldNames.add(Form.FQ_FIELD_FORM_DATEPEREMPTION);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_ENTITY);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_SOURCETEXT);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_SOURCEURL);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_COUNTRY);
        fieldNames.add(Form.FQ_FIELD_INFOGENE_DEPARTMENT);

        for (String fieldName : extension.getFieldNames()) {
            fieldNames.add(extension.getName() + "." + fieldName);
        }

        context.getContextValue(JListCellEditor.class).updateUniverse(fieldNames);
        context.getContextValue(JListCellRenderer.class).updateUniverse(fieldNames);
        if (fieldNames.size() > 0) {
            context.getFieldsTable().setRowHeight(19 * fieldNames.size());
        }

        try {
            Map<String, String> fieldMap = VradiService.getVradiStorageService().getAssociatedFields(
                            extension.getName(), templateName);
            
            // null = no save found, empty configuration
            if (fieldMap == null || fieldMap.isEmpty()) {
                fieldMap = new HashMap<String, String>();

                List<String> templateFields = VradiService.getVradiStorageService().getDocumentFields(extension, templateName);
                for(String field : templateFields) {
                    fieldMap.put(field, null);
                }
            }
            model.setData(fieldMap);
            
        } catch (Exception eee) {
            log.error("Cant update template", eee);
            ErrorDialogUI.showError(eee);
            
        } finally {
            model.fireTableDataChanged();
            model.setValueIsAdjusting(false);
        }
    }

    public void saveTemplateFieldBinding(TemplateFieldBindingUI context) {
        TemplateFieldTableModel model = context.getFieldsModel();
        Map<String, String> fieldMap = model.getData();

        try {
            String extension = context.getExtension();
            String template = context.getTemplate();
            
            VradiService.getVradiStorageService().setAssociatedFields(extension, template,
                    fieldMap);

            context.dispose();

        } catch (Exception eee) {
            log.error("Cant save template ", eee);
        }
    }

    private AdminXmlStreamUI getXmlStreamUI(JAXXContext context) {
        if (context instanceof AdminXmlStreamUI) {
            return (AdminXmlStreamUI) context;
        }
        return null;
    }

    public String getFirstElementPreview(JAXXContext context, XmlStream bean) {
        if (bean == null || bean instanceof WebHarvestStream) {
            return "";
        }
        String url = bean.getUrl();
        if (url == null || url.trim().isEmpty()) {
            return "";
        }

        Element firstElement = null;
        Map<String, Element> xmlStreamFields = XML_STREAM_FIELDS_CACHE;

        if (xmlStreamFields.get(url) == null) {
            AdminXmlStreamUI ui = getXmlStreamUI(context);
            try {
                ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                firstElement = XmlStreamHelper.getFirstElement(url);
                xmlStreamFields.put(url, firstElement);
            } catch (Exception eee) {
                log.error("Can't get preview", eee);
            } finally {
                ui.setCursor(null);
            }

        } else {
            firstElement = xmlStreamFields.get(url);
        }

        if (firstElement == null) {
            return "";
        }

        Map<String, String> elementValues = XmlStreamHelper.getFirstElementValues(firstElement);

        StringBuffer result = new StringBuffer();
        result.append("<html>");

        for (Map.Entry<String, String> entry : elementValues.entrySet()) {
            result.append("<strong>").append(entry.getKey())
                    .append("</strong>").append(" : ");

            String value = entry.getValue();
            if (value.length() > TOOLTIP_ELEMENT_MAX_CHAR_NB) {
                value = StringUtils.replace(value, "\r", "");
                value = StringUtils.replace(value, "\n\n", "\n");
                value = StringUtils.abbreviate(value, TOOLTIP_ELEMENT_MAX_CHAR_NB);
                value = StringUtils.replace(value, "\n", "<br/>");
            }
            result.append(value).append("<br/>");
        }
        
        result.append("</html>");
        return result.toString();
    }

    public void saveXmlStreamConfig(XmlStreamConfigUI ui) {
        int interval = (Integer)ui.getInterval().getSelectedItem();
        String unit = (String)ui.getUnit().getSelectedItem();
        try {
            VradiStorageService service = VradiService.getVradiStorageService();
            if(XmlStreamConfig.HOURS.getValue().equals(unit)) {
                int minute = (Integer)ui.getMinuteHours().getSelectedItem();
                service.autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.HOURS.getValue(), interval,
                        null, minute);
                
            } else if(XmlStreamConfig.DAYS.getValue().equals(unit)) {
                int hour = (Integer)ui.getHourDays().getSelectedItem();
                int minute = (Integer)ui.getMinuteDays().getSelectedItem();
                service.autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.DAYS.getValue(), interval,
                        hour, minute);
                
            } else if(XmlStreamConfig.MINUTES.getValue().equals(unit)){
                service.autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.MINUTES.getValue(), interval,
                        null, 5);
                
            } else {
                service.autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.DAYS.getValue(), 1, 0, 0);
            }

        } catch (Exception e) {
            log.error("Cant save xml stream config", e);
            ErrorDialogUI.showError(e);
        }
    }

    /**
     * Get combo box model with all template registred for specified extension.
     * 
     * @param extension extension to get template list (can be null)
     * @return combo box model
     */
    public ComboBoxModel getTemplateComboBoxModel(WikittyExtension extension) {
        /*File[] files = getVradiStorageService().getTemplates(extension);
        if(files != null) {
            return new DefaultComboBoxModel(files) {
                @Override
                public void setSelectedItem(Object item) {
                    if (item == null || !String.class.equals(item.getClass())) {
                        super.setSelectedItem(item);
                    } else {
                        for (int i = 0; i < this.getSize(); i++) {
                            File f = (File) this.getElementAt(i);
                            if (f.getName().equals(item)) {
                                super.setSelectedItem(f);
                                break;
                            }
                        }
                    }
                }
            };
        } else {
            return new DefaultComboBoxModel();
        }*/

        ComboBoxModel result = null;
        if (extension != null) {
            Collection<String> fileNames = VradiService.getVradiStorageService().getTemplateFilenames(extension);
            if (CollectionUtils.isNotEmpty(fileNames)) {
                result = new DefaultComboBoxModel(fileNames.toArray(new String[fileNames.size()]));
                // ca marche a suposer que ce que appele ce code
                // fait un ComboBoxModel.setSelectedItem(String)
                // avec en argument le nom du template selectionne
            }
        }

        if (result == null) {
            result = new DefaultComboBoxModel();
        }
        
        return result;
    }

    public void showTemplateExample(AdminFormTypeUI context) {
        WikittyExtension extension = context.getBean();

        FormPagedResult formPage = new FormPagedResult();
        formPage = VradiHelper.executeQuery(null, extension, null, null, null, null, null, null, formPage);

        if(formPage.getTotalFoundFormNb() > 0) {
            Form form = formPage.getFormsToShow().get(0);
            try {
                context.setCursor(new Cursor(Cursor.WAIT_CURSOR));

                //File file = VradiHelper.generatePDF(form);
                String uri = VradiService.getVradiStorageService().generatePDF(form, false);
                File exampleFile = VradiService.getFileService().downloadPDF(uri);
                Desktop.getDesktop().open(exampleFile);

            } catch (VradiException e) {
                log.error(e.getMessage(), e);
                ErrorDialogUI.showError(e);
                
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                
            } finally {
                context.setCursor(null);
            }
        } else {
            JOptionPane.showMessageDialog(context,
                    _("vradi.adminFormType.template.showExample.error.message"),
                    _("vradi.adminFormType.template.showExample.error.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Upload local file on server and refresh template list combo box.
     * 
     * @param adminFormTypeUI parent ui
     * @return {@code true} if file has been uploaded
     */
    public boolean uploadNewTemplate(AdminFormTypeUI adminFormTypeUI) {
        
        boolean result = false;

        try {
            File template = FileUtil.getFile(adminFormTypeUI, new FileFilter[0]);

            //File copy = getVradiStorageService().addTemplate(adminFormTypeUI.getBean(), template);

            if (template != null) {

                // FIXME EC20100519 review file URI
                String extensionName = adminFormTypeUI.getBean().getName();
                String uri = VradiService.getFileService().uploadTemplate(template, extensionName);

                JComboBox templates = adminFormTypeUI.getTemplateChoice();
                templates.addItem(uri);
                templates.setSelectedItem(uri);
                
                result = true;
            }

        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error(_("Can't upload file"), ex);
            }
            ErrorDialogUI.showError(ex);
        }
        
        return result;
    }

    /**
     * Ask for user where to save template and download it.
     * 
     * @param adminFormTypeUI parent ui
     */
    public void downloadTemplate(AdminFormTypeUI adminFormTypeUI) {
        try {
            
            String selectedTemplate = (String)adminFormTypeUI.getTemplateChoice().getSelectedItem();
            // bindings has not been done (can be called with empty selection)
            if (StringUtils.isEmpty(selectedTemplate)) {
                return;
            }

            File savedTemplate = FileUtil.getFile(adminFormTypeUI, new FileFilter[0]);

            if (savedTemplate != null) {

                String extensionName = adminFormTypeUI.getBean().getName();

                File downloadedFile = VradiService.getFileService().downloadTemplate(extensionName, selectedTemplate);
                FileUtils.moveFile(downloadedFile, savedTemplate);
            }

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("Can't download file"), eee);
            }
            ErrorDialogUI.showError(eee);
        }
    }
    
    /**
     * Open rss representing group request historique in web broser
     * 
     * @param id of group with request
     */
    public void showHistory(String id) {

        try {
            String url = VradiService.getFileService().getRequestHistoryURL(id + ".rss");

            if (log.isDebugEnabled()) {
                log.debug("Opening request history : " + url);
            }

            try {
                UIHelper.browseURI(url);
            } catch (IOException eee) {
            log.error("Cannot display history for group : " + id);
            JOptionPane.showMessageDialog(null,
                    _("message.cannot.display.history"));

            }

        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.warn(_("Can't download file"), eee);
            }
            JOptionPane.showConfirmDialog(new JFrame(),
                    _("vradi.error.cantOpenHistory"),
                    _("vradi.error.cantOpenHistory.title"), 
                    JOptionPane.ERROR_MESSAGE);
        }
    }
}

