/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui.admin;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Desktop;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileFilter;

import com.jurismarches.vradi.ui.*;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.HBox;
import jaxx.runtime.swing.VBox;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModel;
import jaxx.runtime.swing.navigation.tree.NavigationTreeNode;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdom.Element;
import org.nuiton.util.FileUtil;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.VradiConfig;
import com.jurismarches.vradi.VradiConstants.XmlStreamConfig;
import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlFieldBindingImpl;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.XmlStreamService;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.services.dto.VradiUserDTO;
import com.jurismarches.vradi.services.dto.VradiXmlStreamDTO;
import com.jurismarches.vradi.ui.admin.content.AdminClientUI;
import com.jurismarches.vradi.ui.admin.content.AdminFormTypeUI;
import com.jurismarches.vradi.ui.admin.content.AdminGroupUI;
import com.jurismarches.vradi.ui.admin.content.AdminStatusUI;
import com.jurismarches.vradi.ui.admin.content.AdminUserUI;
import com.jurismarches.vradi.ui.admin.content.AdminXmlStreamUI;
import com.jurismarches.vradi.ui.admin.content.FieldSelectUI;
import com.jurismarches.vradi.ui.admin.content.RequestSelectUI;
import com.jurismarches.vradi.ui.admin.content.StatusEditionUI;
import com.jurismarches.vradi.ui.admin.content.TemplateFieldBindingUI;
import com.jurismarches.vradi.ui.admin.content.XmlStreamConfigUI;
import com.jurismarches.vradi.ui.editors.JListCellEditor;
import com.jurismarches.vradi.ui.helpers.AdminNavigationTreeHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.helpers.XmlStreamHelper;
import com.jurismarches.vradi.ui.models.FieldSelectModel;
import com.jurismarches.vradi.ui.models.FieldTypeModel;
import com.jurismarches.vradi.ui.models.OfferListTableModel;
import com.jurismarches.vradi.ui.models.TemplateFieldTableModel;
import com.jurismarches.vradi.ui.models.XmlStreamFieldTableModel;
import com.jurismarches.vradi.ui.search.SearchHandler;
import com.jurismarches.vradi.ui.search.SearchUI;

/**
 * @author letellier
 */
public class AdminHandler {
    static private final Log log = LogFactory.getLog(AdminHandler.class);

    static public final int TOOLTIP_LINE_MAX_CHAR_NB = 100;

    static public final int TOOLTIP_ELEMENT_MAX_CHAR_NB = 500;

    public AdminPopupUI init(JAXXContext rootContext, Object... datas) {
        AdminPopupUI adminPopupUI = VradiContext.ADMIN_UI_ENTRY_DEF
                .getContextValue(rootContext);

        if (adminPopupUI == null) {
            JAXXInitialContext context = new JAXXInitialContext()
                    .add(rootContext).add(this);

            VradiMainUI mainUI = VradiContext.MAIN_UI_ENTRY_DEF
                    .getContextValue(context);

            adminPopupUI = new AdminPopupUI(context, mainUI);

            VradiContext.ADMIN_UI_ENTRY_DEF.setContextValue(rootContext, adminPopupUI);
        }
        return adminPopupUI;
    }

    AdminPopupUI getUI(JAXXContext context) {
        if (context instanceof AdminPopupUI) {
            return (AdminPopupUI) context;
        }
        AdminPopupUI ui = VradiContext.ADMIN_UI_ENTRY_DEF
                .getContextValue(context);
        return ui;
    }

    void close(JAXXContext context) {
        AdminPopupUI ui = getUI(context);
        ui.setVisible(false);
        ui.dispose();
    }

    private VradiStorageService vradiStorageService = null;

    private XmlStreamService xmlStreamService = null;
    
    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }

    protected XmlStreamService getXmlStreamService() {
        if (xmlStreamService == null) {
            xmlStreamService = VradiService.getXmlStreamService();
        }
        return xmlStreamService;
    }

    private <B> void repaintAdminUI(AdminContentUI<B> context,
            AdminNavigationTreeHelper helper, B bean) throws Exception {
        
        // update bean in ui context
        helper.setSelectedBean(context, bean);
        
        // get current node
        NavigationTreeNode selectedNode = helper.getSelectedNode(context);

        // repaint current node
        selectedNode.setBean(bean);
        selectedNode.reload(context);
        helper.repaintNode(context, selectedNode);
        
        // reopen ui
        context.openUI(selectedNode);
    }
    
    private Group saveGroup(Group group) throws VradiException {
        // save group
        Group newgroup = getVradiStorageService().updateEntity(group);
        
        // update context cache
        VradiContext.replace(newgroup);
        
        return newgroup;
    }
    
    private Collection<Group> saveGroups(Collection<Group> groups) throws VradiException {
        // save groups
        Group[] groupArray = new Group[groups.size()];
        groups.toArray(groupArray);
        groupArray = getVradiStorageService().updateEntities(groupArray);
        
        // update context cache
        for (Group group : groupArray) {
            VradiContext.replace(group);
        }

        Collection<Group> newgroups = Arrays.asList(groupArray);
        return newgroups;
    }
    
    
    private Client saveClient(Client client) throws VradiException {
        // save client
        Client newclient = getVradiStorageService().updateEntity(client);
        
        // update context cache
        VradiContext.replace(newclient);
        
        return newclient;
    }
    
    private User saveUser(User user) throws VradiException {
        // save client
        User newuser = getVradiStorageService().updateEntity(user);
        
        // update context cache
        VradiContext.replace(newuser);
        
        return newuser;
    }
    
    private XmlStream saveXmlStream(XmlStream xmlStream,
            List<XmlFieldBinding> bindings) throws VradiException {
        // save xmlStream
        XmlStream newXmlStream = getVradiStorageService().updateXmlStream(xmlStream, bindings);
        
        // update context cache
        VradiContext.replace(newXmlStream);
        
        return newXmlStream;
    }
    
    public void removeFromGroup(AdminContentUI context,
            AdminNavigationTreeHelper helper) {
        try {
            /*
             * part 1: remove client or user from group
             */
            NavigationTreeNode selectedNode = helper.getSelectedNode(context);
            BusinessEntity bean = (BusinessEntity) selectedNode.getBean();
            
            NavigationTreeNode parentNode = helper.getParentGroupNode(selectedNode);
            Group group = (Group) parentNode.getBean(context);
            
            // FIXME: should save client's group or user's group and refresh context cache
            group.removeClient(bean.getWikittyId());
            group.removeUser(bean.getWikittyId());
            
            // save group
            group = saveGroup(group);
        
            // repaint UI XXX
            helper.reloadGroup(context, parentNode, group);
            helper.selectNode(context, parentNode.getFullPath());
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }
    
    public Group saveGroup(AdminGroupUI context, AdminNavigationTreeHelper helper, Group group) {
        try {
            // save group
            group = saveGroup(group);
            
            // repaint UI XXX
            repaintAdminUI(context, helper, group);
            helper.reloadGroup(context, helper.getSelectedNode(context), group);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return group;
    }
    
    public Client saveClient(AdminClientUI context, AdminNavigationTreeHelper helper,
            Client client, Collection<Group> groups) {
        try {
            // save client
            client = saveClient(client);
            
            // save associated groups
            if (groups != null && !groups.isEmpty()) {
                saveGroups(groups);
            }
            
            // repaint UI XXX
            repaintAdminUI(context, helper, client);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return client;
    }
    
    public User saveUser(AdminUserUI context, AdminNavigationTreeHelper helper,
            User user, Client client, Collection<Group> groups) {
        try {
            // save user
            user = saveUser(user);
            
            // save associated client
            if (client != null) {
                client = saveClient(client);
            }
            
            // save associated groups
            if (groups != null && !groups.isEmpty()) {
                saveGroups(groups);
            }
            
            // repaint UI XXX
            repaintAdminUI(context, helper, user);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return user;
    }
    
    /**
     * Export all clients database (with groups, users, clients) as CSV.
     * 
     * Used in {@link com.jurismarches.vradi.ui.admin.nav.AdminClientNavUI}.
     * 
     * @param adminNavUI parent ui
     */
    public void exportClientUsersDBCSV(AdminNavUI adminNavUI) {

        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminClient.exportGroupTitle"));
            chooser.setApproveButtonText(_("vradi.adminClient.exportButtonText"));
            int returnVal = chooser.showOpenDialog(adminNavUI);

            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();

                if (log.isDebugEnabled()) {
                    log.debug("Exporting groups to : " + selectedFile.getAbsolutePath());
                }

                // creation du criteria wikitty
                // (export de tous les wikitty avec extension "client"
                // "groupement" ou "user"
                Search restriction = Search.query().or(); //definit un "groupement" qui se fera en "or"
                restriction.eq(org.sharengo.wikitty.search.Element.ELT_EXTENSION, Group.EXT_GROUP);
                restriction.eq(org.sharengo.wikitty.search.Element.ELT_EXTENSION, Client.EXT_CLIENT);
                restriction.eq(org.sharengo.wikitty.search.Element.ELT_EXTENSION, User.EXT_USER);
                Criteria criteria = restriction.criteria();

                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                String vscContent = vradiStorageService.exportAsCSV(criteria);
                
                // write content into file
                FileUtils.writeStringToFile(selectedFile, vscContent);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export groups", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export groups", eee);
            }
        }
    }

    /**
     * Import CSV file.
     * 
     * Can be used to import clients, groups or thesaurus.
     * 
     * @param adminNavUI parent ui
     */
    public void importClientUsersDBCSV(AdminNavUI adminNavUI) {
        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminClient.importTitle"));
            chooser.setApproveButtonText(_("vradi.adminClient.importButtonText"));
            int returnVal = chooser.showOpenDialog(adminNavUI);
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();
    
                if (log.isDebugEnabled()) {
                    log.debug("Importing file : " + selectedFile.getAbsolutePath());
                }
                
                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                // FIXME EC-20100415 will not work on remote mode until uri
                // means somethings for server too
                String uri = selectedFile.toURI().toURL().toExternalForm();
                vradiStorageService.importAsCSV(uri);
                
                VradiContext.initEntryDef();
                
                // TODO test
                AdminNavigationTreeHelper helper = adminNavUI.getHelper();
                helper.getTree(adminNavUI).setModel((NavigationTreeModel)helper.createTreeModel());
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
        }
    }

    public boolean answerToSave(AdminContentUI<?> content) {
        if (content != null) {
            if (content.isModified()) {
                int n = JOptionPane.showConfirmDialog(content,
                        _("vradi.admin.saveAnswer")
                        + VradiHelper.getEntityName(content.getBean())
                        + " ?", _("vradi.admin.saveTitle"),
                        JOptionPane.YES_NO_CANCEL_OPTION);
                
                switch (n) {
                case JOptionPane.NO_OPTION:
                    // annule les modifications
                    cancel(content, content.getHelper(), false);
                    break;
                case JOptionPane.OK_OPTION:
                    // sauvegarde les modifications
                    content.postSave();
                    break;
                case JOptionPane.CANCEL_OPTION:
                    // annule le changement d'onglet
                    return false;
                }
            }
        }
        return true;
    }

    public boolean confirmDeletion(AdminContentUI<?> content) {
        int confirm = JOptionPane.showConfirmDialog(content,
                _("vradi.admin.confirmDelete", VradiHelper.getEntityName(content.getBean())),
                _("vradi.admin.deleteTitle"), JOptionPane.YES_NO_OPTION);

        if (confirm == JOptionPane.NO_OPTION) {
            return false;
        }
        return true;
    }
    
    public <B> void delete(AdminContentUI<B> context, AdminNavigationTreeHelper helper,
            B bean) {
        if (!confirmDeletion(context)) {
            return;
        }
        
        try {
            context.setModified(false);
            
            // try to remove node: some exception may be thrown
            NavigationTreeNode selectedNode = helper.getSelectedNode(context);
            NavigationTreeNode parentNode = helper.removeChildNode(selectedNode);

            // deleteEntity is too generic to apply business rules on service layer
            BusinessEntity entity = (BusinessEntity) bean;
            // check if the entity is to be delete or if it is a newly created one
            // which have never been saved
            Object modified = entity.getField(ModificationTag.EXT_MODIFICATION_TAG,
                    ModificationTag.FIELD_LAST_MODIFIED);
            if (modified != null) {
                getVradiStorageService().deleteEntity(entity);
            }
            VradiContext.remove(entity);
            
            // select parent node
            helper.selectNode(context, parentNode);

        } catch (VradiException e) {
            // no business rule is applied, so do not show error to end user
            log.error(e.getMessage(), e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
        }
    }

    <B> void cancel(AdminContentUI<B> content,
            AdminNavigationTreeHelper helper, boolean trigerredByCancelButton) {
        int confirm = -1;
        if(trigerredByCancelButton) {
            confirm = JOptionPane.showConfirmDialog(content,
                    _("vradi.admin.cancel.message"),
                    _("vradi.admin.cancel.title"),
                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }
        
        if(!trigerredByCancelButton || confirm == JOptionPane.YES_OPTION) {
            try {
                NavigationTreeNode selectedNode = helper.getSelectedNode(content);
                content.openUI(selectedNode);
                
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    public ComboBoxModel getTypesModel() {
        ComboBoxModel model = new FieldTypeModel();
        return model;
    }

    // REQUEST
    public void openRequest(final JAXXContext rootContext, List<VradiThesaurusDTO> thesaurus) {
        openRequest(rootContext, null, thesaurus);
    }

    public void openRequest(final JAXXContext rootContext, VradiQueryBean request) {
        openRequest(rootContext, request, null);
    }
    
    public void openRequest(final JAXXContext rootContext, VradiQueryBean request, List<VradiThesaurusDTO> thesaurus) {
        final AdminRequestUI requestUI = createAdminRequestUI(rootContext, request, thesaurus);

        SearchHandler searchHandler = requestUI.getContextValue(SearchHandler.class);
        SearchUI searchUI = requestUI.getSearchUI();
        OfferListTableModel offerListTableModel = requestUI.getContextValue(OfferListTableModel.class);
        String query = requestUI.getRequest();
        
        // execute search
        final VradiFormPageDTO formPageDTO = new VradiFormPageDTO();
        searchHandler.executeQuery(searchUI, searchUI.getCriterias(), formPageDTO, null, query);
        offerListTableModel.setFormPageDTO(formPageDTO);

        requestUI.addPropertyChangeListener(AdminRequestUI.PROPERTY_REQUEST,
            new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    String newValue = (String) evt.getNewValue();
                    RequestSelectUI ui = (RequestSelectUI) rootContext;
                    VradiQueryBean bean = ui.getBean();
                    ui.setBean(bean.setQuery(newValue));
                }
            });

        requestUI.setVisible(true);
    }

    public AdminRequestUI createAdminRequestUI(final JAXXContext rootContext, VradiQueryBean request, List<VradiThesaurusDTO> thesaurus) {
        final JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        final SearchHandler searchHandler = new SearchHandler();
        final OfferListTableModel offerListTableModel = new OfferListTableModel();
        final OfferListColumnFactory offerListColumnFactory = new OfferListColumnFactory();

        context.add(searchHandler);
        context.add(offerListTableModel);
        context.add(offerListColumnFactory);

        final AdminRequestUI requestUI = new AdminRequestUI(context);

        // add listeners
        final JXTable listTable = requestUI.getResultTable();

        // Add listener to open AO
        OfferListUI ui = VradiContext.OFFERT_LIST_UI_ENTRY_DEF.getContextValue(context);
        final MouseListener offerListTableMouseListener =
            UIHelper.getHandler(context, OfferListHandler.class).getOfferListTableMouseListener();
        listTable.addMouseListener(offerListTableMouseListener);

        final TableModelListener offerListTableModelListener =
            getOfferListTableModelListener(listTable);
        offerListTableModel.addTableModelListener(offerListTableModelListener);

        // listTable cellRenderer
        final OfferListTableModel.OfferListTableCellRenderer renderer =
                new OfferListTableModel.OfferListTableCellRenderer();
        listTable.setDefaultRenderer(Boolean.class, renderer);
        listTable.setDefaultRenderer(Date.class, renderer);
        listTable.setDefaultRenderer(Double.class, renderer);
        listTable.setDefaultRenderer(String.class, renderer);

        // listTable highlighter
        final Highlighter highlighter = HighlighterFactory.createAlternateStriping(
                HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        listTable.addHighlighter(highlighter);

        // search ui
        SearchUI searchUI = searchHandler.initCloneUI(context, offerListTableModel, thesaurus);
        String query = null;
        
        if (request != null) {
            query = request.getQuery();
            searchUI.setRequest(query);
            
        } else {
            // If it's for visualisation
            if (thesaurus != null) {
                requestUI.setCanSave(false);
                searchUI.setHidable(false);
            }
        }
        requestUI.getSearchPanel().add(searchUI, BorderLayout.CENTER);
        return requestUI;
    }
    
//    private MouseListener getOfferListTableMouseListener(
//            final AdminRequestUI requestUI,
//            final JTable resultTable,
//            final OfferListTableModel offerListTableModel) {
//
//        // double click on resultTable shows selected offer
//        return new MouseAdapter() {
//            @Override
//            public void mouseClicked(MouseEvent e) {
//                if (e.getClickCount() == 2) {
//                    requestUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
//
//                    int row = resultTable.getSelectedRow();
//                    Form form = offerListTableModel.getFormPageDTO().getFormsToShow().get(row);
//                    UIHelper.openFormPopup(requestUI, form);
//
//                    requestUI.setCursor(null);
//                }
//            }
//        };
//    }

    private TableModelListener getOfferListTableModelListener(final JXTable listTable) {
        return new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                listTable.packAll();
            }
        };
    }

    // STATUS
    public List<Status> getAllStatuses(){
        return VradiContext.getStatusInEntryDef();
    }

    public void saveStatuses(AdminStatusUI adminStatusUI) {
        List<Status> existingStatuses = VradiContext.getStatusInEntryDef();
        List<String> existingStatusIds = new ArrayList<String>();
        
        for(Status status : existingStatuses) {
            existingStatusIds.add(status.getWikittyId());
        }
        
        List<Status> statusesToAdd = new ArrayList<Status>();
        JPanel statusPanel = adminStatusUI.getContent();
        int statusValue = 0;
        
        for(int i = 0 ; i < statusPanel.getComponentCount() ; i++) {
            StatusEditionUI editionUI = (StatusEditionUI) statusPanel.getComponent(i);
            Status status = editionUI.getWikitty();
        
            editionUI.getBean().toWikitty(status);
            status.setValue(statusValue++);
            statusesToAdd.add(status);
        }

        List<String> statusesToDelete = new ArrayList<String>();
        JPanel deletedPanel = adminStatusUI.getDeletedContent();
        
        for(int i = 0 ; i < deletedPanel.getComponentCount() ; i++) {
            StatusEditionUI editionUI = (StatusEditionUI) deletedPanel.getComponent(i);
            
            if(existingStatusIds.contains(editionUI.getWikitty().getWikittyId())) {
                statusesToDelete.add(editionUI.getWikitty().getWikittyId());
            }
        }
        deletedPanel.removeAll();

        try {
            List<Status> updateStatuses = getVradiStorageService().updateStatuses(statusesToAdd);
            getVradiStorageService().deleteStatuses(statusesToDelete);
            VradiContext.setStatusEntryDef(updateStatuses);
            
        } catch(VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } finally {
            adminStatusUI.createStatuses(false);
        }
    }

    // CLIENT
    public Map<Client, Set<String>> getAllClients() {
        SortedMap<Client, Set<String>> clientsAndUsers = new TreeMap<Client, Set<String>>(VradiComparators.CLIENT_COMPARATOR);
        
        // map client id = user ids
        Map<String, Set<String>> clientsIdAndUsers = new TreeMap<String, Set<String>>();

        List<Client> clients = VradiContext.getClientsInEntryDef();
        List<User> users = VradiContext.getUsersInEntryDef();
        
        for (Client client : clients) {
            Set<String> ids = new HashSet<String>();
            clientsAndUsers.put(client, ids);
            clientsIdAndUsers.put(client.getWikittyId(), ids);
        }
        
        for (User user : users) {
            String clientId = user.getClient();
            String userId = user.getWikittyId();
            Set<String> usersIds = clientsIdAndUsers.get(clientId);
            usersIds.add(userId);
        }

        return clientsAndUsers;
    }

    public List<User> getAllUsers() {
        List<User> values = VradiContext.getUsersInEntryDef();

        // always return a safe copy (of the list, not the users)
        List<User>  copy = new ArrayList<User>(values);
        Collections.sort(copy, VradiComparators.USER_COMPARATOR);
        return copy;
    }

//    public List<VradiUserDTO> getAllDisplayUsers() throws VradiException {
//        List<VradiUserDTO> allUserDTOs = getVradiStorageService().getAllUserDTOs();
//        Collections.sort(allUserDTOs, VradiComparators.USERDTO_COMPARATOR);
//        return allUserDTOs;
//    }

    public List<VradiUserDTO> getAllDisplayUsers() {
        List<VradiUserDTO> result = new ArrayList<VradiUserDTO>();
        List<User> allUsers = getAllUsers();
        for (User u : allUsers){
            VradiUserDTO dto = new VradiUserDTO();
            dto.fromWikitty(u);

            Client client = VradiHelper.findEntityInRef(VradiContext.getClientsEntryDef(), u.getClient());
            if (client != null){
                dto.setClientName(client.getName());
            }

            result.add(dto);
        }
        return result;
    }

    public void createClient(AdminClientUI context,
                             AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving client
        getUI(context).getClientAdminNavUI().resetFilter();
        
        NavigationTreeNode addClient = helper
                .addClientToSelected(context, null);
        helper.selectNode(context, addClient);
    }

    // USER
    public void createUser(JAXXContext context,
                           AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving client
        getUI(context).getClientAdminNavUI().resetFilter();
        
        NavigationTreeNode addUser = helper.addUserToSelected(context, null);
        helper.selectNode(context, addUser);
    }

    // GROUP
    public List<Group> getAllGroups() {
        List<Group> values = VradiContext.getGroupsInEntryDef();
        // always return a safe copy
        return new ArrayList<Group>(values);
    }

    public void createGroup(AdminGroupUI context,
                            AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving group
        getUI(context).getGroupAdminNavUI().resetFilter();
        
        NavigationTreeNode addGroup = helper.addGroupToSelected(context, null);
        helper.selectNode(context, addGroup);
    }

    // FORM
    public List<WikittyExtension> getAllFormTypes() {
        List<WikittyExtension> values = VradiContext.getFormsTypeInEntryDef();
        // always return a safe copy
        return new ArrayList<WikittyExtension>(values);
    }

    public Set<String> getAllFormTypeNames() {
        List<WikittyExtension> extensions = VradiContext.getFormsTypeInEntryDef();
        HashSet<String> names = new HashSet<String>();
        
        for (WikittyExtension extension : extensions) {
            names.add(extension.getName());
        }
        
        return names;
    }
    
    public void createFormType(AdminFormTypeUI context,
                               AdminNavigationTreeHelper helper) {
        String name = JOptionPane
                .showInputDialog(getUI(context), _("vradi.adminForm.askName"));

        if (name != null && !name.isEmpty()) {
            WikittyExtension formType = new WikittyExtension(name,
                    WikittyUtil.DEFAULT_VERSION,
                    Form.EXT_FORM, new LinkedHashMap<String, FieldType>());

            // reset tree filter
            // cause NPE when saving form
            getUI(context).getFormAdminNavUI().resetFilter();
            
            NavigationTreeNode addFormType = helper
                    .addFormTypeToSelected(context, formType);
            
            helper.setSelectedNode(context, addFormType);
            helper.selectNode(context, addFormType);
        }
    }

    public void createForms(JAXXContext context) {
        final AdminXmlStreamUI xmlStreamUI = getXmlStreamUI(context);
        
        final XmlStream xmlStream = xmlStreamUI.getBean();
        if (xmlStream == null || xmlStream.getFormTypeName() == null) {
            JOptionPane.showMessageDialog(xmlStreamUI, _("vradi.admin.formsCreated.missingFormType"),
                _("vradi.admin.formsCreated.title"), JOptionPane.WARNING_MESSAGE);
            return;
        }
        
        final VradiUser vradiUser = xmlStreamUI.getContextValue(VradiUser.class);
        
        VradiTask<Object> task = new VradiTask<Object>(context){

            @Override
            public Object doAction() throws Exception {
                return createForms(xmlStream, vradiUser);
            }

            @Override
            public void doWhenDone() throws Exception {
                xmlStreamUI.getCreateForms().setEnabled(true);
                    Object object = get();
                    showFormCreationResults(xmlStreamUI, object);
                }
        };
        
        xmlStreamUI.getCreateForms().setEnabled(false);

        task.execute();
    }
    
    public Object createForms(XmlStream xmlStream, VradiUser vradiUser) {
        try {
            return getVradiStorageService().getFormsFromXmlStream(xmlStream, vradiUser);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return e;
        }
    }

    protected void showFormCreationResults(AdminXmlStreamUI xmlStreamUI, Object object) {
        if (object == null) {
            return;
            
        } else if (object instanceof Exception) {
            ErrorDialogUI.showError((Exception)object);
            return;
            
        } else if (object.getClass().isArray()) {
            int[] formCreationResults = (int[]) object;
            
            int nbFormCreated = formCreationResults[0];
            int alreadyExisting = formCreationResults[1];
            int dateErrorParsing = formCreationResults[2];
            int numberErrorParsing = formCreationResults[3];
            
            /*String optionMessage;
            if (nbFormCreated > 0 && dateErrorParsing > 0 && numberErrorParsing > 0) {
                optionMessage = _("vradi.admin.formsCreated.message.allWarning",
                    nbFormCreated, dateErrorParsing, numberErrorParsing);
                
            } else if (nbFormCreated > 0 && numberErrorParsing > 0) {
                optionMessage = _("vradi.admin.formsCreated.message.numberWarning",
                    nbFormCreated, numberErrorParsing);
                
            } else if (nbFormCreated > 0 && dateErrorParsing > 0) {
               optionMessage = _("vradi.admin.formsCreated.message.dateWarning",
                    nbFormCreated, dateErrorParsing);
               
            } else {
               optionMessage = _("vradi.admin.formsCreated.message.noWarning",
                   nbFormCreated);
            }*/
            
            String optionMessage = _("vradi.admin.formsCreated.message.allWarning",
                    nbFormCreated, alreadyExisting, dateErrorParsing, numberErrorParsing);
            int messageType = JOptionPane.INFORMATION_MESSAGE;
            if (dateErrorParsing > 0 || numberErrorParsing > 0) {
                messageType = JOptionPane.WARNING_MESSAGE;
            }

            JOptionPane.showMessageDialog(xmlStreamUI,
                _("vradi.admin.formsCreated.message") + optionMessage,
                _("vradi.admin.formsCreated.title"), messageType);
        }
    }
    
    public void saveFormType(AdminFormTypeUI context) {
        try {
            List<FieldSelectModel> fields = new ArrayList<FieldSelectModel>();

            VBox contentPanel = context.getContent();
            for(Component component : contentPanel.getComponents()) {
                fields.add(((FieldSelectUI)component).getBean());
            }
            
            WikittyExtension uiExtension = context.getBean();
            WikittyExtension newFormType = new WikittyExtension(uiExtension.getName(),
                    uiExtension.getVersion(), uiExtension.getRequires(), null);
    
            for (int i = 0; i < fields.size(); i++) {
                FieldSelectModel model = fields.get(i);
                String name = model.getNameValue();
                
                if (name != null && model.getType() != null) {
                    FieldType fieldType = model.getType().getFieldType();
                
                    if (model.getDescValue() != null && !model.getDescValue().isEmpty()) {
                        fieldType.addTagValue(VradiHelper.TYPE_DESCRIPTION, model.getDescValue());
                    }
                    
                    fieldType.addTagValue(VradiHelper.TYPE_RANK, String.valueOf(i));
                    newFormType.addField(name, fieldType);
                }
            }
            
            Date now = new Date();
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_DATE,
                    String.format("%1$td/%1$tm/%1$ty", now));
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_TIME,
                    String.format("%1$tk:%1$tM:%1$tS", now));
            newFormType.addTagValue(VradiHelper.TYPE_LAST_MODIFIED_BY,
                    context.getContextValue(VradiUser.class).getName());
    
            File template = (File) context.getTemplateChoice().getSelectedItem();
            if(template != null) {
                newFormType.addTagValue(VradiHelper.TYPE_TEMPLATE, template.getName());
            }
            
            // save extension
            newFormType = getVradiStorageService().updateFormType(newFormType);
            
            // update cache
            List<WikittyExtension> forms = VradiContext.getFormsTypeInEntryDef();
            forms.add(newFormType);
            VradiContext.setFormsTypeEntryDef(forms);

            // get selected node and parent node
            AdminNavigationTreeHelper helper = context.getHelper();
            NavigationTreeNode selectedNode = helper.getSelectedNode(context);
            NavigationTreeNode parentNode = selectedNode.getParent();
            parentNode.setBean(forms);
            
            // add new node
            NavigationTreeNode newNode = helper.addFormType(context, parentNode, newFormType);
            helper.setSelectedNode(context, newNode);
            
            // repaint UI XXX
            repaintAdminUI(context, helper, newFormType);

            // remove old selected node
            helper.removeChildNode(selectedNode);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

    }
    
    public WikittyExtension getExtension(String extensionName) {
        try {
            return getVradiStorageService().getFormType(extensionName);
        } catch (Exception e) {
            return null;
        }
    }

    // RSS
    /**
     * Cache of the already loaded urls : keeps only the first Element parsed
     */
    public static final Map<String, Element> XML_STREAM_FIELDS_CACHE
            = new HashMap<String, Element>();

    public void createXmlStream(AdminXmlStreamUI context,
                                AdminNavigationTreeHelper helper) {
        // reset tree filter
        // cause NPE when saving XmlStream
        getUI(context).getXmlStreamAdminNavUI().resetFilter();
        
        NavigationTreeNode addXmlStream = helper
                .addXmlStreamToSelected(context, null);
        helper.selectNode(context, addXmlStream);
    }

    public XmlStream saveXmlStream(AdminXmlStreamUI context,
            AdminNavigationTreeHelper helper, XmlStream xmlStream,
            List<XmlFieldBinding> bindings) {
        try {
            // save xmlStream
            xmlStream = saveXmlStream(xmlStream, bindings);
            
            // repaint UI XXX
            repaintAdminUI(context, helper, xmlStream);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return xmlStream;
    }

    public List<XmlFieldBinding> getXmlStreamFields(XmlStream xmlStream) {
        try {
            List<XmlFieldBinding> result = getVradiStorageService().getXmlFieldBindings(xmlStream);
            return result;
        } catch (VradiException e) {
            log.warn(e.getMessage(), e);
            return new ArrayList<XmlFieldBinding>();
        }
    }

    private WikittyExtension getFormType(VradiXmlStreamDTO xmlStream) {
        if(xmlStream == null) {
            return null;
        }
        
        String formTypeName = xmlStream.getFormTypeName();
        if(formTypeName == null) {
            return null;
        }
        
        WikittyExtension formType = xmlStream.getFormType();
        if(formType != null && formTypeName.equals(formType.getName())) {
            return formType;
        }
        
        formType = getExtension(formTypeName);
        xmlStream.setFormType(formType);
        
        return formType;
    }

    public void openAdminXmlStreamUI(final AdminXmlStreamUI xmlStreamUI) {
        
        VradiTask<Void> task = new VradiTask<Void>(xmlStreamUI, false){

            @Override
            public Void doAction() throws Exception {
                updateXmlStreamFieldModel(xmlStreamUI);
                return null;
            }
    
            @Override
            public void doWhenDone() throws Exception {
                HBox lastModified = xmlStreamUI.getLastModified();
                lastModified.removeAll();
                
                VradiXmlStreamDTO bean = xmlStreamUI.getBean();
                if(bean != null && bean.getLastModified() != null && bean.getLastModifier() != null) {
                    String labelText = String.format(_("vradi.entity.lastModified.date.user"), bean.getLastModified(), bean.getLastModifier());
                    lastModified.add(new JLabel(labelText));
                }
                lastModified.validate();
                
                xmlStreamUI.setModified(false);
                xmlStreamUI.setCursor(null);
                xmlStreamUI.validate();
            }
        };
        
        task.execute();
    }
    
    public void updateXmlStreamFieldModel(AdminXmlStreamUI xmlStreamUI) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlStreamFieldModel(context)");
        }
        
        XmlStreamFieldTableModel fieldsModel = xmlStreamUI.getFieldsModel();
        fieldsModel.setValueIsAdjusting(true);
        fieldsModel.clear();
            
        try {
            VradiXmlStreamDTO bean = xmlStreamUI.getBean();
            
            String url = bean.getUrl();
            if (url == null || url.trim().isEmpty()) {
                // pas d'url
                return;
            }
            
            WikittyExtension formType = getFormType(bean);
            if (formType == null) {
                // pas de type de formulaire
                return;
            }
            
            // liste des champs xml disponibles
            Set<String> xmlFieldNames = getXmlFieldNames(url);

            // liste des bindings existants
            List<XmlFieldBinding> bindings = getXmlStreamFields(bean);
            
            // map des bindings formfield/xmlfields indexée par formfield
            LinkedHashMap<String, XmlFieldBinding> fieldNames =
                    getAllBindingFields(formType);
            
            if (bindings != null && !bindings.isEmpty()) {
                // si la liste des champs xml est vide (pas de resau ou flux invalide),
                // on la recupère depuis la liste des champs déjà enregistrées.
                boolean empty = xmlFieldNames.isEmpty();
                
                for (XmlFieldBinding binding : bindings) {
                    String formField = binding.getFormField();
                    
                    if (!fieldNames.containsKey(formField)) {
                        // ce champ était présent lors du precedent enregistrement des bindings
                        // mais il n'existe plus dans l'extension formType
                        continue;
                    }
                    
                    fieldNames.put(formField, binding);
                    
                    if (empty) {
                        Set<String> xmlFields = binding.getXmlField();
                        if (xmlFields != null) {
                            xmlFieldNames.addAll(xmlFields);
                        }
                    }
                }
            }
            
            // mise a jour de l'univers des valeurs selectionnables
            ArrayList<String> xmlFieldList = new ArrayList<String>(xmlFieldNames);
            xmlStreamUI.getContextValue(JListCellEditor.class).updateUniverse(xmlFieldList);
            xmlStreamUI.getContextValue(JListCellRenderer.class).updateUniverse(xmlFieldList);

            // ajout des bindings au model
            Collection<XmlFieldBinding> bindingCollection = fieldNames.values();
            fieldsModel.setBindings(bindingCollection);

            JXTable fieldsTable = xmlStreamUI.getFieldsTable();
            if (!xmlFieldNames.isEmpty()) {
                fieldsTable.setRowHeight(19 * xmlFieldNames.size());
            } else {
                fieldsTable.setRowHeight(19);
            }
        
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            
        } finally {
            fieldsModel.fireTableDataChanged();
            fieldsModel.setValueIsAdjusting(false);
        }
    }

    /**
     * Crée une map de tous les bindings formfield/xmlfields indexée par nom de formfield.
     * 
     * @param formType  liste de fields d'un type de formulaire.
     * @return          une map de bindings ordonnée.
     */
    private LinkedHashMap<String, XmlFieldBinding> getAllBindingFields(WikittyExtension formType) {
        LinkedHashMap<String, XmlFieldBinding> fieldMap
            = new LinkedHashMap<String, XmlFieldBinding>();

        // form fields
        fieldMap.put(Form.FQ_FIELD_OBJET, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_DESCRIPTION, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_DATEPUB, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_DATEPEREMPTION, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_ENTITY, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_SOURCETEXT, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_SOURCEURL, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_COUNTRY, new XmlFieldBindingImpl());
        fieldMap.put(Form.FQ_FIELD_DEPARTMENT, new XmlFieldBindingImpl());
        
        // formtype's bindings fields
        Collection<String> fieldNames = formType.getFieldNames();
        for (String fieldName : fieldNames) {
            fieldMap.put(formType.getName() + "." + fieldName, new XmlFieldBindingImpl());
        }
        
        for (Map.Entry<String, XmlFieldBinding> entry : fieldMap.entrySet()) {
            XmlFieldBinding value = entry.getValue();
            value.setFormField(entry.getKey());
        }
        
        return fieldMap;
    }
    
    /**
     * Recherche les noms des elements fils d'un item rss.
     * 
     * @param url   url du flux rss
     * @return      liste de noms d'elements (les noms sont uniques)
     * @throws VradiException
     */
    private Set<String> getXmlFieldNames(String url) throws VradiException {
        Set<String> xmlFieldNames = new LinkedHashSet<String>();
        Map<String, Element> xmlStreamFields = XML_STREAM_FIELDS_CACHE;

        Element element = xmlStreamFields.get(url);
        if (element == null) {
            try {
                Element firstElement = XmlStreamHelper.getFirstElement(url);
                xmlFieldNames = XmlStreamHelper.getRSSFields(firstElement);
                xmlStreamFields.put(url, firstElement);

            } catch (Exception e) {
                log.warn(e.getMessage(), e);
            }
        } else {
            xmlFieldNames = XmlStreamHelper.getRSSFields(element);
        }
        
        return xmlFieldNames;
    }

    public void updateTemplateFieldTableModel(TemplateFieldBindingUI context,
            WikittyExtension extension, File template) {
        if (extension == null || template == null) {
            return;
        }
            
        context.setTemplate(template.getName());
        context.setExtension(extension.getName());
        TemplateFieldTableModel model = context.getFieldsModel();

        model.setValueIsAdjusting(true);
        model.clear();

        // mise a jour de l'univers des valeurs selectionnables
        List<String> fieldNames = new LinkedList<String>();
        fieldNames.add(Form.FQ_FIELD_ID);
        fieldNames.add(Form.FQ_FIELD_OBJET);
        fieldNames.add(Form.FQ_FIELD_DESCRIPTION);
        fieldNames.add(Form.FQ_FIELD_CREATIONDATE);
        fieldNames.add(Form.FQ_FIELD_DATEPUB);
        fieldNames.add(Form.FQ_FIELD_DATEPEREMPTION);
        fieldNames.add(Form.FQ_FIELD_ENTITY);
        fieldNames.add(Form.FQ_FIELD_SOURCETEXT);
        fieldNames.add(Form.FQ_FIELD_SOURCEURL);
        fieldNames.add(Form.FQ_FIELD_COUNTRY);
        fieldNames.add(Form.FQ_FIELD_DEPARTMENT);

        for (String fieldName : extension.getFieldNames()) {
            fieldNames.add(extension.getName() + "." + fieldName);
        }

        context.getContextValue(JListCellEditor.class).updateUniverse(fieldNames);
        context.getContextValue(JListCellRenderer.class).updateUniverse(fieldNames);
        if (fieldNames.size() > 0) {
            context.getFieldsTable().setRowHeight(19 * fieldNames.size());
        }

        try {
            Map<String, String> fieldMap = getVradiStorageService().getAssociatedFields(
                            extension.getName(), template.getName());
            
            if(fieldMap == null) {
                fieldMap = new HashMap<String, String>();
                
                String[] templateFields = VradiService.getMailingService()
                        .getDocumentFields(template);
                
                for(String field : templateFields) {
                    fieldMap.put(field, null);
                }
            }
            model.setData(fieldMap);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            
        } finally {
            model.fireTableDataChanged();
            model.setValueIsAdjusting(false);
        }
    }

    public void saveTemplateFieldBinding(TemplateFieldBindingUI context) {
        TemplateFieldTableModel model = context.getFieldsModel();
        Map<String, String> fieldMap = model.getData();

        try {
            String extension = context.getExtension();
            String template = context.getTemplate();
            
            getVradiStorageService().setAssociatedFields(extension, template,
                    fieldMap);

            context.dispose();

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    private AdminXmlStreamUI getXmlStreamUI(JAXXContext context) {
        if (context instanceof AdminXmlStreamUI) {
            return (AdminXmlStreamUI) context;
        }
        return null;
    }

    public String getFirstElementPreview(JAXXContext context, String url) {
        if (url == null || url.trim().isEmpty()) {
            return "";
        }

        Element firstElement = null;
        Map<String, Element> xmlStreamFields = XML_STREAM_FIELDS_CACHE;

        if (xmlStreamFields.get(url) == null) {
            AdminXmlStreamUI ui = getXmlStreamUI(context);
            try {
                ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                firstElement = XmlStreamHelper.getFirstElement(url);
                xmlStreamFields.put(url, firstElement);
            } catch (Exception eee) {
            } finally {
                ui.setCursor(null);
            }

        } else {
            firstElement = xmlStreamFields.get(url);
        }

        if (firstElement == null) {
            return "";
        }

        Map<String, String> elementValues = XmlStreamHelper.getFirstElementValues(firstElement);

        StringBuffer result = new StringBuffer();
        result.append("<html>");

        for (Map.Entry<String, String> entry : elementValues.entrySet()) {
            result.append("<strong>").append(entry.getKey())
                    .append("</strong>").append(" : ");

            String value = entry.getValue();
            if (value.length() > TOOLTIP_ELEMENT_MAX_CHAR_NB) {
                value = StringUtils.replace(value, "\r", "");
                value = StringUtils.replace(value, "\n\n", "\n");
                value = StringUtils.abbreviate(value, TOOLTIP_ELEMENT_MAX_CHAR_NB);
                value = StringUtils.replace(value, "\n", "<br/>");
            }
            result.append(value).append("<br/>");
        }
        
        result.append("</html>");
        return result.toString();
    }

    public void saveXmlStreamConfig(XmlStreamConfigUI ui) {
        int interval = (Integer)ui.getInterval().getSelectedItem();
        String unit = (String)ui.getUnit().getSelectedItem();
        try {
            if(XmlStreamConfig.HOURS.toString().equals(unit)) {
                int minute = (Integer)ui.getMinuteHours().getSelectedItem();
                getVradiStorageService().autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.HOURS.toString(), interval,
                        null, minute);
                
            } else if(XmlStreamConfig.DAYS.toString().equals(unit)) {
                int hour = (Integer)ui.getHourDays().getSelectedItem();
                int minute = (Integer)ui.getMinuteDays().getSelectedItem();
                getVradiStorageService().autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.DAYS.toString(), interval,
                        hour, minute);
                
            } else if(XmlStreamConfig.MINUTES.toString().equals(unit)){
                getVradiStorageService().autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.MINUTES.toString(), interval,
                        null, 5);
                
            } else {
                getVradiStorageService().autoLoadFormsFromXmlStreams(
                        XmlStreamConfig.DAYS.toString(), 1, 0, 0);
            }

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public ComboBoxModel getTemplateComboBoxModel(final WikittyExtension extension) {
        File[] files = getVradiStorageService().getTemplates(extension);
        if(files != null) {
            return new DefaultComboBoxModel(files) {
                @Override
                public void setSelectedItem(Object item) {
                    if (item == null || !String.class.equals(item.getClass())) {
                        super.setSelectedItem(item);
                    } else {
                        for (int i = 0; i < this.getSize(); i++) {
                            File f = (File) this.getElementAt(i);
                            if (f.getName().equals(item)) {
                                super.setSelectedItem(f);
                                break;
                            }
                        }
                    }
                }
            };
        } else {
            return new DefaultComboBoxModel();
        }
    }

    public void showTemplateExample(AdminFormTypeUI context) {
        WikittyExtension extension = context.getBean();
        
        VradiFormPageDTO formPage = new VradiFormPageDTO();
        formPage = VradiHelper.executeQuery(null, extension, null, null,
                null, null, null, formPage);
        
        if(formPage.getTotalFoundFormNb() > 0) {
            Form form = formPage.getFormsToShow().get(0);
            try {
                context.setCursor(new Cursor(Cursor.WAIT_CURSOR));
                
                File file = VradiHelper.generatePDF(getVradiStorageService(), form);
                
                if(file != null) {
                    Desktop.getDesktop().open(file);
                }
                
             } catch (VradiException e) {
                log.error(e.getMessage(), e);
                ErrorDialogUI.showError(e);
                
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                
            } finally {
                context.setCursor(null);
            }
        } else {
            JOptionPane.showMessageDialog(context,
                    _("vradi.adminFormType.template.showExample.error.message"),
                    _("vradi.adminFormType.template.showExample.error.title"),
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    public void uploadNewTemplate(AdminFormTypeUI context) {
        File template = FileUtil.getFile(context, new FileFilter[0]);
        try {
            File copy = getVradiStorageService().addTemplate(context.getBean(),
                    template);
            
            JComboBox templates = context.getTemplateChoice();
            templates.addItem(copy);
            templates.setSelectedItem(copy);
            
        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }
    
    public void showHistory(String id) {
        String queryHistory = getVradiStorageService().getQueryHistoryFile(id);
        String serviceContext = VradiService.getInstance().getServiceContext();
        
        if (serviceContext == null) {
            // services are local
            VradiConfig vradiConfig = VradiContext.get().getVradiConfig();
            File vradiUserDirectory = vradiConfig.getVradiUserDirectory();
            File rssFile = new File(vradiUserDirectory, queryHistory);
            
            // mini hack
            if (!rssFile.exists()) {
                rssFile = new File(vradiUserDirectory + "-SNAPSHOT", queryHistory);
            }
            
            try {
                URL url = rssFile.toURI().toURL();
                UIHelper.browseURI(url);
                
            } catch (MalformedURLException e) {
                log.warn(e.getMessage(), e);
            }
            
        } else {
            // services are remote
            UIHelper.browseURI(serviceContext + "/" + queryHistory);
        }
    }
}

