/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.swing.DefaultListModel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.filechooser.FileFilter;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.BusinessEntityWikitty;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.FieldTypeEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.services.dto.VradiThesaurusDTO;
import com.jurismarches.vradi.ui.editors.CurrencyEditor;
import com.jurismarches.vradi.ui.editors.DateEditor;
import com.jurismarches.vradi.ui.editors.DateTimeEditor;
import com.jurismarches.vradi.ui.editors.EmailEditor;
import com.jurismarches.vradi.ui.editors.NumEditor;
import com.jurismarches.vradi.ui.editors.StringEditor;
import com.jurismarches.vradi.ui.editors.TextEditor;
import com.jurismarches.vradi.ui.editors.UrlEditor;
import com.jurismarches.vradi.ui.editors.VradiEditor;
import com.jurismarches.vradi.ui.helpers.UIHelper;

/**
 * @author letellier
 */
public class OfferEditHandler {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(OfferEditHandler.class);
    private static final String DEFAULT_EDITOR_NAME = "$default";
    public static final String REQUIRED_FIELD_CONTEXT = "required";

    private VradiStorageService vradiStorageService = null;

    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }
    
    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param data
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public OfferEditUI initUI(JAXXContext rootContext, Form data) {
        JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this);

        // show main ui
        context.add(data);
        UIHelper.getHandler(context, ThesaurusHandler.class);
        OfferEditUI ui = new OfferEditUI(context);

        fillfields(ui, data);

        // Adding in referenciel
        List<OfferEditUI> editUIs = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get());
        if (editUIs == null) {
            editUIs = new ArrayList<OfferEditUI>();
            VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.setContextValue(VradiContext.get(), editUIs);
        }
        VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get()).add(ui);

        //init files
        DefaultListModel fileModel = ui.getFileListModel();
        if(data != null && data.getFiles() != null) {
            for(String fileName : data.getFiles()) {
                fileModel.addElement(fileName);
            }
        }

        //init attachments
        DefaultListModel attachmentModel = ui.getAttachmentListModel();
        if(data != null && data.getAttachments() != null) {
            for(String attachmentName : data.getAttachments()) {
                attachmentModel.addElement(attachmentName);
            }
        }

        // Init propositions
        try {
            List<TreeNodeImpl> thesaurusToPropose =
                    getVradiStorageService().proposeThesaurus(data, VradiContext.getTreeNodeInEntryDef());
            addPropositions(ui, thesaurusToPropose.toArray(
                    new TreeNodeImpl[thesaurusToPropose.size()]));

        } catch(VradiException eee) {
            log.error("Cant get propositions of thesaurus ", eee);
            ErrorDialogUI.showError(eee);
        }

        // Init thesaurus
        ThesaurusHandler thesaurusHandler = UIHelper.getHandler(context, ThesaurusHandler.class);
        thesaurusHandler.initThesaurus(context, ui);

        if (log.isDebugEnabled()) {
            log.debug("There is " + VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(VradiContext.get()).size() + " OfferEditUI in referenciel");
        }

        return ui;
    }

    public void fillfields(OfferEditUI ui, Form form) {
        Collection<String> extensions = form.getExtensionNames();
        for (String extensionName : extensions) {
            if (!extensionName.equals(ModificationTag.EXT_MODIFICATION_TAG)
                    && !extensionName.equals(Form.EXT_INFOGENE)
                    && !extensionName.equals(Form.EXT_FORM)) {
                
                BusinessEntityWikitty entityWikitty = (BusinessEntityWikitty) form;
                WikittyExtension extension = entityWikitty.getWikitty().getExtension(extensionName);
                
                Collection<String> fieldNames = extension.getFieldNames();
                for (String fieldName : fieldNames) {
                    createField(ui, extension, fieldName, form);
                }
            }
        }
    }

    protected static Map<String, Class<? extends VradiEditor<?, ?>>> editorMapping;

    protected void createField(OfferEditUI ui, WikittyExtension extension, String fieldName, BusinessEntity data) {
        JPanel content = ui.getContent();
        FieldType fieldType = extension.getFieldType(fieldName);
        
        if (!fieldType.isCollection()) {
            VradiEditor editor = getEditor(fieldType, ui);
            
            if (editor != null) {
                String extensionName = extension.getName();
                Object value = data.getField(extensionName, fieldName);
                editor.setFieldName(fieldName);
                
                String fieldDesc = VradiHelper.getFieldTypeDescription(fieldType);
                editor.setFieldDescription(fieldDesc);
                
                editor.setExtensionName(extensionName);
                editor.setExtensionFieldName(fieldName);
                
                editor.setValue(value);
                
                if (fieldType.isNotNull()) {
                    editor.setValidatorContext(REQUIRED_FIELD_CONTEXT);
                }
                content.add(editor);
            }
        }
    }

    OfferEditUI getUI(JAXXContext context) {
        if (context instanceof OfferEditUI) {
            return (OfferEditUI) context;
        }
//        OfferEditUI ui = VradiContext.OFFERT_EDIT_UI_ENTRY_DEF.getContextValue(context);
        return null;
    }

    protected void addPropositions(JAXXContext context, TreeNodeImpl[] props) {
        OfferEditUI ui = getUI(context);
        DefaultListModel propositionListModel = ui.propositionListModel;
        propositionListModel.removeAllElements();
        for (TreeNodeImpl p : props) {
            if (log.isDebugEnabled()){
                log.debug("Add proposition element : " + p);
            }
            VradiThesaurusDTO thesaurus = UIHelper.getHandler(context, ThesaurusHandler.class)
                    .findThesaurusInRef(p.getWikittyId());
            propositionListModel.addElement(thesaurus);
        }
    }

    protected void save(JAXXContext context, Form form) {
        log.info("Saving Form: " + form.getWikittyId());
        
        try {
            OfferEditUI offerEditUI = getUI(context);

            // Decrement forms by thesaurus
            Set<String> listThesaurus = form.getThesaurus();
            ThesaurusHandler thesaurusHandler = UIHelper.getHandler(context, ThesaurusHandler.class);
            if (listThesaurus != null){
                for (String id : listThesaurus) {
                    VradiThesaurusDTO thesaurus = thesaurusHandler.findThesaurusInRef(id);
                    if (thesaurus != null) {
                        thesaurus.decFormsForThesaurus();
                    }
                }
            }
    
            // Remove all thesaurus
            form.clearThesaurus();

            // Find all propositions selected
            for (Object s : offerEditUI.getPropositionList().getSelectedValues()){
                VradiThesaurusDTO selected = (VradiThesaurusDTO)s;

                // Save thesaurus
                form.addThesaurus(selected.getWikittyId());

                // Increment nb offer by thesaurus
                selected.incFormsForThesaurus();
            }

            // Find all thesaurus selected
            for (Component c : offerEditUI.getThesaurus().getComponents()) {
                ThesaurusUI thesaurusUI = (ThesaurusUI) c;
                DefaultListModel model = thesaurusUI.getThesaurusSelectedModel();
                Object[] selected = model.toArray();
    
                for (Object s : selected) {
                    VradiThesaurusDTO thesaurus = (VradiThesaurusDTO) s;
                    
                    // Save thesaurus
                    form.addThesaurus(thesaurus.getWikittyId());
    
                    // Increment nb offer by thesaurus
                    if(thesaurus != null) {
                        thesaurus.incFormsForThesaurus();
                    }
                }
            }
            
            // Save dynamic content
            JPanel content = offerEditUI.getContent();
            Component[] components = content.getComponents();
            for (Component c : components) {
                VradiEditor editor = (VradiEditor) c;
                updateFormField(editor, form);
            }
            
            // Save infogene content
            updateFormField(offerEditUI.getObjetEditor(), form);
            updateFormField(offerEditUI.getIdEditor(), form);
            updateFormField(offerEditUI.getSourceTextEditor(), form);
            updateFormField(offerEditUI.getSourceURLEditor(), form);
            updateFormField(offerEditUI.getEntityEditor(), form);
            updateFormField(offerEditUI.getCountryEditor(), form);
            updateFormField(offerEditUI.getDepartmentEditor(), form);
            updateFormField(offerEditUI.getDatePubEditor(), form);
            updateFormField(offerEditUI.getDatePeremeptionEditor(), form);
            updateFormField(offerEditUI.getDescriptionEditor(), form);
            
            if (!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATION_TAG)) {
                BusinessEntityWikitty entityWikitty = (BusinessEntityWikitty)form;
                entityWikitty.getWikitty().addExtension(ModificationTag.EXTENSION_MODIFICATION_TAG);
            }
            
            VradiUser vradiUser = context.getContextValue(VradiUser.class);
            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_MODIFIER,
                    vradiUser != null ? vradiUser.getName() : null);
            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_MODIFIED, new Date());

            getVradiStorageService().uploadFiles(form, offerEditUI.getFilesToUpload());
            getVradiStorageService().uploadAttachments(form, offerEditUI.getAttachmentsToUpload());
            
            getVradiStorageService().updateForm(form);
        } catch (Exception eee) {
            log.error("Cant save form : " + form.getObjet(), eee);
            ErrorDialogUI.showError(eee);
        }
    }

    void updateFormField(VradiEditor editor, Form form) {
        String extensionName = editor.getExtensionName();
        String extensionFieldName = editor.getExtensionFieldName();
        Object value = editor.getValue();
        if (log.isDebugEnabled()) {
            log.debug("updating: " + extensionName + "." + extensionFieldName + "=" + value);
        }
        form.setField(extensionName, extensionFieldName, value);
    }
    
    protected void update(JAXXContext context) {
        VradiMainUIHandler rootHandler = VradiContext.get().getContextValue(VradiMainUIHandler.class);
        rootHandler.goToHome(context);
    }

    protected void findQueries(JAXXContext context) {
        OfferEditUI ui = getUI(context);
        Form form = ui.getData();
        
        try {
            Map<QueryMaker, List<VradiQueryBean>> queries = getVradiStorageService()
                    .findQueriesReturningForm(form);

            RequestFormViewHandler handler = UIHelper.getHandler(context,
                    RequestFormViewHandler.class);
            
            RequestFormViewUI requestFormViewUI = handler.initUI(context, queries);
            requestFormViewUI.setVisible(true);

        } catch (VradiException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
        }
    }
    
    public void viewInPDF(JAXXContext context) {
        final OfferEditUI ui = getUI(context);

        final Form form = ui.getData();

        VradiTask<File> generatePDF = new VradiTask<File>(context, false) {
            @Override
            public File doAction() throws Exception {
                return VradiHelper.generatePDF(getVradiStorageService(), form);
            }

            @Override
            public void doWhenDone() throws Exception {
                File file = get();
                if (file == null){
                    JOptionPane.showMessageDialog(ui,
                            _("vradi.offerEdit.cantGeneratePDF"),
                            _("vradi.offerEdit.cantGeneratePDFTitle"),
                            JOptionPane.OK_OPTION);

                    return;
                }
                java.awt.Desktop.getDesktop().open(file);
            }
        };

        generatePDF.execute();
    }

    // TODO
    public void addFile(JAXXContext context) {
        if(log.isDebugEnabled()) {
            log.debug("addFile(context)");
        }
        FileFilter imageFileFilter = new FileFilter() {

            private final String[] okFileExtensions =
                new String[] {"jpg", "jpeg", "png", "gif"};

            @Override
            public boolean accept(File file) {
                if(file.isDirectory()) {
                    return true;
                }
                for (String extension : okFileExtensions) {
                    if (file.getName().toLowerCase().endsWith(extension)) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public String getDescription() {
                return "Images(.jpg, .jpeg, .png, .gif)";
            }
        };
        File file = FileUtil.getFile(getUI(context),
                new FileFilter[]{imageFileFilter});
        if(file != null) {
            getUI(context).getFilesToUpload().add(file);
            DefaultListModel model = getUI(context).getFileListModel();
            model.addElement(file.getName());
            getUI(context).getData().addFiles(file.getName());
        }
    }

    // TODO
    public void removeFile(JAXXContext context) {
        int fileIndex = getUI(context).getFileList().getSelectedIndex();
        if(fileIndex >= 0) {
            DefaultListModel model = getUI(context).getFileListModel();
            String fileName = (String)model.remove(fileIndex);
            getUI(context).getData().removeFiles(fileName);
        }
    }

    // TODO
    public void addAttachment(JAXXContext context) {
        if(log.isDebugEnabled()) {
            log.debug("addAttachment(context)");
        }
        File attachment = FileUtil.getFile(getUI(context), new FileFilter[0]);
        if(attachment != null) {
            getUI(context).getAttachmentsToUpload().add(attachment);
            DefaultListModel model = getUI(context).getAttachmentListModel();
            model.addElement(attachment.getName());
            getUI(context).getData().addAttachments(attachment.getName());
        }
    }

    // TODO
    public void removeAttachment(JAXXContext context) {
        int attachmentIndex = getUI(context).getAttachmentList().getSelectedIndex();
        if(attachmentIndex >= 0) {
            DefaultListModel model = getUI(context).getAttachmentListModel();
            String attachmentName = (String)model.remove(attachmentIndex);
            getUI(context).getData().removeAttachments(attachmentName);
        }
    }

    protected synchronized VradiEditor getEditor(FieldType type, JAXXContext context) {
        if (editorMapping == null) {
            editorMapping = new TreeMap<String, Class<? extends VradiEditor<?, ?>>>();
            editorMapping.put(FieldTypeEnum.STRING.name(), StringEditor.class);
            editorMapping.put(FieldTypeEnum.DATE_TIME.name(), DateTimeEditor.class);
            editorMapping.put(FieldTypeEnum.DATE.name(), DateEditor.class);
            editorMapping.put(FieldTypeEnum.EMAIL.name(), EmailEditor.class);
            editorMapping.put(FieldTypeEnum.NUMERIC.name(), NumEditor.class);
            editorMapping.put(FieldTypeEnum.TEXT.name(), TextEditor.class);
            editorMapping.put(FieldTypeEnum.URL.name(), UrlEditor.class);
            editorMapping.put(FieldTypeEnum.CURRENCY.name(), CurrencyEditor.class);
            editorMapping.put(DEFAULT_EDITOR_NAME, StringEditor.class);
        }

        FieldTypeEnum typeEnum = FieldTypeEnum.valueOf(type);
        if (log.isDebugEnabled()) {
            log.debug("typeEnum of type " + type + " = " + typeEnum);
        }
        String editorName = typeEnum == null ? DEFAULT_EDITOR_NAME : typeEnum.name();
        if (log.isDebugEnabled()) {
            log.debug("editor name for type [" + type.getType() + "] : " + editorName);
        }

        Class<? extends VradiEditor<?, ?>> editorClass = editorMapping.get(editorName);

        if (log.isDebugEnabled()) {
            log.debug("editor class to use : " + editorClass);
        }
        try {
            Constructor[] constructors = editorClass.getConstructors();
            for (Constructor constructor : constructors) {
                if (constructor.getParameterTypes() != null
                        && constructor.getParameterTypes().length == 1
                        && constructor.getParameterTypes()[0].equals(JAXXContext.class)) {
                    VradiEditor editor = (VradiEditor) constructor.newInstance(context);
                    editor.init();
                    return editor;
                }
            }
            return null;
        } catch (Exception eee) {
            // should never happens
            throw new IllegalStateException(eee);
        }
    }

    /**
     * Changes the status of the form and save it.
     *
     * @param context the actual context of the ui
     * @param form the form to modify and save
     * @param status the new status of the form
     */
    public void changeStatusAndSave(JAXXContext context, Form form, Status status) {
        String intialStatus = form.getStatus();
        String updateStatus = status.getWikittyId();
        
        if (intialStatus == null || !updateStatus.equals(intialStatus)) {
            if (!form.getExtensionNames().contains(ModificationTag.EXT_MODIFICATION_TAG)) {
                BusinessEntityWikitty entityWikitty = (BusinessEntityWikitty)form;
                entityWikitty.getWikitty().addExtension(ModificationTag.EXTENSION_MODIFICATION_TAG);
            }
            
            VradiUser vradiUser = context.getContextValue(VradiUser.class);
            String statusModifier = vradiUser != null ? vradiUser.getName() : null;
            
            log.info(String.format("Updating: statusModifier=%s on form: %s", statusModifier, form.getWikittyId()));
            log.info(String.format("New Status: %s, Old Status: %s", intialStatus, updateStatus));
            
            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_STATUS_MODIFIER,
                    statusModifier);

            form.setField(ModificationTag.EXT_MODIFICATION_TAG, ModificationTag.FIELD_LAST_MODIFIER,
                    statusModifier);
        }
        
        form.setStatus(updateStatus);
        save(context, form);
    }

    public void resizeMe(OfferEditUI ui){
//        JPanel p = ui.getFormContentPane();
//        p.setPreferredSize(new Dimension(p.getSize()));
//        p.revalidate();
    }

}
