/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyUtil;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the form research
 *
 * @author schorlet
 * @date 2010-01-29 12:40:26
 * @version $Revision: 474 $ $Date: 2010-02-07 10:52:07 +0100 (dim., 07 févr. 2010) $
 */
public class SearchManager {
    private static final Log log = LogFactory.getLog(SearchManager.class);

    private final WikittyProxy proxy;
    private final ClientManager clientManager;
    
    private static final DateFormat dateFormat = DateFormat.getDateInstance(
            DateFormat.SHORT, Locale.FRANCE);
    
    /**
     * Match dd/mm/yyyy, requiring leading zeros.
     * 
     * Match the pattern but not the validity of the date, so that <code>00/00/0000</code> will match.
     * 
     */
    private static final Pattern dateFormatPattern =
            Pattern.compile("^(3[01]|[12][0-9]|0[1-9])/(1[0-2]|0[1-9])/[0-9]{4}$");
    
    
    public SearchManager(WikittyProxy proxy, ClientManager clientManager) {
        this.proxy = proxy;
        this.clientManager = clientManager;
    }

    public SearchManager(ClientManager clientManager) {
        this.proxy = ServiceHelper.getWikittyProxy();
        this.clientManager = clientManager;
    }

    public void findForms(String query, VradiFormPageDTO formPageDTO)
            throws UnsupportedQueryException {
        findForms(query, null, null, null, null, null, formPageDTO);
    }
    
    public void findForms(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate, List<String>[] thesaurus,
            VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "findForms(query:%s, extension:%s, dateType:%s, beginDate:%tc, endDate:%tc)",
                    query, extension, dateType, beginDate, endDate));
        }

        FilterList filter = VradiQueryParser.parse(query);
        Search search = Search.query();
        buildSearch(filter, search);

        //add the extension in the criteria
        if (extension != null) {
            search.eq(Element.ELT_EXTENSION, extension.getName());
        }

        //add the date in the criteria
        if (dateType != null) {
            if (beginDate != null && endDate != null) {
                String beginString = WikittyUtil.solrDateFormat.format(beginDate);
                String endString = WikittyUtil.solrDateFormat.format(endDate);
                search.bw(dateType, beginString, endString);
            }
        }

        //add the thesaurus in the criteria
        if (thesaurus != null) {
            for (int i = 0; i < thesaurus.length; i++) {
                
                if (thesaurus[i] != null) {
                    Search subSearch = search.or();
                    for (String th : thesaurus[i]) {
                        subSearch.eq(Form.FQ_FIELD_THESAURUS, th);
                    }
                }
            }
        }

        Criteria criteria = search.criteria();
        //sets the pagination parameters
        int firstIndex = (formPageDTO.getPageToShow() - 1) * formPageDTO.getNbFormsToShow();
        int lastIndex = formPageDTO.getPageToShow() * formPageDTO.getNbFormsToShow() - 1;

        if (firstIndex >= 0 && lastIndex > 0 && lastIndex > firstIndex) {
            criteria.setFirstIndex(firstIndex).setEndIndex(lastIndex);
        }
        
        if (formPageDTO.getFieldToSort() != null) {
            if (!formPageDTO.isAscending()) {
                criteria.addSortDescending(formPageDTO.getFieldToSort());

            } else {
                criteria.addSortAscending(formPageDTO.getFieldToSort());
            }
        }

        criteria.addSortAscending(Form.FQ_FIELD_ID);

        //finds the forms
        PagedResult<Form> queryResult = proxy.findAllByCriteria(Form.class, criteria);
        List<Form> result = new ArrayList<Form>(queryResult.getAll());

        if (log.isTraceEnabled()) {
            for (Form form : result) {
                log.trace("found: " + form.toString());
            }
        }

        formPageDTO.setFormsToShow(result);
        formPageDTO.setTotalFoundFormNb(queryResult.getNumFound());
    }

    public List<String> getQueriesReturningForm(Form form)
            throws TechnicalException {

        List<String> queries = new ArrayList<String>();
        List<String> allQueries = new ArrayList<String>();

        //get all the queries of the query makers
        List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
        queryMakers.addAll(clientManager.getAllClients());
        queryMakers.addAll(clientManager.getAllGroups());
        queryMakers.addAll(clientManager.getAllUsers());

        for (QueryMaker queryMaker : queryMakers) {
            if (queryMaker.getQueries() != null) {
                for (String query : queryMaker.getQueries()) {
                    if (!allQueries.contains(query)) {
                        allQueries.add(query);
                    }
                }
            }
        }

        for (String query : allQueries) {
            try {
                FilterList filter = VradiQueryParser.parse(query);
                
                Search search = Search.query();
                search.eq(Form.FQ_FIELD_ID, form.getWikittyId());
                buildSearch(filter, search);

                Criteria criteria = search.criteria();
                PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
                
                if (forms.getNumFound() > 0) {
                    queries.add(query);
                }
                
            } catch (UnsupportedQueryException e) {
                log.warn("it seems that a persisted query is not supported anymore", e);
            }
        }

        return queries;
    }
    
    static void buildSearch(FilterList list, Search search) {
        Search subSearch = null;
        FilterList.Operator operator = list.getOperator();

        if (operator == FilterList.Operator.MUST_PASS_ONE) {
            subSearch = search.or();
            
        } else if (operator == FilterList.Operator.MUST_PASS_ALL) {
            subSearch = search.and();
            
        } else if (operator == FilterList.Operator.MUST_NOT_PASS) {
            subSearch = search.not();
        }
        
        List<Filter> filters = list.getFilters();
        for (Filter filter : filters) {
            
            if (filter instanceof FilterList) {
                buildSearch((FilterList) filter, subSearch);
        
            } else if (filter instanceof CompareFilter) {
                buildSearch(operator, (CompareFilter) filter, subSearch);
            }
            
        }
    }
    
    static void buildSearch(FilterList.Operator operator, CompareFilter compareFilter,
            Search search) {
        
        String name = compareFilter.getName();
        String value = compareFilter.getValue();
     
        if (VradiQueryParser.DEFAULT_FIELD.equals(name)) {
            search.keyword(value);
            return;
        }
        
        Search subSearch = search.or();
        
        if (operator == FilterList.Operator.MUST_NOT_PASS) {
            // date search
            Matcher matcher = dateFormatPattern.matcher(value);
            if (matcher.matches()) {
                try {
                    Date time = dateFormat.parse(value);
                    String timeString = WikittyUtil.solrDateFormat.format(time.getTime());
                    
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, timeString);
                    
                } catch (ParseException e) {
                    log.warn(value + " cannot be a date.");
                }
            }
            
        } else {
            // date search
            Matcher matcher = dateFormatPattern.matcher(value);
            if (matcher.matches()) {
                try {
                    Date time = dateFormat.parse(value);
        
                    Calendar calendar = new GregorianCalendar();
                    calendar.setTime(time);
                    
                    Calendar beginCal = new GregorianCalendar();
                    beginCal.set(calendar.get(Calendar.YEAR),
                            calendar.get(Calendar.MONTH),
                            calendar.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
                    
                    Calendar endCal = new GregorianCalendar();
                    endCal.set(calendar.get(Calendar.YEAR),
                            calendar.get(Calendar.MONTH),
                            calendar.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
                    endCal.add(Calendar.DAY_OF_MONTH, 1);
                    
                    String beginString = WikittyUtil.solrDateFormat
                            .format(beginCal.getTime());
                    String endString = WikittyUtil.solrDateFormat
                            .format(endCal.getTime());
                    
                    subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, beginString, endString);
                    
                } catch (ParseException e) {
                    log.warn(value + " cannot be a date.");
                }
            }
        }
        
        // text search
        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.TEXT, value);
     
        // string search
        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
        
        // boolean search
        if (value.equalsIgnoreCase("true")
                || value.equalsIgnoreCase("false")) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.BOOLEAN, value);
        }
        
        // number search
        if (NumberUtils.isNumber(value)) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, value);
        }

        // extension id search
//        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
//                + name + Criteria.SEPARATOR
//                + Element.ElementType.WIKITTY, value);
    }
}
